;
;EPROM SOFTWARE FOR MULTIBUS DUAL DENSITY DISK CONTROLLER (MDDDC)
;THIS VERSION IMPLEMENTS A DOUBLE DENSITY CONTROLLER ON PORT 78
;	AND A SINGLE DENSITY CONTROLLER ON PORT 88
;IT ALSO IMPLEMENTS LINKED IOPB'S
;COPYRIGHT (C) 1979, MICROMATION INC.
;
;        VERSION  I2.0


;	UPDATED DECEMBER 9, 1980

	MACLIB	MOD85

;HARDWARE MEMORY-MAPPED PORTS

PROM	EQU	0
DMAC	EQU	2000H		;DMA CONTROLLER
RAM	EQU	4000H		;1K OF READ/WRITE MEMORY
DISKIO	EQU	6000H		;BASE OF DISK I/O PORTS

BUFFER	EQU	RAM + 300H	;256 BYTE DMA BUFFER

WRMRKC	EQU	DISKIO		;WRITE MARK TO DISK, RESET CRC GENERATOR
RDMRKC	EQU	DISKIO		;READ MARK FROM DISK, RESET CRC CHECKER
WRMARK	EQU	DISKIO + 100H	;WRITE MARK TO DISK
RDMARK	EQU	DISKIO + 100H	;READ MARK FROM DISK
WRDATA	EQU	DISKIO + 200H	;WRITE DATA TO DISK
RDDATA	EQU	DISKIO + 200H	;READ DATA FROM DISK
WRCRC	EQU	DISKIO + 300H	;SEND DATA FROM CRC GENERATOR TO DISK
RDSYNC	EQU	DISKIO + 300H	;SYNC CONTROLLER TO SYNC FIELD OF DISK
WRPORT0	EQU	DISKIO + 400H	;WRITE TO PORT 89 (SD RESULT TYPE)
WRPORT1	EQU	WRPORT0 + 1	;WRITE TO PORT 8B (SD RESULT BYTE)
WRPORT2	EQU	WRPORT1 + 1	;WRITE TO PORT 79 (DD RESULT TYPE)
WRPORT3	EQU	WRPORT2 + 1	;WRITE TO PORT 7B (DD RESULT BYTE)
RDPORT0	EQU	WRPORT0		;READ FROM PORT 78 (NOT USED)
RDPORT1	EQU	WRPORT1		;READ FROM PORT 79 (IOPB LOW ADDRESS)
RDPORT2	EQU	WRPORT2		;READ FROM PORT 7A (IOPB HIGH ADDRESS)
RDPORT3	EQU	WRPORT3		;READ FROM PORT 7B (NOT USED)
WRCLK	EQU	DISKIO + 500H
RDPORT4	EQU	DISKIO + 500H	;READ FROM PORT 88 (NOT USED)
RDPORT5	EQU	RDPORT4 + 1	;READ FROM PORT 89 (IOPB ADDRESS LOW)
RDPORT6	EQU	RDPORT5 + 1	;READ FROM PORT 8A (IOPB ADDRESS HIGH)
RDPORT7	EQU	RDPORT6 + 1	;READ FROM PORT 8B (NOT USED)

DDIOPB	EQU	RDPORT1
SDIOPB	EQU	RDPORT5

;HARDWARE I/O MAPPED PORTS

WRCONT	EQU	(DISKIO SHR 8)+6	;CONTROL SIGNALS TO DRIVES
RDSTAT	EQU	WRCONT		;STATUS SIGNALS FROM DRIVES
WRCON1	EQU	WRCONT + 1	;CONTROL SIGNALS TO CONTROLLER AND SYSTEM
RDRDY	EQU	WRCON1		;READY LINES FROM 4 DRIVES (BITS 0-3)
				;STATUS OF INTS TO SYSTEM (BITS 6-7)
CLRINT	EQU	80H		;CLEARS INTERRUPT FROM SYSTEM

;
;SCRATCHPAD RAM VARIABLES
;
	ORG	RAM

;IOPB IMAGE AREA
CHANWD	DS	1		;CHANNEL WORD
OPCODE	DS	1
NOSECS	DS	1		;NUMBER OF SECTORS TO TRANSFER
TRACK	DS	1
SECTOR	DS	1
DMA	DS	2		;COMPLEMENT OF DMA ADDRESS
BLOCK	DS	1		;COMPLEMENT OF BLOCK NUMBER (LINKED ONLY)
NXIOPB	DS	2		;COMPLEMENT OF ADDRESS OF NEXT IOPB

;SCRATCH VARIABLES
CONTROL	DS	1		;IMAGE OF WRCONT
PBADDR	DS	2		;ADDRESS OF IOPB
DISK	DS	1		;CURRENTLY SELECTED DISK (FF IF NONE)
TRKTBL	DS	4		;TABLE OF TRACK NUMBERS FOR 4 DRIVES
DENSITY	DS	1		;00 FOR SINGLE DENSITY
				;FF FOR DOUBLE DENSITY
TIMER	DS	1		;DISK REVOLUTION COUNTER
READY	DS	1		;IMAGE OF RDRDY
RW	DS	1		;00 FOR READ, FF FOR WRITE
MARK	DS	1		;DATA MARK PATTERN
ERRFLG	DS	1		;ERROR TYPE FOR LAST ERROR FOUND
				;(00 IF NO ERRORS FOUND)
NOSEC1	DS	1		;NUMBER OF SECTORS TO READ ON SECOND
				;PASS OF DISK (WE SKEW MULT SECTOR COMMANDS)
SECT1	DS	1		;SECTOR TO START WITH ON SECOND PASS
LOGONTBL DS	4		;TABLE OF LOGGED ON DRIVES


STPDLY	EQU	8		;STEP TIME FOR SHUGART DRIVES
HDSTTL  EQU     45              ;HEAD SETTLING TIME FOR SHUGS & QUMES
SECLEN	EQU	128		;128 BYTE SECTORS


	ORG	PROM

INIT:	IN	CLRINT		;RESET INTERRUPTS FROM SYSTEM
	XRA	A
	LXI	SP,RAM+200H	;SET UP STACK POINTER
	LXI	H,RAM
INIT1:	MOV	M,A		;CLEAR SCRATCHPAD AREA
	INR	L
	JNZ	INIT1
	IN	RDRDY		;INPUT DRIVE READY STATUS
	ANI	0FH
	STA	READY
UNLOAD:	MVI	A,0FFH		;DESELECT DISK
	STA	DISK
	XRA	A
	OUT	WRCON1		;AND UNLOAD HEAD
	JMP	IDLE
;
;INTERRUPT VECTORS
;
	ORG	24H
TRAP:	PUSH	PSW
	JMP	INDEX

	ORG	2CH
RST55:	LHLD	SDIOPB		;GET IOPB ADDRESS
	IN	CLRINT
	JMP	OP88

	ORG	34H
RST65:	LHLD	DDIOPB		;GET IOPB ADDRESS
	IN	CLRINT
	JMP	OP78

	ORG	3CH
RST75:				;"STOP AFTER CURRENT IOPB"
	PUSH	PSW
	LDA	CHANWD
	ANI	0FBH		;RESET SUCCESSOR BIT
	STA	CHANWD
	MVI	A,10H		;RESET RST 7.5 INT
	SIM
	POP	PSW
	RET


INDEX:	LDA	TIMER		;INDEX PULSE RECEIVED FROM DISK
	DCR	A		;DECREMENT TIMEOUT COUNTER
	STA	TIMER
	RIM			;RESTORE RIM STATUS AFTER TRAP
	POP	PSW
	RET
;
;END OF INTERRUPT VECTORS
;
;IDLE LOOP
;
IDLE:
	MVI	A,8
	EI
	SIM			;ENABLE ALL INTERRUPTS
	IN	RDRDY		;GET DRIVE READY STATUS AND INT. BITS
	MOV	B,A		;SAVE DRIVE READY STATUS
	ANI	0C0H		;ANY INTERRUPTS TO SYSTEM?
	JNZ	IDLE1		;IF YES, DON'T PUT OUT READY CHANGE INTERRUPT
	MOV	A,B		;RECALL DRIVE READY STATUS
	ANI	0FH		;MASK FOR DRIVE READY STATUS
	LXI	H,READY
	CMP	M		;COMPARE WITH OLD DRIVE READY STATUS
	CNZ	RDYCHG		;IF CHANGED, CALL READY CHANGE HANDLER
IDLE1:	LDA	TIMER
	ORA	A		;TIME TO UNLOAD THE HEAD?
	JNZ	IDLE
	JMP	UNLOAD


RDYCHG:	MOV	C,A
	RRC
	RRC
	MOV	B,A
	MOV	A,C
	RLC
	RLC
	ORA	B
	ORI	0FH
	STA	WRPORT1		;DRIVE READY STATUS IN SD RESULT BYTE
	STA	WRPORT3		;DRIVE READY STATUS IN DD RESULT BYTE
	MVI	A,0FDH
	STA	WRPORT0		;RESULT TYPE = READY STATUS CHANGE
	STA	WRPORT2
	LDA	DISK
	INR	A		;HEAD UNLOADED?
	JZ	NOLOAD
	MVI	A,8
NOLOAD:	ORI	0C0H
	OUT	WRCON1		;PUT OUT INTERRUPTS TO PORTS 78, 88
	MOV	M,C		;SAVE NEW DRIVE READY STATUS
	RET


OP78:		;DISK OPERATION HANDLER FOR PORTS 78-7F
	POP	D		;THROW AWAY RETURN ADDRESS
	CALL	DDOP		;DO DISK OPERATION
	CMA
	STA	WRPORT3		;PUT RESULT IN RESULT BYTE
	MVI	A,0FFH
	STA	WRPORT2		;RESULT TYPE = I/O COMPLETE
	MVI	A,4
	OUT	WRCON1		;TAKE AWAY WRITE ENABLE
	MVI	A,8
	STA	TIMER		;SET UP TIMEOUT COUNT FOR HEAD UNLOAD
	LDA	CHANWD
	ANI	10H		;INHIBIT INTERRUPT?
	JNZ	IDLE		;IF YES, GO TO IDLE LOOP
	MVI	A,44H
	OUT	WRCON1		;PUT OUT INTERRUPT
	JMP	IDLE		;GO TO IDLE LOOP


OP88:		;DISK OPERATION HANDLER FOR PORTS 88-8F
	POP	D		;THROW AWAY RETURN ADDRESS
NXTOP:	DI
	CALL	SDOP		;DO DISK OPERATION
	PUSH	PSW		;SAVE ERRORS
	CMA
	STA	WRPORT1		;PUT RESULT IN RESULT BYTE
	LDA	CHANWD
	MOV	B,A		;TEMP STORE CHANNEL WORD
	ANI	4		;SUCCESSOR BIT SET?
	JZ	NOBLOK
	LDA	BLOCK
	RLC
	RLC
	ORI	3
NOBLOK:	DCR	A
	STA	WRPORT0		;RESULT TYPE = I/O COMPLETE
	POP	PSW
	ORA	A		;ANY ERRORS ON DISK OP?
	CNZ	RST75		;IF YES, RESET SUCCESSOR BIT
	MVI	A,0DH
	EI
	SIM			;ENABLE RST 7.5
	MVI	A,4
	OUT	WRCON1		;TAKE AWAY WRITE ENABLE
	MVI	A,8
	STA	TIMER		;SET UP TIMEOUT COUNT FOR HEAD UNLOAD
	MOV	A,B		;RECALL CHANNEL WORD
	PUSH	PSW		;SAVE IT
	RLC			;LOCK OVERRIDE?
	JC	OVERRD
	LHLD	PBADDR
	PUSH	H
	MVI	H,(BUFFER SHR 8)
	RRC
	ORI	1		;IF NO, SET WAIT BIT IN HOST RAM
	CMA
	MOV	M,A
	POP	H
	LXI	B,4000H
	CALL	DODMA
OVERRD:	POP	PSW		;RECALL CHANNEL WORD
	MOV	B,A		;TEMP STORE CHAN WORD
	ANI	20H		;FORCE INTERRUPT?
	JNZ	FORCE
	LHLD	NXIOPB
	MOV	A,B		;RECALL CHANNEL WORD
	ANI	4		;LINKED IOPB?
	JNZ	NXTOP		;IF YES, DO NEXT IOPB
	MOV	A,B		;RECALL CHANNEL WORD
	ANI	10H		;INHIBIT INTERRUPT?
	JNZ	IDLE		;IF YES, GO TO IDLE LOOP
FORCE:	MVI	A,84H		;ELSE, PUT OUT INTERRUPT
	OUT	WRCON1
FORCE1:	LDA	CHANWD
	ANI	4		;DID WE GET A STOP COMMAND?
	JZ	IDLE
	MVI	A,0AH
	SIM			;ANOTHER SD COMMAND GETS US OUT
	IN	RDRDY
	ANI	80H		;INTERRUPT SERVICED?
	JNZ	FORCE1		;IF NO, LOOP
	LHLD	NXIOPB
	JMP	NXTOP		;DO NEXT IOPB


DDOP:	CALL	GETPB		;GET THE I/O PARAMETER BLOCK
	LDA	OPCODE		;FIND DISK NUMBER
	RRC
	RRC
	RRC
	RRC
	ANI	3
	MOV	C,A		;DISK NUMBER IN C REG
	MVI	E,4
	MVI	A,0BH
	STA	MARK		;DOUBLE DENSITY DATA MARK
	MVI	A,0FFH		;DO DOUBLE DENSITY
	JMP	DOOP

SDOP2:	LHLD	NXIOPB
	RRC			;BRANCH ON WAIT?
	JC	SDOP
SDOP1:	MVI	B,10
	CALL	DELAY
	LHLD	PBADDR

SDOP:	SHLD	PBADDR
	CALL	GETPB		;GET THE I/O PARAMETER BLOCK
	LDA	CHANWD
	RRC			;WAIT BIT SET?
	JC	SDOP2
	LDA	OPCODE		;FIND DISK NUMBER
	RRC
	RRC
	RRC
	ANI	6
	MOV	C,A
	RRC
	XRA	C
	ANI	3
	MOV	C,A		;DISK NUMBER IN C REG
	MVI	E,5
	MVI	A,0FBH
	STA	MARK		;SINGLE DENSITY DATA MARK
	XRA	A		;DO SINGLE DENSITY
DOOP:	STA	DENSITY
SELDSK:	LXI	H,DSKTBL
	MVI	B,0
	DAD	B		;DISK NUMBER IS IN C REG
	DAD	B
	MOV	D,M		;GET MASK TO CHECK READY STATUS
	IN	RDRDY
	ANA	D		;READY?
	MVI	A,80H		;RETURN NOT READY ERROR IF NO
	RNZ
	PUSH	B
	INX	H
	MOV	A,M		;GET DISK SELECT MASK
	OUT	WRCONT		;SELECT DISK
	STA	CONTROL
	LDA	DISK
	CMP	C		;DISK ALREADY SELECTED?
	MOV	A,C
	STA	DISK		;STORE DISK NUMBER
	MOV	A,E
	OUT	WRCON1		;LOAD HEAD
	MVI	B,HDSTTL
	CNZ	DELAY		;WAIT FOR HEAD SETTLING TIME
	POP	B		;RECALL DISK NUMBER
	LXI	H,LOGONTBL
	DAD	B
	MOV	A,M		;GET LOGON STATUS
	MVI	M,0FFH		;MARK DRIVE AS LOGGED ON
	ORA	A		;WAS IT ALREADY LOGGED ON?
	CZ	HOME		;IF NO, HOME THE DRIVE
	LXI	H,JMPTAB	;SET UP FOR JUMP TO PROPER HANDLER
	LDA	OPCODE
	RLC
	ANI	0EH
	MOV	E,A
	MVI	D,0
	DAD	D
	MOV	E,M
	INX	H
	MOV	D,M
	XCHG			;PUT HANDLER ADDRESS IN HL
	PCHL			;EXECUTE JUMP

JMPTAB:	DW	NODSKOP		;NO-OP
	DW	SEEK
	DW	FORMAT
	DW	HOME
	DW	READ
	DW	VERIFY
	DW	WRITE
	DW	DELWRT

DSKTBL:	DB	1,0BFH
	DB	2,0DFH
	DB	4,0EFH
	DB	8,0F7H


GETPB:
	LXI	B,8009H		;SET UP TO READ 10 BYTES
	CALL	DODMA		;GET IOPB INTO DMA BUFFER
	MVI	H,(BUFFER SHR 8)
	LXI	D,CHANWD
	MVI	B,0FFH
GETPB2:	MVI	C,5
GETPB1:	MOV	A,M		;MOVE IOPB INTO IMAGE AREA
	XRA	B		;COMPLEMENT FIRST 5 BYTES ONLY
	STAX	D
	INX	D
	DCR	L
	DCR	C
	JNZ	GETPB1
	INR	B
	JZ	GETPB2
	RET


DODMA:		;DMA CONTROLLER DRIVER
		;HL = COMPLEMENT OF DMA ADDRESS
		;B = R/W BITS:  80H TO READ
		;		40H TO WRITE
		;C = (LENGTH OF DMA TRANSFER) - 1
;
;NOTE - THE MULTIBUS ADDRESS LINES ARE INVERTED, SO THE ADDRESS SENT TO THE
;	DMA CONTROLLER MUST ALSO BE COMPLEMENTED AND START AT THE LAST
;	DMA ADDRESS.

	PUSH	H		;SAVE DMA ADDRESS
	MOV	A,C
	CMA
	MOV	E,A
	MVI	D,0FFH
	DAD	D
	INX	H
	XCHG
	LXI	H,DMAC
	MOV	M,E
	MOV	M,D
	INX	H
	MOV	M,C
	MOV	M,B
	MVI	A,41H
	STA	DMAC + 8	;START DMA CONTROLLER
			;NOTE--PROCESSOR IS IN HOLD STATE DURING DMA TRANSFER
			;WHEN PROCESSOR RESUMES EXECUTION, DMA IS COMPLETE
	POP	H
	RET


NODSKOP:  ;NO-OP TO DISK CONTROLLER
	RET

SEEK:		;SEEK HANDLER
	LDA	TRACK
	ORA	A
	JZ	HOME
	MOV	D,A		;DESIRED TRACK IN D
	CPI	77		;TRACK < 77 ?
	MVI	A,8
	RNC			;RETURN ERROR IF NO
	MOV	A,D
	CPI	43		;TRACK > 43 ?
	JC	SEEK1
	LXI	H,CONTROL
	MOV	A,M
	ANI	0FBH		;IF YES, SET ABV43
	MOV	M,A
SEEK1:	CALL	GETTRK		;POINTS HL AT ACTUAL TRACK
SEEK2:	MOV	A,D
	SUB	M		;AT DESIRED TRACK?
	RZ			;RETURN 0 IF YES
	CALL	STEP
	JMP	SEEK2


STEP:		;STEPS ONE TRACK
		;STEPS OUT (TOWARD TRK 0) IF C FLAG SET
		;ELSE STEPS IN
		;ASSUMES HL POINTS AT ACTUAL TRACK, ADJUSTS TRACK #
	LDA	CONTROL
	JC	STPOUT
	ANI	0FDH		;SET DIRECTION = IN
	INR	M
	INR	M
STPOUT:	DCR	M
	OUT	WRCONT		;SET UP DIRECTION
	ANI	0FEH
	OUT	WRCONT		;PULSE STEP LINE LOW
	ORI	1
	OUT	WRCONT		;BRING STEP LINE BACK HIGH
	ORI	3
	OUT	WRCONT		;RESET DIRECTION
	MVI	B,STPDLY	;DELAY STEP TIME


DELAY:		;DELAYS (B) MILLISECONDS
		;USES C REG
	MVI	C,214
MSEC	DCR	C
	JNZ	MSEC
				;END OF 1 MILLISECOND DELAY
	DCR	B		;DECREMENT # OF MILLISECONDS TO DELAY
	JNZ	DELAY
	RET


GETTRK:		;POINTS HL AT ACTUAL TRACK
		;USES BC REG
	LXI	H,TRKTBL
	LDA	DISK
	MOV	C,A
	MVI	B,0
	DAD	B
	RET


HOME:		;STEPS DRIVE TO TRACK 0
	CALL	GETTRK		;POINT HL TO ACTUAL TRACK NUMBER
HOME1:	MVI	M,0		;TRACK # = 0
	IN	RDSTAT
	ANI	40H		;TRK0 = 0?
	RZ
	STC
	CALL	STEP		;IF NO, STEP OUT
	JMP	HOME1


WRENBL:	LDA	DENSITY
	CMA
	ANI	1
	ORI	6		;SET WRITE ENABLE TO CONTROLLER
	OUT	WRCON1
	IN	RDSTAT
	ANI	20H		;IS DISK WRITE ENABLED?
	MVI	A,20H
	RET


VERIFY:		;VERIFY CRC HANDLER

READ:		;READ DISK HANDLER
	XRA	A		;SET UP RW BYTE FOR READ
	JMP	DSKOP

DELWRT:		;WRITE DELETED DATA HANDLER
	LXI	H,MARK
	MOV	A,M
	ANI	0F8H
	MOV	M,A

WRITE:		;WRITE DISK HANDLER
	CALL	WRENBL		;WRITE ENABLE CONTROLLER
	RZ			;IF DISK IS PROTECTED, RETURN PROTECT ERROR
	MVI	A,0FFH		;SET UP RW BYTE FOR WRITE

DSKOP:	STA	RW
	CALL	SEEK		;GO TO DESIRED TRACK
	ORA	A
	RNZ			;RETURN IF ADDRESS ERROR ON SEEK
	LXI	H,SECTOR
	MVI	B,54
	LDA	DENSITY		;FIND DENSITY
	ORA	A		;SINGLE DENSITY?
	MOV	A,M		;GET SECTOR NUMBER
	JNZ	DDSK		;IF DOUBLE DENSITY, SKIP SD CODE
	ANI	1FH		;SINGLE DENSITY. STRIP OFF HIGH 3 BITS
	MOV	M,A		;AND STORE BACK IN SECTOR
	MVI	B,28
DDSK:	INR	A
	STA	SECT1
	DCR	A
	MVI	A,8
	RZ			;IF SECTOR # = 0, RETURN ADDRESS ERROR
	LXI	D,NOSECS
	LDAX	D		;GET NUMBER OF SECTORS
	ADD	M
	CMP	B		;SECTOR # + # SECTORS < 28(SD) OR 54(DD)?
	MVI	A,8
	RNC			;IF NO, RETURN ADDRESS ERROR
	LDAX	D
	ORA	A		;RESET CARRY
	RAR			;DIVIDE # SECTORS BY 2
				;(FOR SKEW OF 2 ON MULTIPLE SECTOR COMMANDS)
	STA	NOSEC1
	ACI	0
	STAX	D		;STORE BACK IN NOSECS

REPEAT:	XRA	A
	STA	ERRFLG		;CLEAR ERROR FLAG
	LHLD	DMA
	LXI	B,807FH		;SET UP TO READ 128 BYTES
	LDA	RW
	ORA	A		;IS THIS A SECTOR WRITE?
	CNZ	DODMA		;IF YES, READ 128 BYTES FROM SYSTEM
	MVI	A,4
	STA	TIMER		;SET UP TIMEOUT FOR BAD DISK OPERATION

SYNC:				;SYNC ON SYNC FIELD
	LXI	H,WRCLK
	LXI	D,RDDATA
	LDA	DENSITY
	ORA	A		;SINGLE DENSITY?
	JZ	SSYNC		;IF YES, GO TO SINGLE DENSITY ROUTINE

DSYNC:	MVI	M,0		;SYNC ON 00 CLOCK, FF DATA PATTERN IN DD
	IN	RDSTAT		;RESET CRC GENERATOR
DSYNC1:	LDA	TIMER
	ORA	A		;TIMED OUT ON DISK OP?
	JM	TIMOUT
DSYNC2:	INR	B		;AFTER EACH 255 BYTES OF TRYING TO DSYNC,
	JZ	DSYNC1		;  GO BACK TO CHECK THE TIMEOUT COUNTER
	LDA	RDSYNC		;SYNC ON FF CLOCK
	INR	A		;FF DATA?
	JNZ	DSYNC2		;IF NO, TRY AGAIN
	MVI	C,6		;SYNC ON 7 OF THE 10 BYTES IN SYNC FIELD
DSYNC3:	LDAX	D		;GET DATA BYTE
	INR	A		;FF DATA?
	JNZ	DSYNC2		;IF NO, RESYNC
	DCR	C		;DONE?
	JNZ	DSYNC3
				;FOUND SYNC FIELD
	MVI	M,70H		;SET UP CLOCK PATTERN FOR ID MARK
	LDAX	D		;SYNC WITH -EOW
	LDA	RDMARK		;LOOK FOR MARK
	CPI	0EH		;CORRECT MARK?
	JNZ	DSYNC		;IF NO, RESYNC
MRKOK:				;FOUND ID MARK
	LDAX	D		;TRACK BYTE FROM DISK
	LXI	H,TRACK
	CMP	M		;CORRECT TRACK?
	JNZ	TRKERR
	LDAX	D		;SIDE BYTE (IGNORE)
	INX	H
	LDAX	D		;SECTOR BYTE FROM DISK
	CMP	M
	JNZ	SYNC		;WRONG SECTOR, TRY AGAIN
				;FOUND CORRECT SECTOR
	LDAX	D		;RECORD LENGTH BYTE (IGNORE)
	LDAX	D		;CRC BYTE
	LDAX	D		;CRC BYTE
	LDAX	D		;READ 1 BYTE PAST CRC
	IN	RDSTAT
	RAL			;CHECK ID CRC
	JC	IDERR		;IF NO GOOD, GO TO ERROR ROUTINE
	LDAX	D		;GAP BYTE 2
	INR	M		;NEXT SECTOR = CURRENT SECTOR + 2
	INR	M
	LDAX	D		;GAP BYTE 3
	LDA	DENSITY
	ORA	A		;SINGLE DENSITY?
	JZ	SDIDOK		;IF YES, GO TO SINGLE DENSITY ROUTINE
	LDAX	D		;GAP BYTE 4
	LDA	RW
	ORA	A		;SECTOR READ?
	JNZ	WRTDD		;IF NO, GO TO WRITE ROUTINE

				;DOUBLE DENSITY READ
	MVI	B,18
DDGAP:	LDAX	D		;READ 18 BYTES OF GAP
	DCR	B
	JNZ	DDGAP
	LDAX	D		;SYNC FIELD BYTE 5
	LXI	H,WRCLK
	MVI	M,0
RESYNC	NOP
	LDA	RDSYNC		;RE-SYNC IN SYNC FIELD
	INR	A
	JNZ	RESYNC
	MVI	M,70H		;CLOCK PATTERN FOR DATA MARK
	LDAX	D		;SYNC WITH -EOW
	LDA	RDMARK		;LOOK FOR DATA MARK
	CPI	0BH		;CORRECT MARK?
	JNZ	DMKERR		;IF NO, GO TO ERROR HANDLER

DOREAD:	LHLD	DMA		;SET UP L REG WITH PROPER ADDRESS IN BUFFER
	MVI	H,(BUFFER SHR 8)
	LDAX	D		;GET FIRST DATA BYTE
	CMA
	MOV	M,A		;STORE DATA IN BUFFER
	MVI	C,SECLEN - 1	;SET UP LOOP COUNTER
RDLOOP:	DCR	L
	LDAX	D		;GET DATA BYTE
	CMA
	MOV	M,A		;STORE IT
	DCR	C
	JNZ	RDLOOP		;LOOP UNTIL DONE
	LDAX	D		;DATA CRC BYTE
	LDAX	D		;DATA CRC BYTE
	LDAX	D		;READ 1 BYTE PAST CRC
	IN	RDSTAT
	RAL			;DATA CRC OK?
	MVI	A,2
	RC			;IF NO, RETURN CRC ERROR

	LHLD	DMA
	LXI	B,407FH
	LDA	OPCODE
	ANI	1		;ARE WE DOING A VERIFY?
	CZ	DODMA		;IF NO, DMA 128 BYTES INTO SYSTEM
	JMP	NXTSEC		;FIND OUT IF WE'RE FINISHED

WRTDD:				;DOUBLE DENSITY WRITE ROUTINE
	LDAX	D		;GAP BYTE 5
	MVI	C,9
WDGAP:	LDAX	D		;READ 9 BYTES OF GAP
	DCR	C
	JNZ	WDGAP
	STAX	D		;WRITE GAP BYTE 17, 18 (THERE IS A
	STAX	D		;	2 BYTE TURN-ON DELAY)
	MVI	A,0FFH
	MVI	C,8
WRSYNC:	STAX	D		;WRITE 10 BYTES OF FF
	DCR	C
	JNZ	WRSYNC
	LXI	B,WRMARK	;PORT TO WRITE MARK IN DD
DOWRT:	STAX	D
	LHLD	DMA		;SET UP L REGISTER TO CORRECT ADDR IN BUFFER
	MVI	H,(BUFFER SHR 8)
	STAX	D
	LDA	MARK		;GET DATA MARK
	STAX	B		;WRITE MARK
	MOV	A,M		;GET FIRST DATA BYTE
	CMA
	STAX	D		;PUT IT OUT TO DISK
	MVI	C,SECLEN - 1	;SET UP LOOP COUNTER
WRLOOP:	DCR	L
	MOV	A,M		;GET DATA BYTE
	CMA
	STAX	D		;WRITE IT ON DISK
	DCR	C
	JNZ	WRLOOP		;LOOP UNTIL DONE
	STA	WRCRC		;WRITE CRC BYTE
	LDA	DENSITY
	CMA
	STA	WRCRC		;WRITE CRC BYTE
	STAX	D		;WRITE 2 BYTES OF GAP (FOR TURN-OFF DELAY)
	STAX	D
	STAX	D
	STAX	D

NXTSEC:	LXI	H,DMA+1
	DCR	M		;INCREMENT DMA ADDRESS
	LXI	H,NOSECS
	DCR	M		;LAST SECTOR?
	JNZ	REPEAT		;IF NO, DO NEXT SECTOR
	LXI	D,NOSEC1
	LDAX	D		;GET NUMBER OF SECTORS FOR NEXT PASS
	ORA	A		;ALREADY DONE?
	RZ			;IF YES, RETURN A 00
	MOV	M,A		;ELSE, STORE IT IN NOSECS
	XRA	A
	STAX	D		;STORE 00 FLAG IN NOSEC1
	LDA	SECT1		;GET FIRST SECTOR NUMBER OF NEXT PASS
	MOV	D,A
	LXI	H,SECTOR
	MOV	A,M
	MOV	M,D		;STORE IT IN SECTOR
				;COMPUTE DMA ADDRESS
	SUB	D
	ORA	A
	RAR
	MOV	D,A
	MVI	E,80H
	LHLD	DMA
	DAD	D
	SHLD	DMA
	JMP	REPEAT		;DO FIRST SECTOR OF SECOND PASS

;DISK ERROR HANDLERS

TIMOUT:	LDA	ERRFLG
	ORA	A		;ALREADY FOUND ERROR?
	RNZ			;IF YES, RETURN FOUND ERROR CODE
	MVI	A,0EH		;ELSE, RETURN "NO ADDRESS MARK"
	RET

TRKERR:	LDAX	D
	LDAX	D
	LDAX	D
	LDAX	D
	LDAX	D
	LDAX	D
	IN	RDSTAT
	RAL
	JC	IDERR		;IF ID CRC IS BAD, RETURN ID CRC ERROR
	MVI	A,4
	STA	ERRFLG		;ELSE, STORE TRACK ERROR
	CALL	HOME
	CALL	SEEK
	JMP	SYNC		;TRY UNTIL TIMEOUT

IDERR:	MVI	A,0AH
	STA	ERRFLG		;STORE ID CRC ERROR
	JMP	SYNC		;TRY AGAIN UNTIL TIMEOUT

DMKERR:	CPI	8
	JMP	MRKERR

SMKERR:	CPI	0F8H		;DELETED MARK?
MRKERR:	MVI	A,0FH
	STA	ERRFLG
	MVI	A,1		;IF YES, RETURN DELETED MARK ERROR
	RZ
	LXI	H,SECTOR
	DCR	M		;ELSE, DECREMENT SECTOR BY 2
	DCR	M		;(BECAUSE WE ALREADY INCREMENTED BY 2)
	JMP	SYNC		;AND TRY AGAIN


;SINGLE DENSITY ROUTINTES
SSYNC:				;SYNC ON FF CLOCK, 00 DATA IN SYNC FIELD
	MVI	M,0FFH		;SET UP FF CLOCK PATTERN
SSYNC1:	LDA	TIMER
	ORA	A		;TIMED OUT ON DISK OP?
	JM	TIMOUT
SSYNC2:	INR	B		;AFTER EACH 255 BYTES OF TRYING TO SYNC,
	JZ	SSYNC1		;  GO BACK TO CHECK THE TIMEOUT COUNTER
	LDA	RDSYNC		;SYNC ON FF CLOCK
	ORA	A		;00 DATA?
	JNZ	SSYNC2		;IF NO, TRY AGAIN
	LDAX	D
	ORA	A		;2ND 00 DATA BYTE?
	JNZ	SSYNC2
	LDAX	D
	ORA	A		;3RD 00 DATA BYTE?
	JNZ	SSYNC2
				;FOUND SYNC FIELD
SSYNC3:	MVI	M,0C7H		;SET UP CLOCK PATTERN FOR ID MARK
IDMARK:	LDA	RDMRKC		;LOOK FOR MARK, RESET CRC CHECKER
	ORA	A
	JZ	IDMARK		;STILL IN SYNC FIELD, TRY AGAIN
	CPI	0FEH		;ID MARK?
	JZ	MRKOK		;IF YES, CONTINUE
	MVI	M,0FFH		;ELSE, RESYNC
	LDA	RDSYNC
	ORA	A		;STILL IN SYNC FIELD?
	JZ	SSYNC3		;IF YES, LOOK FOR MARK
	JMP	SSYNC		;ELSE, RESYNC

SDIDOK:	LDA	RW
	ORA	A		;SECTOR READ?
	JNZ	WRTSD		;IF NO, GO TO WRITE ROUTINE
				;SINGLE DENSITY READ
	LDAX	D		;GAP BYTE 5
	MVI	B,7
SDGAP:	LDAX	D		;READ 7 BYTES OF GAP
	DCR	B
	JNZ	SDGAP
	LDAX	D		;SYNC FIELD BYTE 2
	LXI	H,WRCLK
	MVI	M,0FFH
	LDA	RDSYNC		;RE-SYNC IN SYNC FIELD
	MVI	M,0C7H		;CLOCK PATTERN FOR DATA MARK
	LDA	RDMRKC		;LOOK FOR DATA MARK
	CPI	0FBH		;CORRECT MARK?
	JNZ	SMKERR		;IF NO, GO TO ERROR HANDLER
	JMP	DOREAD

WRTSD:				;SINGLE DENSITY WRITE ROUTINE
	LDAX	D		;GAP BYTE 5
	LDAX	D		;GAP BYTE 6
	MVI	A,0FFH		;START WRITING GAP (THERE IS A 2 BYTE
				;	TURN-ON DELAY)
	STAX	D		;GAP BYTE 9
	STAX	D		;GAP BYTE 10
	STAX	D		;GAP BYTE 11
	XRA	A
	STAX	D		;WRITE 6 BYTES OF 00
	STAX	D
	STAX	D
	STAX	D
	LXI	B,WRMRKC	;PORT TO WRITE MARK IN SD
	JMP	DOWRT


FORMAT:		;FORMAT HANDLER
	CALL	SEEK		;GO TO DESIRED TRACK
	ORA	A
	RNZ			;RETURN IF ADDRESS ERROR ON SEEK
	CALL	WRENBL		;WRITE ENABLE CONTROLLER
	RZ			;IF DISK WRITE PROTECTED, RETURN ERROR
	LXI	B,8068H
	LHLD	DMA
	CALL	DODMA
	MVI	H,(BUFFER SHR 8)
	LDA	CHANWD
	ANI	40H		;RANDOM FORMAT SEQUENCE BIT SET?
	JNZ	FORM2		;IF YES, SKIP SKEW CODE
	PUSH	H
	MOV	A,M
	LXI	B,34FEH		;B=LOOP COUNTER, C=COMPLEMENT OF 1
FORM1:	MOV	M,C		;NEXT SECTOR NUMBER INTO BUFFER
	DCR	L
	MOV	M,A		;DATA BYTE INTO BUFFER
	DCR	L
	DCR	C
	DCR	B
	JNZ	FORM1
	POP	H
FORM2:	LXI	B,TIMER
	LXI	D,WRDATA
	MVI	A,1
	STAX	B		;TIMER = 1
	LDA	DENSITY
	ORA	A
	JZ	SDFORM
				;DOUBLE DENSITY FORMAT CODE
DDFORM:	LDAX	B
	ORA	A		;WAIT FOR INDEX INTERRUPT
	JNZ	DDFORM
				;FOUND INDEX
	XRA	A
	MVI	C,63
DFORM1:	STAX	D		;64 00'S
	DCR	C
	JNZ	DFORM1
	STAX	D
	MVI	B,52		;B = SECTOR LOOP COUNTER
DDNXT:	IN	RDSTAT		;RESET CRC GENERATOR
	MVI	A,0FFH
	MVI	C,9
DFORM2:	STAX	D		;10 FF'S
	DCR	C
	JNZ	DFORM2
	STAX	D
	MVI	A,0EH
	STA	WRMARK		;ID MARK
	LDA	TRACK
	STAX	D		;TRACK
	XRA	A
	STAX	D		;SIDE
	MOV	A,M
	CMA
	STAX	D		;SECTOR
	XRA	A
	DCR	L
	STAX	D		;RECORD LENGTH
	STA	WRCRC		;CRC
	STA	WRCRC		;CRC
	MVI	C,17
DFORM3:	STAX	D		;18 00'S
	DCR	C
	JNZ	DFORM3
	STAX	D
	MVI	A,0FFH
	MVI	C,9
DFORM4:	STAX	D		;10 FF'S
	DCR	C
	JNZ	DFORM4
	STAX	D
	MVI	A,0BH
	STA	WRMARK		;DATA MARK
	MOV	A,M
	CMA
	MVI	C,SECLEN-1
DFORM5:	STAX	D		;128 BYTES DATA
	DCR	C
	JNZ	DFORM5
	STAX	D
	XRA	A
	STA	WRCRC		;CRC BYTE
	DCR	L
	STA	WRCRC		;CRC BYTE
	MVI	C,17
DFORM6:	STAX	D		;18 00'S
	DCR	C
	JNZ	DFORM6
	STAX	D
	DCR	B		;DECREMENT SECTOR COUNTER
	JNZ	DDNXT		;REPEAT FOR 52 SECTORS
	STAX	D
	JMP	FORM8

				;SINGLE DENSITY FORMAT ROUTINE
SDFORM:	LDAX	B
	ORA	A		;WAIT FOR INDEX INTERRUPT
	JNZ	SDFORM
				;FOUND INDEX
	MVI	A,0FFH
	MVI	C,72
SFORM4:	STAX	D		;73 FF'S
	DCR	C
	JNZ	SFORM4
	STAX	D
	MVI	B,26		;B = SECTOR LOOP COUNTER
SDNXT:	XRA	A
	STAX	D		;6 00'S
	STAX	D
	STAX	D
	STAX	D
	STAX	D
	STAX	D
	MVI	A,0FEH
	STA	WRMRKC		;ID MARK
	LDA	TRACK
	STAX	D		;TRACK
	XRA	A
	STAX	D		;SIDE
	MOV	A,M
	CMA
	STAX	D		;SECTOR
	XRA	A
	DCR	L
	STAX	D		;RECORD LENGTH
	MVI	A,0FFH
	STA	WRCRC		;CRC
	STA	WRCRC		;CRC
	MVI	C,10
SFORM5:	STAX	D		;11 FF'S
	DCR	C
	JNZ	SFORM5
	STAX	D
	XRA	A
	STAX	D		;6 00'S
	STAX	D
	STAX	D
	STAX	D
	STAX	D
	STAX	D
	MVI	A,0FBH
	STA	WRMRKC		;DATA MARK
	MOV	A,M
	CMA
	MVI	C,SECLEN-1
SFORM6:	STAX	D		;128 BYTES DATA
	DCR	C
	JNZ	SFORM6
	STAX	D
	MVI	A,0FFH
	STA	WRCRC		;CRC BYTE
	DCR	L
	STA	WRCRC		;CRC BYTE
	MVI	C,26
SFORM7:	STAX	D		;27 FF'S
	DCR	C
	JNZ	SFORM7
	STAX	D
	DCR	B		;DECREMENT SECTOR COUNTER
	JNZ	SDNXT		;REPEAT FOR 26 SECTORS
	STAX	D
				;DOUBLE DENSITY REJOINS HERE
FORM8:	XCHG			;(HL)=WRDATA
	LXI	D,TIMER
	MOV	B,A		;(B)=FF
FORM9:	MOV	M,B		;FILL WITH FF'S UNTIL INDEX
	LDAX	D		;GET TIMER
	ORA	A		;STILL POSITIVE?
	JP	FORM9
	XRA	A		;SUCCESSFUL OPERATION
	RET
