	title	'C1-board monitor, relocating version.  26 Dec 83'
;
;	Copyright (C) 1982, 1983 by V. Nadvornik
;
;	REV .3:  10 dec 83
;	Modify boot routine to support dma
;
;       REV .2:
;	Includes self relocating feature
;
;	REV .1:
;	Modified memory mapper routines to support modified hardware
;	Modified SIO initialization table to start with WR4.
;
;	VERSION 1. - REV .0:
;	Initial coding.
;
	maclib	z80
$*Macro
;
true		equ	0ffffh		;true condition
false		equ	not true	;false
;
irq_mode	equ	false		;set to true if console interrupts enabled
reloc		equ	true
;
	if reloc
ROM		equ	8000h
	else
ROM		equ	0000h		;origin of this rom
	endif
;
vers		equ	1		;version number
rev		equ	3		;revision
release		equ	'D'
;
msize		equ	64		;cp/m memory size in kilobytes
;
extra		equ	msize*1024-0c00h	;allocate extra space for large bios
bios		equ	extra-600h	;base of bios
bdos		equ	bios-0e00h	;base of bdos
ccp		equ	bdos-800h	;base of ccp
;
;	breakpoint constants
;
nbkpts		equ	2		;number of breakpoints
bkbyte		equ	0cfh		;restart 1 instruction
rst1		equ	0008h		;software breakpoint
;
tries		equ	15		;disk boot retries
auto_boot	equ	1110_0000b	;monitor/auto boot switches
;
	include c1equ
;
;	misc. literals
;
lf		equ	0ah		;line feed
cr		equ	0dh		;carrage return
space		equ	' '		;space (blank) char.
coma		equ	','
cntlc		equ	'C'-40h		;control C char.
bksp		equ	'H'-40h		;back space char
cntls		equ	'S'-40h		;control S char.
;
;	register save and misc. variable storage areas.
;	stack frame memory image
;
regbase		equ	ROM+1100h	;top of register and variable
					;storage area.
;
disloc		equ	regbase-2	;last dump address
;
bk2byt		equ	disloc-1	;bkpt #2 inst. byte
bk2adr		equ	bk2byt-2	;bkpt #2 inst. address
bk1byt		equ	bk2adr-1	;bkpt #1 inst. byte
bk1adr		equ	bk1byt-2	;bkpt #1 inst. address
;
;	standard (8080) registers
;
pc_reg		equ	bk1adr-2	;<PC> register storage
h_reg		equ	pc_reg-1	;<H>
l_reg		equ	h_reg-1		;<L>
hl_reg		equ	l_reg
sk_reg		equ	hl_reg-2	;<SP>
d_reg		equ	sk_reg-1	;<D>
e_reg		equ	d_reg-1		;<E>
de_reg		equ	e_reg
b_reg		equ	de_reg-1	;<B>
c_reg		equ	b_reg-1		;<C>
bc_reg		equ	c_reg
a_reg		equ	bc_reg-1	;<A>
f_reg		equ	a_reg-1		;[FLAGS]
af_reg		equ	f_reg
;
;	special Z-80 registers
;
y_reg		equ	af_reg-2	;<Y>
x_reg		equ	y_reg-2		;<X>
r_reg		equ	x_reg-1		;<refresh counter>
i_reg		equ	r_reg-1		;<mode 2 irq, high byte reg.>
;
;	Z-80 "prime" register bank
;
h_p_reg		equ	i_reg-1		;<H>'
l_p_reg		equ	h_p_reg-1	;<L>'
d_p_reg		equ	l_p_reg-1	;<D>'
e_p_reg		equ	d_p_reg-1	;<E>'
b_p_reg		equ	e_p_reg-1	;<B>'
c_p_reg		equ	b_p_reg-1	;<C>'
a_p_reg		equ	c_p_reg-1	;<A>'
f_p_reg		equ	a_p_reg-1	;[FLAGS]'
;
		page
;
		ORG	ROM		;start of monitor
;
C1mon:
	if reloc
		jmp	init-ROM
	else
		jmp	init		;cold entry point
	endif
		jmp	wboot		;warm boot prom entry point
		dw	0000h		;pad to restart 1 location
		jmp	bkpt		;overlay restart 1 code
;
;	Comtbl
;
;	comtbl contains the addresses of the action routines,
;	the executive uses it to branch to the desired routine.
;
comtbl		dw	assign		;A-assign memory bank
		dw	boot		;B-disk cold boot
		dw	cmderr		;C-not used
		dw	dump		;D-dump memory
		dw	examn		;E-examine/change memory
		dw	fillm		;F-fill memory with constant
		dw	goto		;G-reload reg's and enter user program
		dw	hmath		;H-hex 'add' and 'sub' routine
		dw	inport		;I-input from port
		dw	mtest		;J-memory test
		dw	cmderr		;K-not used
		dw	cmderr		;L-not used
		dw	move		;M-moves a block of memory
		dw	next		;N-next (trace) command
		dw	otport		;O-output to port
		dw	cmderr		;P-not used
		dw	cmderr		;Q-not used
		dw	xamin		;R-examine/change registers
		dw	search		;S-search for byte string
;
;	interrupt vectors for SIO console
;
		ORG	($+1) and 0fffeh	;force even boundary
;
intvec		dw	ioerr		;bad interrupt, should never get here
		dw	ioerr
		dw	int		;receive interrupt vector for
					;serial interrupts.
		dw	ioerr
;
dma_vec		dw	dma_isr		;pointer to DMA interrupt serivce
					;routine
;
	page
;--------------
;
;	Assign memory routine
;
;	this routine allows the user to reassign
;	memory into the active region.  except
;	for block 0, where the monitor stack and
;	local variables reside.
;
;	  phyical  memory    block #
;	---------------------------------------
;	| 3c000h - 3ffffh	F  |
;	| 38000h - 3bfffh	D  | top 64k
;	| 34000h - 37fffh	E  |
;	| 30000h - 33fffh	C  |___________
;	| 2c000h - 2ffffh	B  |
;	| 28000h - 2bfffh	9  | third 64k
;	| 24000h - 27fffh	A  |
;	| 20000h - 23fffh	8  |___________
;	| 1c000h - 1ffffh	7  |
;	| 18000h - 1bfffh	5  | second 64k
;	| 14000h - 17fffh	6  |
;	| 10000h - 13fffh	4  |___________
;	| 0c000h - 0ffffh	3  |
;	| 08000h - 0bfffh	1  | bottom 64k
;	| 04000h - 07fffh	2  |
;	| 00000h - 03fffh	0  |
;	---------------------------------------
;
;	syntax:
;		Ax,y<cr>	where xx is the block to reassign (0-f)
;				and y is the active block to be assigned
;				to (1,2 or 3).
;
;	note:  blocks 0 - 3 are defaulted by the monitor.
;
assign:		call	exlf		;returns with
					;(E) = 0 to 15 = memory block
					;(L) = 0 to 3  = memory page
		mov	a,e		;test, only 16 blocks available
		cpi	16
		jnc	cmderr
		mov	a,l		;test, only 4 pages active at one time
		ora	a		;test for active page 00
		jz	cmderr		;no no.
		cpi	4
		jnc	cmderr
		cmc			;clear carry
	rept 4
		rlc			;position page bits
	endm
		ora	e		; set bank field = (E)
		out	mem_map		; write bank change into mapper
		ret
;
;	Boot disk
;
boot:		lxi	sp,0000h	;reset stack
		call	wboot
		jmp	bios		;finish init
;
;	DMA interrupt service routine
;
dma_isr:	push	psw
		mvi	a,force_irq	;terminate FDC read command
		out	fdc_cmd
		mvi	a,0c3h
		out	5ch		;reset dma controller
		pop	psw
		reti
;
wboot:		mvi	a,high(dma_vec)	;high byte of DMA interrupt vector
		stai			;into vector register
		in	sasi_stat	;get boot jumper switches
		rlc
		rlc
		rlc			;position switches
		ani	0000_0011b
		jz	sasi_boot	;boot from the hard disk
		mvi	b,0000_0001b	;8" select bits
		dcr	a
		jrz	rd_sys
		mvi	b,0001_0001b	;5.25" select bits
;
;	Read the system off of tracks 0 and 1
;
rd_sys:		in	sys_ctl		;set DMA mux
		ani	1111_1000b
		ori	dma_fdc
		out	sys_ctl
		mov	a,b		;get select bits
		out	drv_ctl		;start up drive
;
wait_rdy:	in	fdc_stat	;wait for FDC to go ready
		ora	a
		jm	wait_rdy
		mvi	a,0000_1111b	;home command, longest step time
		call	cmd_fdc		;do the restore
		ani	seek_err	;check for errors
		jrnz	hom_err		;wait till were home
;
;		call	motron		;test motor speed
		mvi	e,tries		;retry count
		lxi	h,dma_rd0	;first dma block of load
		call	rd_trk		;read track 0
;
		mvi	a,0101_1111b	;step in command
		call	cmd_fdc
;
		mvi	e,tries		;error counter
		lxi	h,dma_rd1
		call	rd_trk		;read track 1
		ret
;
hom_err		lxi	h,hom_msg	;no disk error msg
		call	prnmsg
		call	ci
		jmp	wboot		;retry from the top
;
rd_trk:		push	d		;save reg.s
		push	h
		mvi	a,01
		out	fdc_sec		;and sector #1
		call	set_dma
		mvi	a,1001_1100b	;read multi. sectors
		call	cmd_fdc		;start FDC reading
		ani	read_err	;error test
		pop	h
		pop	d		;restore original condition
		rz
		dcr	e
		jnz	rd_trk		;retry
		push	b
		push	d
		push	h
		lxi	h,booterr
		call	prnmsg
		call	ci
		pop	h
		pop	d
		pop	b
		mvi	e,tries
		jmp	rd_trk
;
cmd_fdc:	out	fdc_cmd		;command controller
		xthl
		xthl			;delay for status
		xthl
		xthl
;
wait_busy:	in	fdc_stat	;get status
		bit	0,a		;test busy bit
		jrnz	wait_busy
		ret
;
;	Motor start code, test if the drive requires motor control,
;	if so then time the index pulses until the drive motor is up
;	to speed.
;
;	Motor control required.  wait a maximum of 20 revolutions
;	for motor to get to speed, error out if no index or motor slow.
;
motron:		mvi	d,21		;loop counter
;
mtron:		dcr	d
		jrz	mot_err
		lxi	b,0000
;
;	Wait for index to go true (high)
;
mtron1:		in	fdcstat
		ani	fdc_index
		jrnz	mtron1a		;index true, start timing
		dcx	b
		mov	a,b
		ora	c
		jrz	mot_err
		jr	mtron1
;
;	Wait for index to go low, and time to next index
;
mtron1a:	lxi	b,14483		;approx. 210ms
;
mtron2:		in	fdcstat		;wait for index to go low
		ani	fdc_index
		jrnz	mtron2
;
		in	fdcstat		;get index bit
		ani	fdc_index
		rnz			;exit when motor is at speed
;
		dcx	b		;timer
		mov	a,b
		ora	c
		jrz	mtron		;timeout try again
		jr	mtron2		;continue timing index pulse
;
mot_err:	lxi	h,mot_msg
		call	prnmsg
		call	ci
		jmp	motron

;
;	Initialize and start the DMA chip
;
set_dma:	lxi	b,(dma_len shl 8) + dma	;length and port
		outir
		ei			;start interrupts
		ret			;DMA setup
;
dma_rd0		db	1100_0011b	;reset DMA
		db	0111_1101b	;port address and block count follow
		dw	fdc_data	;port address
		dw	(256*18)-1	;block count for track 0
		db	0011_1100b	;port a is fixed
		db	0001_0000b	;port b is memory, incrementing
		db	1101_1101b	;port B address follows
		dw	ccp-256		;base of system load
		db	0001_0010b	;interrupt at EOB, vector follows
		db	low(dma_vec)	;low byte of vector
		db	1000_0010b	;read active low, end of block
		db	0cfh		;load registers
		db	05h		;transfer A -> B
		db	0cfh		;reload registers
		db	8bh
		db	0abh		;enable interrupts
		db	87h		;arm the DMA
;
dma_len		equ	$-dma_rd0	;length of DMA string
;
dma_rd1		db	1100_0011b	;reset DMA
		db	0111_1101b	;port address and block count follow
		dw	fdc_data	;port address
		dw	(256*18)-1	;block count for track 1
		db	0011_1100b	;port a is fixed
		db	0001_0000b	;port b is memory, incrementing
		db	1101_1101b	;port B address follows
		dw	(ccp-256)+(256*18)	;base of system load
		db	0001_0010b	;interrupt at EOB, vector follows
		db	low(dma_vec)	;low byte of vector
		db	1000_0010b	;read active low, end of block
		db	0cfh		;load registers
		db	05h		;transfer A -> B
		db	0cfh		;reload registers
		db	8bh
		db	0abh
		db	87h		;arm the DMA
;
;	Boot CP/M from the SASI port
;
sasi_boot:	jmp	cmd_err
;
;	Display memory routine
;
;	this routine displays a block of memory on the
;	current console device (console dump).
;
;	syntax:
;		D<cr>		displays 256 bytes of memory
;				starting at the end of the last dump.
;		Dxxxx<cr>	displays 256 bytes of memory
;				starting at address xxxx.
;		Dxxxx,yyyy<cr>	displays memory from xxxx to
;				yyyy inclusive.
;
;	the display is organized to display up to 16 bytes
;	per display line, with all columns aligned so
;	each column has the same last hex digit in its address.
;
dump:		call	pchk		;check for delimiter or value
		jrc	dishl		;display memory(<hl>)
		mvi	b,01		;go get value
		lxi	h,0000h
		call	ex1
		jrc	disblk		;one value display 256 bytes
		mvi	b,01		;two values
		call	expr		;display range
		pop	d
		pop	h
		jr	dis1		;display given range
;
;	display 256 bytes starting at given address
;
disblk:		pop	d		;get it
		jr	dishl1
;
;	display memory starting at
;	the location pointed to by <hl>
;
dishl:		lded	disloc		;get start address
		inx	d		;point to next byte
;
dishl1:		lxi	h,0ffh		;one page long
		dad	d		;end address
		xchg			;<de>= end, <hl>=start
;
dis1:		sded	disloc		;save ending address
;
dis1a:		call	ladrb		;display the start address
		mov	a,l		;see if on 16 byte boundary
		call	trplsp		;skip over to right column
		push	h		;save <hl>
;
dis2:		mov	a,m		;get the contents
		call	hex1		;output it
		call	hilo		;increment, check pointer
		jrc	dis7		;done if carry set
		call	blk		;make columns
		mov	a,l		;ready for new line?
		ani	0fh
		jrnz	dis2
;
dis3:		pop	h		;reget line start address
		mov	a,l		;skip over to right space
		ani	0fh
		call	trpl2
;
dis4:		mov	a,m		;get memory value
		mov	c,a		;set up for output
		cpi	' '		;see if printable in ascii
		jrc	dis5		;jump if so
		cpi	7fh
		jrc	dis6
;
dis5:		mvi	c,'.'		;else, print a dot
;
dis6:		call	co
		call	hilox		;increment <hl> and see if done
		mov	a,l		;not done, ready for new line?
		ani	0fh
		jrnz	dis4		;jump if not
;
	if not irq_mode
		call	chk_abort	;check for key strike on console
	endif
;
		jr	dis1a		;do the next line
;
dis7:		xra	a
		call	trplsp
		jr	dis3		;go print the ascii
;
trplsp:		ani	0fh		;isolate the low four bits
		mov	b,a		;prepare to space over to right column
		add	a		;triple the count
		add	b
;
trpl2:		mov	b,a		;put back into b
		inr	b		;adjust counter
;
trpl1:		call	blk		;do the spacing
		djnz	trpl1		;no, do another column
		ret
;
;	Examine/change memory routine
;
;	this routine allows the user to inspect any ram memory location
;	and alter the contents, if desired.
;	the contents may be left unaltered
;	by entering a space or comma.  if a carriage return
;	is entered, the routine is terminated.
;	if a space or comma is entered, the routine
;	proceeds to the next location and presents the user
;	with an opportunity to alter it.
;	if a back-space is entered then the previous
;	address is displayed.
;
;	systax:
;		Exxxx<cr>	display and/or replace
;				memory.
;
examn:		call	expr1		;go get one parameter
		pop	h		;get the start address
		jr	subs4
;
sub1:		mov	a,m		;get the contents of the address
		call	dash1		;display it on console and a dash
		call	pchk		;get, check character
		rc			;done if carriage return
		jrz	sub2		;no change if blank or,
		cpi	bksp		;see if previous byte wanted
		jrz	sub3		;yes, do it
		push	h		;save memory pointer
		call	exf		;go get rest of new value
		pop	d		;new value to <e> register
		pop	h		;restore memory pointer
		mov	m,e		;put down new value
		mov	a,c		;get the delimiter
		cpi	cr		;see if done (carriage return)
		rz			;yes, return to monitor
;
sub2:		inx	h		;no, increment memory pointer
		inx	h		;allow a fall-through on the next instruction
;
sub3:		dcx	h		;adjust <hl> as appropriate
;
subs4:		call	ladra		;print new location
		call	blk		;print a space between addr and data
		jr	sub1		;go do the next location
;
;	Fill memory routine
;
;	this routine fills a block of memory with a user
;	determined constant.  it expects three parameters
;	to be entered in the following order:
;
;	start address
;	finish address
;	fill value
;
;	syntax:
;		Fssss,ffff,vv<cr>	fill memory from
;					ssss to ffff with vv
;
fillm:		call	expr3		;get three parameters
;
fill1:		mov	m,c		;put down the fill value
		call	hilo		;increment and check the pointer
		jrnc	fill1		;not done yet, jump
		jmp	command		;incase stack was overwritten
;
;	Go routine
;
;	the "go" command transfers control to a specified address.
;	it allows the setting of up to two breakpoints
;	as well as allowing any console input to breakpoint
;	the run, as long as interrupts are active.
;
;	syntax:
;		G<cr>			goto address stored in register
;					save area.
;		Gxxxx<cr>		goto address specified by xxxx.
;		G,(yyyy),(zzzz)<cr>	goto address saved in pc and set
;					break-points at (yyyy) & (zzzz).
;		Gxxxx,(yyyy),(zzzz)<cr>	goto address specified
;					by xxxx, and set break-
;					points at (yyyy) & (zzzz).
;
goto:		call	pchk		;see if old address wanted
		jrc	go3		;  yes, jump
		jrz	go0		;  yes, but set some breakpoints
		call	exf		;get new goto address
		pop	d 
		sded	pc_reg		;put address in <pc> location 
		mov	a,c 
		cpi	cr		;see if a cr was last entered
		jrz	go3 
;
go0:		mvi	b,nbkpts 
		lxi	h,bk1adr	;point to trap storage
;
go1:		push	b		;save number of breakpoints
		push	h		;save storage pointer
		mvi	b,2		;set up to get a trap address
		call	expr1		;get a trap address
		pop	d		;get the trap address into <de>
		pop	h		;reget the storage address
;
bkgo:		mov	m,e		;save the breakpoint address
		inx	h 
		mov	m,d 
		inx	h 
		ldax	d		;save the instruction from the bp address
		mov	m,a 
		inx	h 
		mvi	a,bkbyte	;insert the breakpoint
		stax	d 
;
go2:		mov	a,c		;reget the delimiter to see
		cpi	cr		;  if we are done setting breakpoints
		pop	b		;  unload the stack first
		jrz	go3		;yes, jump
		djnz	go1		;jump if not at bp limit
;
go3:		call	crlf 
		mvi	a,jmp		;setup restart 1
		sta	rst1
		lxi	h,bkpt		;address of break-point routine
		shld	rst1+1
		pop	h		;clean up stack
		pop	psw		;restore all registers
		pop	b
		pop	d
		pop	h
		exaf
		exx			;swap register banks
		pop	b
		mov	a,c
		stai			;put back irq reg
		popix
		popiy
		pop	psw
		pop	b
		pop	d
		pop	h		;stack pointer
		sphl
		lhld	pc_reg		;get user <pc>
		push	h		;save on user stack
		lhld	hl_reg		;get user <hl>
		ei			;reenable interrupts
		ret			;enter user code
;
;	Hex math routine
;
;	this routine adds and subtracts two hexadecimal 16-bit
;	unsigned numbers and displays the results on the
;	console.
;
hmath:		call	exlf		;get the two numbers
		push	h		;save it for the subtract
		mvi	b,'+'		;display formatting
		call	hmath1		;print xxxx+yyyy
		dad	d		;add them
		call	hmath2		;print =zzzz<crlf>
		pop	h		;reget the first number
		mvi	b,'-'
		call	hmath1		;print xxxx-yyyy
		ora	a		;clear the carry bit
		dsbc	d		;do the subtract
;
hmath2:		mvi	c,'='		;print an "=" sign
		call	co
		jmp	ladr		;print <hl>
;
hmath1:		call	ladra		;print <crlf>xxxx
		mov	c,b
		call	co		;print "+" or "-"
		xchg
		call	ladr		;print yyyy
		xchg
		ret
;
;	general purpose input/output routines
;
;	these routines allow byte-by-byte input or output from
;	the current console device.  they are invoked by
;	the monitor "i" or "o" command.
;
;	syntax:
;		Ixx		prints the value from port xx
;		Oxx,vv		outputs the value vv to port xx
;
inport:		call	expr1		;get input port number
		call	crlf
		pop	d
		mov	a,e
		call	dash1
		mov	c,e
		inp	e		;read value into <e> register
;
bits:		mvi	b,8		;loop control for 8 bits
;
bits1:		mov	a,e		;get next bit
		rlc			;into carry
		mov	e,a		;save rest
		mvi	a,'0'/2		;build ascii 1 or 0
		ral			;carry determines which
		mov	c,a		;now, output it
		call	co
		djnz	bits1
		ret
;
otport:		call	expr		;get the address and data for output
		pop	d		;data value into <e>
		pop	b		;port into <c>
		outp	e		;do the output
		ret
;
;	Memory test routine
;
;	this routine tests a block of memory to
;	see if any hard data bit failures exist.  it is
;	not an exhaustive test, but just a quick indication
;	of the memory's operativeness.
;
;	syntax:
;		Jx<cr>		test memory block # x.
;
mtest:		call	expr1		;returns with (L) = block #
		pop	h
		mov	a,l		
		ora	a		;can't test block 0
		jz	cmderr
		cpi	16		;test range
		jnc	cmderr		;out of range error out
		mvi	a,0011_0000b	;get page address of c000-ffffh
		ora	l		;get block # into <A>
;
		lxi	b,00ffh		;init counter and first value
		lxi	d,(32 shl 8) and 0ff00h	;outer loop counter and second value
		lxi	h,0c000h	;start of last active block
		push	h
;
mt1:		mov	m,c		;first init memory alternate ff 00 pattern
		inx	h
		mov	m,e
		inx	h
		djnz	mt1		;do 256 words
;
		dcr	d		;outer loop counter, do 32 * 256 words
		jrnz	mt1
;
mt1_2:		pop	h		;get block start address
		push	h		;and save on stack
;
mt2:		mov	a,c		;start test, check first location
		xra	m		;test
		cnz	mterr		;error if not perfect match
		dcr	m		;change pattern
		inx	h		;next location
		mov	a,e		;test second pattern
		xra	m
		cnz	mterr
		inr	m		;this one counts up
		inx	h
		mov	a,h		;test for overflow to addr 0000, end of pass
		ora	a
		jrnz	mt2		;loop
;
		dcr	c		;next patterns for check
		inr	e
		dcr	d		;do loop 256 times check all patterns
		jrnz	mt1_2
;
		mvi	a,33h		;restore block three
		out	mem_map
		jmp	command		;all done
;
mterr:		push	b
		push	d
		push	h
		mov	e,a
		call	ladrb		;print error address and a blank
		call	bits		; "ones" show error bits
		call	crlf
		pop	h
		pop	d
		pop	b
		ret			;back to main line
;
;	Move routine
;
;	this routine expects three parameters,
;	entered in the following order:
;
;	source first byte address
;	source last byte address
;	destination first byte address
;
;	CAUTION:  this routine does not check for an overlap
;		of source and destination addresses.
;
;	syntax:
;		Mxxxx,yyyy,zzzz		moves a block of memory
;					to zzzz, starting from xxxx
;					to yyyy.
;
move:		call	expr3		;get three parameters
;
mov1:		mov	a,m		;get next byte
		stax	b		;move it
		call	hiloxb		;go increment, check source pointer
		jr	mov1		;not there yet, go do it again
;
;	Single step routine
;
;	this routine examines the instruction pointed
;	to by the <pc> and figures out how many byte
;	to the next instruction, it then sets a break-
;	point at that address and goes to the "go" routine.
;
;	syntax:
;		N		start execution at the
;				current <pc>, and displays
;				the registers after it break-points.
;
fndoff:		inx	d		;point to offset
		ldax	d
		inx	d
		ora	a		;test for negative
		jp	fwdref
		add	e		;neg. offset
		mov	e,a
		jr	exnext
;
fwdref:		add	e	
		mov	e,a
		mvi	a,00		;add 8-bit to 16-bit
		adc	d
		mov	d,a
;
exnext:		push	b		;fix up stack
		lxi	b,010dh		;fake goto
		lxi	h,bk1adr
		jmp	bkgo
;
next:		lded	pc_reg		;get user <pc>
		ldax	d		;get current instruction
		cpi	0ddh		;check for index
		jrz	opixiy		;indexed instruction
		cpi	0fdh
		jrz	opixiy
		cpi	0cbh		;check for bit op
		jz	skip2		;yes, so instr is 2 bytes long
		cpi	0edh		;check for specials
		jrz	edopc
		cpi	09
		jrz	skip1
		cpi	0d9h
		jrz	skip1
		cpi	010h		;check for uncond rel branch
		jrz	spljmp
		cpi	018h
		jrz	fndoff
		lxi	b,4		;search rel jump table
		lxi	h,reltbl
		ccir
		push	psw		;save instr
		mov	a,b
		ora	c		;check for not in table
		jrz	op8080		;must be 8080 op-code
		pop	psw
		ani	0001_1000b	;mask all but cond bits
		call	fndcond		;check cond true/false
		jr	skip2		;condition failed so inc by 2
		jr	fndoff		;cond passed, so find offset
;
badopc:		jmp	cmderr		;can't figure out op-code
					;so do error exit
;
spljmp:		lda	b_reg		;test for djnz instruction
		dcr	a
		jrz	skip2		;test passed so bypass
		jr	fndoff		;onward
;
opixiy:		inx	d		;move past prefix byte
		ldax	d		;get second instr byte
		cpi	0cbh		;all index bit op's are 3 bytes long
		jrz	skip3
		cpi	036h		;special
		jrz	skip3
		cpi	035h
		jrz	skip2
		cpi	034h
		jrz	skip2
		push	psw		;save
		ani	0fh		;check for non8080 opcode
		cpi	006
		jrz	ck4tob		;check high nibble
		cpi	00eh
		jrnz	op8080		;8080 indexed op code
;
ck4tob:		pop	psw		;restore
		push	psw		;resave
		ani	0f0h		;now test upper nibble
		cpi	040h
		jrc	op8080		;not z80
		cpi	0c0h
		jrnc	op8080
		pop	psw
		jr	skip2		;2 byte opcode
;
edopc:		inx	d		;get next byte
		ldax	d
		lxi	b,006		;check for special 'ed' op's
		lxi	h,edtbl
		ccir
		mov	a,b		;test count
		ora	c
		jrz	skip1		;failed
;
skip3:		inx	d		;skip the appropriate # of bytes
;
skip2:		inx	d
;
skip1:		inx	d
		jmp	exnext
;
op8080:		pop	psw		;get byte
		mov	b,a
		lxi	h,byt1tbl
		mvi	c,10		;test for 1-byte instr
		call	stab		;search
		jr	skip1
		lxi	h,byt2tbl
		mvi	c,3		;test for 2-byte instr
		call	stab
		jr	skip2
		lxi	h,byt3tbl	;test for 3-byte instr
		mvi	c,2
		call	stab
		jr	skip3
		lxi	h,xfrtbl
		mvi	c,4
		call	stab		;now check for jmp,call op's
		jr	xfropc
;
spcal:		mov	a,b		;get new copy of instr
		cpi	1110_1001b	;is it a pchl
		jrz	oppchl
		cpi	1100_1001b	;how about a return
		jrz	opret
		ani	1100_0111b	;now check for cond ret
		cpi	1100_0000b
		jrz	condret
		cpi	1100_0111b
		jnz	badopc
		mov	a,b		;new copy
		ani	0011_1000b
		mov	e,a		;must be rst instr
		mvi	d,00
		jmp	exnext
;
condret:	mov	a,b
		call	fndcond		;figure out flags
		jr	skip1		;failed
		jr	opret
;
oppchl:		lded	hl_reg		;get <hl> into <de>
		jmp	exnext		;and exit
;
opret:		lhld	sk_reg		;get sp
		mov	e,m
		inx	h
		mov	d,m		;get return address in <de>
		jmp	exnext
;
xfropc:		mov	a,b
		ani	1		;test for unconditional
		jrnz	uncond
		mov	a,b
		call	fndcond
		jr	skip3
;
uncond:		xchg
		inx	h		;put target in <de>
		mov	e,m
		inx	h
		mov	d,m
		jmp	exnext
;
stab:		mov	a,b
		xra	m
		inx	h
		ana	m
		rz
		inx	h
		dcr	c
		jrnz	stab
;
skipret:	xthl
		inx	h
		inx	h
		xthl
		ret
;
fndcond:	lhld	af_reg		;get flags
		push	h
		ani	0011_1000b	;mask all but flag bits
		rrc
		mov	c,a
		mvi	b,0
		lxi	h,condtbl
		dad	b
		pop	psw
		pchl
;
condtbl:	rz
		jmp	skipret
		rnz
		jmp	skipret
		rc
		jmp	skipret
		rnc
		jmp	skipret
		rpe
		jmp	skipret
		rpo
		jmp	skipret
		rm
		jmp	skipret
		rp
		jmp	skipret
;
;	Examine/change register routine
;
;	examine registers command inspects the values of the
;	the registers stored by the last encountered breakpoint.
;	the values may be modified if desired.
;
;	syntax:
;		R<cr>		displays all z80 registers.
;		R'<cr>		displays the z80 prime registers.
;		Rrr		displays the register rr and
;				opens it for modification.
;		R'rr		displays the z80 prime register rr
;				and opens it for modification.
;
xaa:		inx	h 		;skip over to next entry
		inx	h 
;
xa:		mov	a,m		;get table char
		inr	a		;check for end of table
		rz
		dcr	a		;fix char
		ani	7fh		;kill crlf bit
		cmp	c		;char passed in <c>
		jrnz	xaa		;no, go try again
		call	blk 		;yes, prepare to show current value
		call	prtval		;go print the value
		call	dash 		;prompt a new value
		call	pchk 		;get the input
		rc			;done if carriage return
		jrz	xf 		;jump if no change desired
		push	h 		;to be changed, save pointer
		call	exf 		;get the new value
		pop	h 		;  into <hl>
		mov	a,l 		;get the new low byte
		inx	d		;adjust pointer
		stax	d 		;put it down
		xthl			;recover the table pointer
		mov	a,m		;get the attributes
		xthl			;set the stack straight
		rlc			;see if 8 bit register
		jrnc	xe		;jump if so
		inx	d		;register pair, do other 8 bits 
		mov	a,h 
		stax	d 
;
xe:		pop	h		;restore the table pointer 
;
xf:		mov	a,c		;see if it was a cr 
		cpi	cr 
		rz			;done if so
;
xamin:		lxi	h,actbl		;get address of register look-up table
;
xmne1:		call	pchk		;find out what action is wanted
		mov	c,a		;save char in <c>
		jrc	xg		;show all if carriage return 
		jrz	xmne1		;ignore blanks or commas
		cpi	''''		;see if primes wanted
		jrnz	xa		;no, must be single register
		lxi	h,prmtb		;yes, set table address
		jr	xmne1		;  and find out which one
;
xg:		mov	a,m 
		adi	20h		;make lower case to print
		mov	c,a 
		sui	20h
		inr	a		;see if at end of table
		rz			;done if so
		cm	crlf		;start a new line if bit 7 is set
		call	co 
		call	dash		;prompt for a new value 
		call	prtval		;go print the value
		call	blk		;formatter
		inx	h		;point to next entry
		jr	xg		;do the next value
;
prtval:		inx	h		;point to next entry
		mov	a,m		;get offset and attributes byte
		ani	3fh		;isolate the offset
		push	h
		mov	e,a
		mvi	d,0
		lxi	h,regbase
		ora	a
		dsbc	d
		xchg
		pop	h
		mov	a,m		;now find out attributes
		mvi	b,1		;set up for single reg value
		ani	0c0h		;get attributes
		jrz	onebyt
		cpi	80h
		jrc	tstflg
		jrz	twobyt
		push	h
		lhld	hl_reg
		mov	a,m
		pop	h
		jr	prnbyt
;
tstflg:		push	b
		push	d
		ldax	d
		mov	e,a
		call	bits
		pop	d
		pop	b
		dcx	d		;adjust <de> so fill reg will work
		ret
;
twobyt:		inr	b
;
onebyt:		ldax	d		;get the register contents
;
prnbyt:		call	hex1		;output the value
		dcx	d		;adjust the memory pointer
		djnz	onebyt
		ret
;
;	Search memory routine
;
;	search memory for the entered number of bytes.
;
;	syntax:
;		Sxxxx,yyyy,<byte-1>,<byte-2>,<etc.><cr>
;
;			where xxxx is the starting address,
;			and yyyy is the ending address.
;
search:		call	expr		;get start and end addresses
		popix
		popiy			;save in <x> and <y>
		mvi	d,00		;clear byte count
;
sear1:		mvi	b,01		;init "expr" count
		call	expr		;get byte to search for
		pop	h		;recover from stack
		mov	h,l
		push	h		;save single byte on stack
		inx	sp		;correct stack
		inr	d		;count=count+1
		mov	c,a
		cpi	cr		;check for end of string
		jrnz	sear1		;loop 'till all bytes in
		mvi	h,00
		mov	l,d		;count to <hl>
		dcr	l
		dad	sp		;pointer to search string in <hl>
		push	h
		pushiy
		pop	h		;recover start address to <hl>
		pushix
		mov	b,d
		mov	c,d		;count to <b> and save in <c>
		pop	d		;end address of search
		popix			;<x> has start of search string
;
find:		ldx	a,00		;first byte
		cmp	m		; =??
		jrz	found		;yes, go check rest of string
		call	hilo		;range check
		jc	command		;past end of range
		jr	find		;loop
;
found:		pushix			;save pointer to string
		push	h		;and first address
;
fndlop:		cmp	m		;check rest of string
		jrnz	notfnd		;no compare
		dcxix
		inx	h		;next byte
		ldx	a,0
		djnz	fndlop		;loop as long as they compare
		jrz	tell		;found string, print address
;
notfnd:		pop	h
;
found1:		popix			;restore
		mov	b,c
		inx	h		;point to next memory loc.
		jr	find
;
tell:		pop	h		;restore memory address
		push	b		;save
		call	ladra		;print address
		pop	b
		jr	found1		;some more?
;
;	Comand routine
;
;	this is the main loop of the monitor
;	control is ALLWAYS returned here.
;	interrupts are disabled, and the stack
;	is re-initialized. then the system prompt
;	is issued and the routine waits for a single
;	command character.  the character must be between
;	certain values (A to S) or the system error message
;	will be issued.  if the character is in range
;	command will jump the the proper routine.
;
command:	di			;kill interrupts
		lxi	sp,f_p_reg	;init stack
		call	crlf
		call	decho		;go get first command
		lxi	h,command	;init sp for return
		push	h
		lxi	h,comtbl	;point to command table
		mvi	b,2		;start with 2 parameters
		sui	'A'		;remove ascii bias
		jrc	cmderr		;command error
		cpi	'S'-'A'+1
		jrnc	cmderr
		add	a		;times 2
		mov	e,a
		mvi	d,00
		dad	d		;<hl> now has indirect pointer
		mov	e,m		;get low byte
		inx	h
		mov	h,m		;now high byte
		mov	l,e		;<hl> now point to routine
		ei			;start up interrupt system
		pchl			;goto routine
;
;	Breakpoint routine
;
;	this routine gains control when a restart 1
;	instruction is executed.  the trapped register contents
;	are stored in the register stack area for later
;	use by the "go" and the "examine" registers commands.
;
bkpt:		di			;kill interrupts
		shld	hl_reg		;save user <hl>
		pop	h		;get user <pc>
		dcx	h		;correct
		shld	pc_reg
		push	psw		;save flags first
		pop	h
		shld	af_reg
		lxi	h,0000
		dad	sp		;get user stack
		lxi	sp,sk_reg+2	;point to register save area
		push	h		;save user <sp>
		push	d		;save user registers
		push	b
		dcx	sp		;ajust stack
		dcx	sp
		pushiy
		pushix
		ldar
		mov	b,a
		ldai
		mov	c,a
		push	b		;save <i> & <r> reg's
		exaf
		exx			;swap banks
		push	h
		push	d
		push	b
		push	psw
		lxi	b,bk1byt	;setup to clear bkpt
		lhld	bk1adr		;get a bkpt address
		lded	pc_reg		;get user <pc>
		ora	a		;clear carry
		dsbc	d		;compare bkpt with <pc>
		jrz	clrbkpt		;clear bkpt
		lxi	b,bk2byt
		lhld	bk2adr
		ora	a
		dsbc	d
		jrnz	exbkpt
;
clrbkpt:	ldax	b		;get byte from table
		stax	d		;restore instruction
		xra	a		;clear <a>
		stax	b
		dcx	b
		stax	b
		dcx	b
		stax	b		;clear table
;
exbkpt:		lxi	h,brkmsg
		call	prnmsg
		lhld	pc_reg
		call	ladr
		mvi	c,'='
		call	co
		mov	a,m
		call	hex1
		lxi	h,actbl		;get register table
		call	xg		;go print registers
		jmp	command		;return to monitor
;
;	this is the command error routine
;	it prints and beeps the bell, then jumps back
;	to get another command.
;
cmderr:		lxi	h,errmsg	;get address of question mark msg
cmder1:		call	prnmsg
		jmp	command		;return to main on error
;
ioerr:		lxi	h,iomsg		;print "io err" message
		jr	cmder1
;
;	The cold initialization code
;
init:		di			;disable interrupts
;
;	Set stack RAM, stack-pointer and Initialize memory-mapper
;
;	Since we are guaranteed a minimum of 64k of system memory,
;	set the mapper as follows:
;
;	Page 0 from 0000-3fffh is bank 0 0000-3fffh
;	Page 1 from 4000-7fffh is bank 1 4000-7fffh
;	Page 2 from 8000-bfffh is bank 3 8000-bfffh
;	Page 3 from c000-ffffh is bank 4 c000-ffffh
;
	if reloc
		lxi	h,map_tbl-rom
		mov	b,m		;get byte count
		inx	h
		mov	c,m		;get port number
		inx	h		;point to data
		outir			;block output
	else
		lxi	h,map_tbl	;point to mapper init table
		mov	b,m		;get byte count
		inx	h
		mov	c,m		;get port number
		inx	h		;point to data
		outir			;block output
	endif
;
;	init CTC, PIO, SIO
;
init1:		lxi	sp,regbase	;set stack
;
	if reloc
		lxi	h,init_tbl-rom
		call	ioinit-rom
	else
		lxi	h,init_tbl	;point to initialization table
		call	ioinit
	endif
;
;	clear register save area and local variables
;
		mvi	b,18		;loop count
		lxi	d,0000		;init to zero's
;
init2:		push	d
		djnz	init2		;loop
		lxi	sp,f_p_reg	;init user stack
		sspd	sk_reg
		dcx	d
		sded	disloc
		lxi	d,0
		lxi	h,0
		lxi	b,16
		ldir
;
	if reloc
		lxi	h,0000
		lxi	d,rom
		lxi	b,1000h
		ldir
		jmp	init3
	endif
;
;	Test for auto disk boot
;
;		in	sasi_stat
;		ani	auto_boot
;		xri	auto_boot
;		jnz	boot
;
init3:		mvi	a,high(intvec)	;init int vector register
		stai
		im2			;set mode 2 interrupts
		ei
		lxi	h,signon	;print message
		call	prtmcr		;write it
		jmp	command		;enter monitor
;
ioinit:		mov	a,m
		cpi	0ffh		;end of table flag
		rz
		mov	b,m		;get byte count
		inx	h
		mov	c,m		;get port number
		inx	h		;point to data
		outir			;block output
		jr	ioinit
;
;	routine exf reads one parameter.  it expects the first
;	character of the parameter to be in the <a> register
;	on entry.
;
exf:		mvi	b,1		;set up for one parameter
		lxi	h,0
		jr	ex1		;first character in <a> already
;
;	routine exlf reads two parameters, puts them into the
;	<de> and <hl> registers, then does a carriage return,
;	line feed sequence.
;
exlf:		call	expr		;go get two parameters
		pop	d
		pop	h
		jmp	crlf		;do cr, lf
;
;	routine expr3 gets three parameters, does a cr, lf and 
;	then loads <bc>, <de>, and <hl> with the parameters.
;
expr3:		inr	b		;2 is already in the <b> register
		call	expr		;get the parameters
		pop	b		;put parameters into registers
		pop	d
		jmp	crlfa		;go do the carriage return sequence
;
;	routine expr reads parameters from the console
;	and develops a 16 bit hexadecimal for each one.
;	the number of parameters wanted is in the <b> reg
;	on entry.  a carriage return will terminate the
;	entry sequence, a blank or a comma will end the
;	current parameter entry.  each parameter only
;	takes the last 4 digits typed in, any excess is
;	discarded.  a non-hex digit will terminate the
;	entry sequence and cause a warm boot of the monitor.
;
ex3:		jnz	cmderr		;non-zero is error
;
expr1:		dcr	b		;more parameters?
		rz			;no, return
;
expr:		lxi	h,0		;initialize parameter
;
ex0:		call	echo		;get next number
;
ex1:		mov	c,a		;save char for later use
		call	nibble
		jrc	ex2		;not a number, jump
		dad	h		;multiply by 16
		dad	h
		dad	h
		dad	h
		ora	l		;add on new digit
		mov	l,a
		jr	ex0		;go get next digit
;
ex2:		xthl			;put under return address on stack
		push	h		;restore return address
		mov	a,c		;reget the last character
		call	p2c		;test for delimiter
		jrnc	ex3		;jump if not carriage return
		dcr	b
		jnz	cmderr
;
;		djnz	cmderr		;carret with more param means error
		ret
;
;	General purpose routines
;
;	routine conv converts the low order nibble of the 
;	accumulator to its ascii equivalent.  it
;	puts the result into c for later output.
;
conv:		ani	0fh		;strip off bits 4-7
		adi	90h		;put on the ascii zone
		daa
		aci	40h
		daa
		mov	c,a		;put in output pass register
		ret
;
;	this routine reads a byte from a full-duplex console
;	device, then echoes the character back to the
;	console.
;
decho:		call	dash		;print a dash
;
echo:		call	ci		;console read routine
;
ech1:		push	b		;  save <bc>
		mov	c,a		;  pass character in <c> register
		cpi	space		;don't echo control chars.
		jrc	echo2
		call	co		;  output it
;
echo2:		mov	a,c		;  put character back into <a>
		cpi	'a'		;less than lower case a ?
		jrc	echo3
		sui	space		;no, convert to upper case
;
echo3:		pop	b		;  restore <bc>
		ret
;
;	this routine increments <hl>.  it then checks for and
;	disallows a wrap-around situation.  if it occurs,
;	the carry bit will be set on return.  if no wrap-
;	around occurred, <hl> is compared to <de> and
;	the flag bits set accordingly.
;
hilo:		inx	h		;increment <hl>
		mov	a,h		;test if zero
		ora	l		;  in <hl>
		stc			;set carry for <hl>=0
		rz			;return if <hl> = 0
		mov	a,e		;compare <hl> to <de>
		sub	l
		mov	a,d
		sbb	h
		ret			;return with flags set
;
;	this routine increments <hl>, compares it to <de> and
;	if equal, returns control to the monitor executive.
;	otherwise, control returns to the calling routine.
;
hiloxb:		inx	b		;increment <bc>
;
hilox:		call	hilo		;inc and check <hl>

		jc	command		;done if carry set
		ret
;
;	this routine converts the ascii characters 0-9 and
;	a-f to their equivalent hexadecimal value.  if
;	the character is not in range, the carry bit is set to
;	flag the error.
;
nibble:		sui	'0'		;ascii to hex conversion
		rc			;  done if out of range
		cpi	'G'-'0'		;check upper end
		cmc			;  toggle the carry bit
		rc			;  done if out of range
		cpi	'9'-'0'+1	;see if numeric
		cmc			;  toggle the carry bit
		rnc			;  done if so
		sui	'A'-'9'-1	;subtract the alpha bias
		cpi	10		;  set carry for invalid char
		ret
;
;	this routine reads a character from the console, then
;	checks it for a delimiter. if it is not
;	a delimiter, a non-zero condition is returned.
;	if it is a delimiter, a zero condition is returned.
;	further, if the delimiter is a carriage return,
;	the carry bit is set.  a blank or a comma resets
;	the carry bit.
;
pchk:		call	echo		;get, test for delimiter
;
p2c:		cpi	space		;  blank?
		rz			;  yes, done
		cpi	coma		;  no, comma?
		rz			;  yes, done
		cpi	cr		;  no, carriage return?
		stc			;  show it in carry bit
		rz			;  done if cr
		cmc			;clear carry for no delimiter
		ret
;
;	this routine prints the contents of <hl> on the 
;	current console, either at the start of a new
;	line or at the current location.
;
ladra:		call	crlf		;start a new line
;
ladr:		mov	a,h		;get high two digits
		call	hex1		;print them
		mov	a,l		;get low two digits
;
hex1:		push	psw		;save the low digit
		rrc			;put high nibble into bits 0-3
		rrc
		rrc
		rrc
		call	hex2		;go print single digit
		pop	psw		;reget the low digit
;
hex2:		call	conv		;go insert ascii zone
		jr	co		;do the character output
;
;	this routine types a dash on the current console device.
;
dash1:		call	hex1		;first, print accum as two hex digits
;
dash:		mvi	c,'-'		;get an ascii dash
		jr	co		;go type it
;
;	this routine prints <hl> follow by a space (blank).
;	fall through to "blk".
;
ladrb:		call	ladra		;output <hl> as 4 ascii digits
;
;	this routine prints a single space (blank) on the console.
;
blk:		mvi	c,' '		;output a blank
		jr	co		;print it
;
;	this routine prints an ascii string onto the console.
;	the string must be terminated by bit 7 set in the
;	last character of the string.  the string may start
;	a new line or continue on the same line .
;
prtmcr:		call	crlf		;start a new line
;
prnmsg:		push	b		;save <bc>
;
prta:		mov	c,m		;get next character from memory
		call	co		;output it
		inx	h		;increment memory pointer
		mov	a,c
		rlc			;test for bit 7 delimiter
		jrnc	prta		;no delimiter, go do next character
;
prtb:		pop	b		;restore <bc>
		ret
;
;	this routine generates a carriage return, line feed
;	sequence on the current console to start a new line
;
crlf:		push	h		;save the contents of <hl>
;
crlfa:		lxi	h,crmsg		;address of cr,lf message
		call	prnmsg		;  output it
		pop	h		;restore <hl>
		ret
;
;	this routine handles console interrupts.
;	one of three (3) exits may be taken depending on
;	what was typed on the console:
;
;	1). control C, causes control to be passed directly
;		to the monitor, via the breakpoint routine.
;	2). control S, causes the console display to be
;		held until another character is typed.
;	3). all other characters are ignored, and control is
;		returned to the point that was interrupted.
;
int:		push	b		;save machine state
		push	psw
		mvi	c,cons_stat	;point to SIO data reg.
		mvi	a,rst_irq
		outp	a
		dcr	c
		inp	a		;get character
		pop	b		;restore <bc>
		cpi	cntls		;hold display ?
		jrnz	abort
		call	ci		;wait for next character
;
abort:		cpi	cntlc		;check for abort to monitor
		jrnz	none
		pop	psw		;clear stack and restore <psw>
		jmp	bkpt		;enter monitor through bk-point
;
none:		pop	psw		;return undisturbed
		reti
;
;	monitor console input and output routines.
;
	if not irq_mode
;
chk_abort:	push	psw
		in	cons_stat
		ani	rxrdy
		jrz	abort_ret
		call	ci
		cpi	cntls
		jnz	command
		call	ci
;
abort_ret:	pop	psw
		ret
	endif
;
;	console status routine
;
csts:		mvi	c,cons_stat	;get status port
		inp	a
		ana	b		;xmit or rcv mask passed in <b>
		jrz	csts		;return when xmit is empty or rcv is full
		ret
;
;	console input routine, char. is returned in <a>
;
ci:		di			;not interruptable
		push	b		;save <bc>
		mvi	b,rxrdy		;receive char mask
;
ci_wait:	call	csts		;wait for a char to be typed
		jrz	ci_wait
		dcr	c		;point to data port
		inp	a		;get char, save in <a>
					;return char in <a>
		pop	b		;restore <bc>
		ei
		reti			;reenable interrupt system
;
;	console output routine, char. is passed in <c>
;
co:		push	b		;save <bc>
		mov	a,c		;save char in <a>
		push	psw
		mvi	b,txrdy		;xmit empty mask
;
co_wait:	call	csts		;wait for xmitter to unload
		jrz	co_wait
		pop	psw		;restore output char
		dcr	c		;point to data port
		outp	a
		pop	b
		ret
;
;	system messages
;
booterr		db	cr,lf,' Boot ERROR, type any key to retry '
		db	cr,lf+80h
mot_msg		db	cr,lf,'Index ERROR, type any key to retry '
		db	cr,lf+80h
hom_msg		db	cr,lf,'No diskette, type any key to retry '
		db	cr,lf+80h
;
brkmsg		db	'Break ',' '+80h
errmsg		db	07h,' What','?'+80h
signon		db	'M-Labs, C1 monitor v'
		db	vers+'0','.',rev+'0',release
crmsg		db	cr,lf+80h
iomsg		db	07h,'i/o ER','R'+80h	;bad interrupt
;
;	data tables for single step routine
;
byt1tbl		db	00,0c7h
		db	03,0c3h
		db	09,0cfh
		db	02,0e7h
		db	04,0c6h
		db	40h,0c0h
		db	80h,0c0h
		db	0c1h,0cbh
		db	0e3h,0e7h
		db	0f9h,0ffh
;
byt2tbl		db	06,0c7h
		db	0c6h,0c7h
		db	0d3h,0f7h
;
byt3tbl		db	22h,0e7h
		db	01,0cfh
;
xfrtbl		db	0c3h,0ffh
		db	0cdh,0ffh
		db	0c2h,0c7h
		db	0c4h,0c7h
;
reltbl		db	20h,28h,30h,38h
;
edtbl		db	43h,4bh,53h,5bh
		db	73h,7bh
;
;	data tables for examine/change registers
;	register name and offset from top of register save area.
;
actbl		db	80h+'A',regbase-a_reg
		db	'F',(regbase-f_reg) or 40h
		db	'B',regbase-b_reg
		db	'C',regbase-c_reg
		db	'D',regbase-d_reg
		db	'E',regbase-e_reg
		db	'H',regbase-h_reg
		db	'L',regbase-l_reg
		db	'M',0c0h
		db	'S',(regbase-sk_reg-1) or 80h
		db	'P',(regbase-pc_reg-1) or 80h
;
;	Rest of z-80, (prime), register offsets
;
prmtb		db	80h+'A',regbase-a_p_reg
		db	'F',(regbase-f_p_reg) or 40h
		db	'B',regbase-b_p_reg
		db	'C',regbase-c_p_reg
		db	'D',regbase-d_p_reg
		db	'E',regbase-e_p_reg
		db	'H',regbase-h_p_reg
		db	'L',regbase-l_p_reg
		db	'I',regbase-i_reg
		db	'R',regbase-r_reg
		db	'X',(regbase-x_reg-1) or 80h
		db	'Y',(regbase-y_reg-1) or 80h
;
		db	0ffh		;end of table
;
;
;	Memory mapper initilization table
;	Memory consists of 4 64K banks which is mapped as 
;	16 PAGES of 16 K. each. Only 4 are available to the CPU at
;	any instant and these are termed as the  4 BLOCKS.
;	The function of the mapper is to translate each of the four 
; logical_                 _into a unique->--
;         |                |                | 
;         |                |                |
;         -Block Address->--                |_Page Address
;          (Cell Address)                     (Cell Data)
;        /-------^-------\                 /-------^-------\
;       |                 |               |                 |
;       |                 |          _____|___ Bank # MSB   |
;       |                 |          |____|___ Bank # LSB   |
;       | Mapper Cell LSB_|_________ ||___|___ A7  Value    |
;       | Mapper Cell MSB_|_______ | |||__|___ A14 Value    |
;				 | | ||||
;     Mapper port write bits -> 7654 3210
;				---- ----
;  (inital	Page 0, block 0 x0x0 0000  =00h \
;  values)	Page 1, block 1 x0x1 0010  =12h  |-->--
;		Page 2, block 2 x1x0 0001  =41h  |    |
;		Page 3, block 3 x1x1 0011  =53h /     |
;						      | 	
map_tbl		db	4,mem_map       ;             |
		db	00H,12h,41h,53h	;<-------------
		db	0ffh
;
;
;	CTC, SIO, PIO initialization tables
;
init_tbl	equ	$		;start of initialization tables
;
dma_tbl		db	6,dma		;reset the DMA controller
		db	0c3h,0c3h,0c3h
		db	0c3h,0c3h,0c3h
;
ctc_tbl		db	2,ctc_u5+cntr_a	;byte count, base timer
					; 0--- ----  No interrupts
					; -1-- ----  counter mode
					; --0- ----  /16 prescaler
					; ---1 ----  rising edge trigger
					; ---- 0---  start at load 
					; ---- -1--  time constant follows
					; ---- --1-  software reset
					; ---- ---1  control word
					;+__________
		db	0101_0111b	;=0101 0111
;
					; 1.8432Mhz / 12 (time const.)
					; = 156.300khz
					;_________
		db	12		; /16 in SIO = 9600 baud
;
sio_tbl		db	9,cons_stat	;byte count and base port address
		db	0001_1000b	;software reset command
;
;	It is mandatory to initialize write port # 4 first
;
		db	wr4		; write register # 4
					; 01-- ----  16X clock mode
					; --00 ----  8 bit sync char.
					; ---- 01--  1 stop bit / char.
					; ---- --0-  odd parity
					; ---- ---0  parity disable
					;+__________
		db	0100_0100b	;=0100 0100
;
		db	wr1		; write register #1
					; 000- ----  No wait/ready
					; ---0 0---  disable Rx interrupts
					; ---- -0--  no stat affect vector
					; ---- --0-  disable tx interrupts
					; ---- ---0  disable ext interrupts
					;+__________
		db	0000_0000b	;=0000 0000
;
		db	wr3		; write register #3
					; 11-- ---- 8 bits / character
					; --0- ---- no auto enables
					; ---0 ---- don't enter hunt
					; ---- 0--- Rx CRC disabled
					; ---- -0-- not in SDLC
					; ---- --0- sync char load inhibit
					; ---- ---1 Rx is enabled
					;+__________
		db	1100_0001b	;=1100 0001
;
		db	wr5             ; write register #5
					; 0--- ---- DTR clear
					; -11- ---- 8 Tx char bits
					; ---0 ---- don't send break
					; ---- 1--- Tx enabled
					; ---- -0-- not SLDC
					; ---- --0- RTS inactive
					; ---- ---0 Tx CRC disabled
					;+__________
		db	0110_1000b	;=0110 1000
;
		db	2,list_stat
		db	wr2		; write register # 2
					; interrupt vector is
					; B side of SIO
		db	low(intvec) and 0feh
;
pio_tbl		db	1,sys_ctl	;byte count and port
		db	0000_0000b	;set data register to zeros
;
		db	2,sys_ctl+2	;byte count and port
		db	1111_1111b	;mode 3, (control mode)
		db	1100_0000b	;d7&d6 input, rest output
;
		db	1,sys_stat
		db	0000_0000b
;
		db	2,sys_stat+2	;byte count and base pio address
		db	1111_1111b	;mode 3, (control mode)
		db	1001_1111b	;all input, except d6&d5
;
		db	0ffh		;end of table
;		
;
		end			;THATS ALL FOLKS
