(* This is the file open.pas, containing open and close for Pascal/M.  Note:
  because M does not allow files to be part of records, we have changed all
  the file procedures to use CASE statements.  Most of the CASE statements
  have 'otherwise' clauses; these might be filled with error-recovery routines
  of your choice. *)
(*  declared elsewhere:

Procedure message ( billboard : string ); forward;

procedure tool_close (fd : filedesc); forward;
*)

{ open -- (M) open a file for reading or writing }
{   non-portable -- uses the standard M file opening commands }
{   status can be returned, fortunately }
function open (var name : string; mode : integer) : filedesc;
const
    open_err_msg = ': error in opening file (open)';
var
    iohold : integer;
    fd : filedesc;
    i : filesearch;	(* not filedesc because we can overflow if no file is
			  found *)
    erasefile,		(* Should we erase the file before opening it ? *)
    found : boolean;

procedure file_open ( var fd : filedesc; var fyle : text; var name : argstring;
		prevopened : boolean );
var
  iohold : integer;
begin
(*$I- turn off i/o checking so we aren't interrupted by the system *)
  if prevopened then begin
    close(fyle);
    iohold := ioresult;
    if (iohold <> 0) then begin
      putstr(name, STDERR);
      message(': i/o error on closing file for reopening');
      fd := IOERROR;
      exit(file_open);
    end (* if iohold *);
(*  if (openlist[i].mode = IOWRITE) then remove(name); *)
(* for systems that require resetting a file a second time without an outside
  world name, see OPEN.SRC for an example. *)
  end (* if prevopened *);
  if (openlist[i].mode = IOREAD) then
    reset(fyle, name)
  else begin
    remove(name);	(* this is so we don't get the "file already exists"
			  error *)
    rewrite(fyle, name);
  end;
  iohold := ioresult;
{DIAG}
  writeln('In open_file, ioresult is : ', iohold);
  if (iohold <> 0) then
  begin
    putstr(name, STDERR);
    message(open_err_msg);
    fd := IOERROR;		(* otherwise, we finished successfully *)
  end;

(*$I+ turn i/o checking back on *)
end (* file_open *);

begin (* open *)

  { find a free slot in openlist }
  open := IOERROR;	(* if no slot found, error. This will be the default
			  value if we exit open abnormally. *)
  found := false;
  i := FIRSTAVAIL;	(* start there because first 3 files are taken
				  by standard files; don't bother *)
		{STDIN, STDOUT, STDERR are 1, 2, 3 }
{DIAGNOSTIC}
WRITELN('IN OPEN, NAME = ',NAME);
  while (i <= LASTAVAIL) and (not found) do
  begin
    found := equal(name, openlist[i].name);
    i := i + 1;
  end;
  i := i - 1;	(* we increment once too often *)
  if found then begin
    if (openlist[i].mode = IOERROR) then begin
      putstr(name, STDERR);
      message(': attempt to open file with errors');
      exit(open);	(* OPEN will have value IOERROR *)
    end;
    erasefile := true;
  end
  else begin		(* searching for an empty slot *)
    i := FIRSTAVAIL;
    while (i <= LASTAVAIL) and (not found) do
    begin
      found := openlist[i].mode = IOAVAIL;
      i := i + 1;
    end;
    i:= i - 1;	(* once again to get number correct *)
    if (not found) then begin
      putstr(name, STDERR);
      message(': no open slot available');
      exit(open);	(* OPEN will have value IOERROR *)
    end
    else
      openlist[i].name := name;
    erasefile := false;
  end   (* if found *);
  openlist[i].mode := mode;	(* set mode to correct mode *)
  (* we have now found the slot and done all work preparatory to sending it off
    to be opened. *)
  fd := i;
  case fd of
    4: file_open(fd, file1, name, erasefile);	(* the numbers should *)
    5: file_open(fd, file2, name, erasefile);	(* correspond to *)
    6: file_open(fd, file3, name, erasefile);	(* the files you *)
    7: file_open(fd, file4, name, erasefile);	(* declared in *)
    8: file_open(fd, file5, name, erasefile);	(* TOOLSINC.PAS or such. *)
  otherwise
    putstr(name, STDERR);
    error(open_err_msg);
  end (* case *);
  open := fd;
{DIAGNOSTIC}
      writeln('File name is ', name, ' File descriptor number is ', fd);
end(* open *);

{ mustopen -- (M/MT+) open file or DIE }
function mustopen (var name : string; mode : integer)
	: filedesc;
var
    fd : filedesc;
begin
    fd := open(name, mode);
    if (fd = IOERROR) then begin
	putstr(name, STDERR);
	error(': can''t open file')
    end;
    mustopen := fd
end (* mustopen *);

