/* "grep" Generalized regular expression pattern matcher */
/* 'n' flag says display line numbers */
/* 'v' flag says show only failures instead of successes */
/* 'c' flag says case counts in matches */

#include "stdio.h"
#include "printf.h"

static int linenums, reverse, casecounts;

main(argc, argv)
char **argv;
{
	auto struct fcb fcb;
	auto FILE fil;
	FILE *fp;
	auto char buf[8192];
	char *pattern;
	int i;


	if (argc < 2 || argv[1][0] == '-' &&
	    (!strany(argv[1][1], "nvc") || argc < 3)) {
		fprintf(stderr,
			"Usage: grep [-nvc] pattern file1 file2 ... \n");
		exit(1);
	}
	pattern = argv[1];
	linenums = reverse = casecounts = 0;
	if (pattern[0] == '-') {
		/* Handle switches */
		if (strany('n', pattern)) ++linenums;
		if (strany('v', pattern)) ++reverse;
		if (strany('c', pattern)) ++casecounts;
		++argv;
		--argc;
		pattern = argv[1];
	}
	fp = &fil;
	fp->f_fcbp = &fcb;
	fp->f_max = sizeof(buf);
	fp->f_datap = buf;
	for (i = 2; i < argc; i++) {
		fp = fileopen(&fil, argv[i], "r");
		if (fp == NULL) {
			fprintf(stderr, "%s: Cannot open\n", argv[i]);
			continue;
		}
		do1grep(fp, pattern, argc > 3? argv[i]: "");
	}
	if (argc == 2) do1grep(stdin, pattern, "");
	exit(0);
}

do1grep(fp, pattern, fname)
FILE *fp;
register char *pattern;
char *fname;
{
	/* scan until eof, looking for pattern */
	char line[256];
	int begonly;
	register char *cp;
	int lnum, mflg;
	register int fast, fast2, c1;

	begonly = 0;
	if (pattern[0] == '^') {
		/* Only match at beg. of line */
		++begonly;
		++pattern;
	}
	/* Set fast to first char of pattern unless special */
	fast = pattern[0];
	if (begonly || strany(fast, "*[?")) {
		fast = 0;	/* Optimization not useful */
	} else if (fast == '\\') {
		fast = pattern[1];
	}
	fast2 = casecounts? fast: upper(fast);
	for (lnum = 1; getln(fp, cp = line, sizeof(line)); ++lnum) {
		/* Mflg = 1 if line matches pattern */
		mflg = 0;
		/* cp == &line[0] */
		if (fast) {
			/* First char is not special, do quicker check */
			for ( ; c1 = *cp; ++cp) {
				if (c1 != fast && c1 != fast2) continue;
				if (match(cp, pattern)) {
					++mflg;
					break;
				}
			}
		} else if (begonly) {
			/* First char is '^', only check first position */
			if (match(cp, pattern)) ++mflg;
		} else {
			/* Slowest case, first char is special */
			for ( ; *cp; ++cp) {
				if (match(cp, pattern)) {
					++mflg;
					break;
				}
			}
		}
		if (mflg != reverse) {
			/* match, or no match and "-v" set */
			if (fname && *fname)
				printf("%s:", fname);
			if (linenums) printf("%d:", lnum);
			printf("%s\n", line);
		}
	}
}

getln(fp, line, max)
register FILE *fp;
char *line;
int max;
{
	/* Read in a line, null-terminate (do not copy newline) */
	/* Return non-zero unless EOF */
	/* Break line if longer than max */
	/* Max must be at least 2!! */
	register char *cp;
	register c;

	c = getc(fp);
	if (c == EOF) return(0);
	cp = line;
	max -= 2;	/* Pre decrement to leave room for null-term */
	while (c != '\n' && c != EOF) {
		*cp++ = c;
		/* Check if buffer full */
		if (--max < 0) break;
		c = getc(fp);
	}
	*cp = 0;
	return(1);
}

match(str, pat)
register char *str;
char *pat;
{
	/* Return non-zero if head of str matches pattern */
	/*** Recursive ***/
	register char *pp;
	static int pc, sc;

	for (pp = pat; pc = *pp++; ) {
		switch(pc) {
		case '\\':
			/* Escape character */
			if (*str++ == *pp++) continue;
			break;
		case '?':
			/* Match any single char */
			if (*str++) continue;
			break;
		case '*':
			/* Match any string */
			do {
				/*** Recurse *** (sc, pc clobbered) ***/
				if (match(str, pp)) return(1);
			} while (*str++);
			break;
		case '[':
			/* See if matches any char within brackets */
			pc = 0;
			sc = *str++;
			if (!casecounts) sc = lower(sc);
			while (*pp && *pp != ']') {
				if (*pp == '-' && pc) {
					/* Specified a range */
					++pp;
					if (sc > pc && sc <= *pp) {
						pc = sc;
						break;
					}
				}
				pc = *pp++;
				if (pc == '\\' && *pp) pc = *pp++;
				if (sc == pc) break;
			}
			while (*pp && *pp++ != ']');
			if (sc == pc) continue;
			break;
		case '$':
			/* Match end of line, if at end of pattern */
			if (*pp == 0) {
				if (*str == 0) continue;
				break;
			}
			/* drop into default */
		default:
			sc = *str++;
			if (!(pc -= sc)) continue;
			/* Fail if more than a bit off or case counts */
			if (casecounts || pc != ('a'-'A')) break;
			/* Succeed if sc alphabetic */
			if (sc >= 'A' && sc <= 'Z') continue;
			break;
		}
		return(0);  /* failure */
	}
	return(1);	/* success */
}
	}
	