
REM *************************************************************************
REM *                                                                       *
REM *         GRAPHICS.BAS,  Version 4.0,  June 1983,  R.Schrader.          *
REM *                                                                       *
REM *               Copyright 1983, Kaypro Corporation.                     *
REM *                                                                       *
REM *     Graphics routines to be used on a Kaypro 10 with the S-BASIC      *
REM *     compiler.  To include, use:  #INCLUDE "GRAPHICS" .                *
REM *                                                                       *
REM *************************************************************************


REM Video attributes for Kaypro 10.
REM Attributes for 'set_on' and 'set_off' are as follows:
REM	inverse   = '0'
REM	reduced   = '1'
REM	blinking  = '2'
REM	underline = '3'
REM	cursor    = '4'
REM Note: these are the characters 0 though 4, not 0th through 4th ASCII.

PROCEDURE SET_ON ( ATT = CHAR )
	PRINT CHR(27); 'B'; ATT;
END

PROCEDURE SET_OFF ( ATT = CHAR )
	PRINT CHR(27); 'C'; ATT;
END


REM When dealing with screen addresses on a Kaypro 10, an increment
REM of 32 is used ( 31 if you like natural numbers ); this is neccesary
REM because some control characters ( ie: characters less than 32 ASCII )
REM are intercepted by the system.

REM Thus the upper-leftmost point on the screen is: ( 32, 32 ).
REM The 5 following procedures add 31 so that the upper-leftmost point
REM is 'naturally':  ( 1 , 1 )  

REM All invalid coordinates are wrapped around in an 8-bit register by 
REM the BIOS.  It, unfortuately, returns no error messages.

REM Check your Kaypro 10 manual for a discussion of escape sequences
REM if you want to understand the use of 'CHR(27)'.

PROCEDURE POSITION ( VERT, HORZ = CHAR )
	REM Position the cursor.
	PRINT CHR(27); '=' ; VERT + 31; HORZ + 31;
END

PROCEDURE PIXON ( VERT1, HORZ1 = CHAR )
	REM Turn on a pixel.
	PRINT CHR(27); '*'; VERT1 + 31; HORZ1 + 31;
END

PROCEDURE PIXOFF ( VERT1, HORZ1 = CHAR )
	REM Turn off a pixel.
	PRINT CHR(27); ' '; VERT1 + 31; HORZ1 + 31;
END

PROCEDURE LINEON ( VERT1, HORZ1, VERT2, HORZ2 = CHAR )
	REM Draw a line of pixels.
	PRINT CHR(27); 'L'; VERT1 + 31; HORZ1 + 31; VERT2 + 31; HORZ2 + 31;
END

PROCEDURE LINEOFF ( VERT1, HORZ1, VERT2, HORZ2 = CHAR )
	REM Erase a line of pixels.
	PRINT CHR(27); 'D'; VERT1 + 31; HORZ1 + 31; VERT2 + 31; HORZ2 + 31;
END



FUNCTION SQRT( N = INTEGER ) = INTEGER
	REM Square root detemined by Newton's method.
	REM This is used by the circle drawing routine.
	VAR ROOT1, ROOT2 = INTEGER

	ROOT1 = N
	ROOT2 = 1
	WHILE ( ROOT1 > ROOT2 ) DO BEGIN
		ROOT1 = ( ROOT1 + ROOT2 ) / 2
		ROOT2 = N / ROOT1
	END
END = ROOT1


PROCEDURE CIRCLE ( CENTER_Y, CENTER_X, RADIUS = INTEGER )
	REM Center_x and center_y are cartesian coordinates of the circle.

	VAR X, Y, OFFSET = INTEGER
	REM X and Y are the coordinates of the point being drawn.
	REM Offset is the difference between either axis and a point on the
	REM circle corresponding to 45 degrees.  The location of offset is
	REM critical to maintain a constant density of pixels.

	REM Offset = square root of 2 * radius.  
	OFFSET = ( RADIUS * 100 ) / 141

	REM The top and bottom points of the circle, adjusted
	REM slightly inward for visual continuity.
	PIXON CENTER_Y + RADIUS - 1, CENTER_X
	PIXON CENTER_Y - RADIUS + 1, CENTER_X

	REM The top and bottom quarters of the circle.
	FOR X = 1 TO OFFSET
		Y = SQRT( RADIUS^2 - X^2 ) 		
		PIXON CENTER_Y + Y , CENTER_X + X 
		PIXON CENTER_Y + Y , CENTER_X - X 
		PIXON CENTER_Y - Y , CENTER_X + X 
		PIXON CENTER_Y - Y , CENTER_X - X 
 	NEXT 

	REM The leftmost and rightmost points of the circle, adjusted
	REM slightly inward for visual continuity.
	PIXON CENTER_Y, CENTER_X + RADIUS - 1
	PIXON CENTER_Y, CENTER_X - RADIUS + 1

	REM The left and right quarters of the circle.
	FOR Y = 1 TO OFFSET
		X = SQRT( RADIUS^2 - Y^2 )
		PIXON CENTER_Y + Y , CENTER_X + X 
		PIXON CENTER_Y + Y , CENTER_X - X 
		PIXON CENTER_Y - Y , CENTER_X + X 
		PIXON CENTER_Y - Y , CENTER_X - X 
 	NEXT 
END


PROCEDURE RECTANGLE ( Y, X, HEIGHT, WIDTH = INTEGER )
	REM X and Y are cartesian coordinates of the upper left corner.
	LINEON Y, X, Y + HEIGHT-1, X
	LINEON Y, X + WIDTH-1, Y + HEIGHT-1, X + WIDTH-1
	LINEON Y, X, Y, X + WIDTH-1
	LINEON Y + HEIGHT-1, X, Y + HEIGHT-1, X + WIDTH-1
END


PROCEDURE RECT_FILL ( Y, X, HEIGHT, WIDTH = INTEGER )
	REM Solid filled rectangle.  X and Y are upper left coordinates.
	REM Filled by a series of horizontal lines.
	VAR I = INTEGER
	FOR I = Y TO Y + HEIGHT-1
		LINEON I, X, I, X + WIDTH-1
	NEXT I
END


PROCEDURE RECT_SHADE(Y, X, HEIGHT, WIDTH = INTEGER)
	REM Checkered rectangle.  X and Y are upper left coordinates.

	VAR T, B, L, R  = INTEGER
	REM T and B are vertical coordinates of the top and bottom.
	REM L and R are horizontal coordinates of the left and right.

	REM The algorithm draws a series of parallel lines at 45 degrees,
	REM skipping 1 line every time.  The result is a checkerbord pattern.

	REM The first line of the series is drawn across the upper left
	REM corner.  Progressively longer lines are drawn until one of
	REM the endpoints reaches the top right or bottom left corner.
	REM At this point, a triangle has been shaded.  Then lines of 
	REM constant length are drawn, shading in a parallelogram, until
	REM another corner is encountered.  Finally, the lines begin to
	REM decrease, shading the final triangle.

	REM             +---------------------------------------+
	REM             |       /                             / |
	REM   first ->  |     /       parallelogram         /   |  <- last
	REM  triangle   |   /                             /     |    triangle
	REM             | /                             /       |
	REM             +---------------------------------------+
  
	VAR TR_FLAG, BL_FLAG = INTEGER
	REM These keep track of which corners have been encountered.
	REM At this point, we have encountered neither corner, so...
	TR_FLAG = 0
	BL_FLAG = 0

	REM Draw the border.
	RECTANGLE Y, X, HEIGHT, WIDTH

	REM Set endpoints for short line in upper left.
	T = Y
	L = X
	B = Y
	R = X

	REPEAT
	    BEGIN
		REM Draw the current line.
		LINEON T, R, B, L

		REM Adjust the top right endpoint.
		IF ( R + 2 ) <= ( X + WIDTH - 1 ) \
			THEN R = R + 2
			ELSE IF TR_FLAG \
				THEN T = T + 2
				ELSE BEGIN
					T = Y + 2 + ( R - ( X + WIDTH - 1) )
					R = X + WIDTH - 1
					TR_FLAG = 1
				END

		REM Adjust the left bottom endpoint.
		IF ( B + 2 ) <= ( Y + HEIGHT - 1 ) \
			THEN B = B + 2
			ELSE IF BL_FLAG \
				THEN L = L + 2
				ELSE BEGIN
					L = X + 2 + ( B - ( Y + HEIGHT - 1 ) )
					B = Y + HEIGHT - 1
					BL_FLAG = 1
				END
	    END
	UNTIL ( ( T >= B ) OR ( L >= R ) )
	REM Stop when there is a line length of 0.
END


PROCEDURE SQUARE ( Y, X, SIDE = INTEGER )
	REM A square is just a special instance of a rectangle.
	RECTANGLE Y, X, SIDE, SIDE
END


PROCEDURE BAR( REF_Y,REF_X,HEIGHT,WIDTH,DEPTH,INSIDE,VERT_P,HORZ_P = INTEGER)
	REM Draws bars for a bar graph in 45 degree perspective.
	REM Ref_x and ref_y are coordinates of reference point of bar.
	REM Inside is a boolean value indicating filled or empty.
	REM Vert_p and horz_p are values for vertical and horizontal
	REM perspective.  When they both are 1, the bar appears to 
	REM extend down to the left.  In the bar shown below, horz_p = 1,
	REM vert_p = -1.

	VAR CENTER_X, CENTER_Y, REAR_X, REAR_Y = INTEGER

	REM
	REM                      _________    <- rear point
	REM                     /        /|  
	REM                   /________/  |
	REM                   |       |   |    <- center
	REM                   |       |   |        point
	REM                   |       |   |
	REM                   |       |   |
	REM                   |       |   |
	REM                   |       |   |
	REM                   |       |   |
	REM    reference      |       |  /
	REM      point  ->    |_______|/
	REM

	REM Drawing a 3-D bar constists of drawing 3 sides ( one rectangle
	REM and two parallelograms ) and 6 lines.  When the bar is filled,
	REM the external lines are redundant.

	VAR I = INTEGER
	REM A local counter.

	REM Define other critical points on the bar.
	CENTER_X = REF_X    + ( WIDTH * HORZ_P )
	CENTER_Y = REF_Y    + ( HEIGHT * VERT_P )
	REAR_X   = CENTER_X + ( DEPTH * HORZ_P )
	REAR_Y   = CENTER_Y + ( DEPTH * VERT_P )

	REM Draw empty bar.
	IF ( INSIDE = 0 ) THEN BEGIN
		REM Clear space for it.
		REM Clear rectangle.
		FOR I = REF_X TO CENTER_X STEP HORZ_P
			LINEOFF REF_Y, I, CENTER_Y, I
		NEXT I
		REM Clear two parallelograms.
		FOR I = 0 TO ( WIDTH * HORZ_P ) STEP HORZ_P 
			LINEOFF CENTER_Y, CENTER_X - I, REAR_Y, REAR_X - I
 		NEXT I
		FOR I = 0 TO ( HEIGHT * VERT_P ) STEP VERT_P
			LINEOFF CENTER_Y - I, CENTER_X, REAR_Y - I, REAR_X
		NEXT I

		REM Draw the outline.
		LINEON REF_Y, REF_X, CENTER_Y, REF_X
		LINEON REF_Y, REF_X, REF_Y, CENTER_X
		LINEON CENTER_Y, CENTER_X, CENTER_Y, REF_X
		LINEON CENTER_Y, CENTER_X, REF_Y, CENTER_X
		LINEON REAR_Y, REAR_X, REAR_Y - ( HEIGHT * VERT_P ), REAR_X
		LINEON REAR_Y, REAR_X, REAR_Y, REAR_X - ( WIDTH * HORZ_P )

		REM Draw internal lines.
		LINEON CENTER_Y, CENTER_X, REAR_Y, REAR_X
		LINEON REF_Y, CENTER_X, REAR_Y - ( HEIGHT * VERT_P), REAR_X
		LINEON CENTER_Y, REF_X, REAR_Y, REAR_X - ( WIDTH * HORZ_P )
	END

	REM Draw filled bar.
	IF ( INSIDE = 1 ) THEN BEGIN
		REM Fill the entire area.
		REM Fill rectangle.
		FOR I = REF_X TO CENTER_X STEP HORZ_P
			LINEON REF_Y, I, CENTER_Y, I
		NEXT I
		REM Fill two parallelograms. 
		FOR I = 0 TO ( WIDTH * HORZ_P ) STEP HORZ_P 
			LINEON CENTER_Y, CENTER_X-I, REAR_Y, REAR_X - I
 		NEXT I
		FOR I = 0 TO ( HEIGHT * VERT_P ) STEP VERT_P
			LINEON CENTER_Y-I, CENTER_X, REAR_Y - I, REAR_X
		NEXT I

		REM Erase the three internal lines.
		LINEOFF CENTER_Y, CENTER_X, CENTER_Y, REF_X
		LINEOFF CENTER_Y, CENTER_X, REF_Y, CENTER_X
		LINEOFF CENTER_Y, CENTER_X, REAR_Y, REAR_X
	END
END
REF_X
		LINEON 