/* xwdtopbm.c - read an X11 or X10 window dump file and write a portable bitmap
**
** Copyright (C) 1988 by Jef Poskanzer.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
*/

#include <stdio.h>
#include "pbm.h"
#include "x10wd.h"
#include "x11wd.h"

main( argc, argv )
int argc;
char *argv[];
    {
    FILE *ifd;
    bit **bits, getbit();
    int rows, cols, padright, row, col;

    if ( argc > 2 )
	{
	fprintf( stderr, "usage:  %s [xwdfile]\n", argv[0] );
	exit( 1 );
	}

    if ( argc == 2 )
	{
        ifd = fopen( argv[1], "r" );
        if ( ifd == NULL )
	    {
	    fprintf( stderr, "%s: can't open.\n", argv[1] );
	    exit( 1 );
	    }
	}
    else
	ifd = stdin;

    getinit( ifd, &cols, &rows, &padright );

    bits = pbm_allocarray( cols, rows );

    for ( row = 0; row < rows; row++ )
	{
        for ( col = 0; col < cols; col++ )
	    bits[row][col] = getbit( ifd );
        for ( col = 0; col < padright; col++ )
	    (void) getbit( ifd );
	}

    if ( ifd != stdin )
	fclose( ifd );
    
    pbm_writepbm( stdout, bits, cols, rows );

    exit( 0 );
    }


char buf[4];
char *byteP;
short *shortP;
long *longP;
int bits_per_item, bits_used, bit_shift, bit_order, bit_invert, byte_swap;

short bs_short();
int bs_int();
long bs_long();


getinit( file, colP, rowP, padrightP )
FILE *file;
int *colP, *rowP, *padrightP;
    {
    /* Assume X11 headers are larger than X10 ones. */
    unsigned char header[sizeof(X11WDFileHeader)];
    X10WDFileHeader *h10P;
    X11WDFileHeader *h11P;
    char junk[10000];
    int i;

    h10P = (X10WDFileHeader *) header;
    h11P = (X11WDFileHeader *) header;

    if ( sizeof(*h10P) > sizeof(*h11P) )
	{
	fprintf( stderr, "ARGH!  On this machine, X10 headers are larger than X11 headers!\n" );
	fprintf( stderr, "You will have to re-write xwdtopbm.\n" );
	exit( 1 );
	}

    /* Read an X10 header. */
    if ( fread( &header[0], sizeof(*h10P), 1, file ) != 1 )
	{
	fprintf( stderr, "Couldn't read XWD file header.\n" );
	exit( 1 );
	}

    if ( h10P->file_version == X10WD_FILE_VERSION ||
	 bs_int( h10P->file_version ) == X10WD_FILE_VERSION )
	{
	if ( h10P->file_version != X10WD_FILE_VERSION )
	    {
	    byte_swap = 1;
	    h10P->header_size = bs_int( h10P->header_size );
	    h10P->file_version = bs_int( h10P->file_version );
	    h10P->display_type = bs_int( h10P->display_type );
	    h10P->display_planes = bs_int( h10P->display_planes );
	    h10P->pixmap_format = bs_int( h10P->pixmap_format );
	    h10P->pixmap_width = bs_int( h10P->pixmap_width );
	    h10P->pixmap_height = bs_int( h10P->pixmap_height );
	    h10P->window_width = bs_short( h10P->window_width );
	    h10P->window_height = bs_short( h10P->window_height );
	    h10P->window_x = bs_short( h10P->window_x );
	    h10P->window_y = bs_short( h10P->window_y );
	    h10P->window_bdrwidth = bs_short( h10P->window_bdrwidth );
	    h10P->window_ncolors = bs_short( h10P->window_ncolors );
	    }
	if ( fread( junk, h10P->header_size - sizeof(*h10P), 1, file ) != 1 )
	    {
	    fprintf( stderr, "Couldn't read rest of X10 XWD file header.\n" );
	    exit( 1 );
	    }
	if ( fread( junk, sizeof(X10Color), h10P->window_ncolors, file ) !=
	     h10P->window_ncolors )
	    {
	    fprintf( stderr, "Couldn't read X10 XWD colormap.\n" );
	    exit( 1 );
	    }

	/* Check whether we can handle this dump. */
	if ( h10P->window_ncolors != 0 )
	    {
	    fprintf( stderr, "Can't handle X10 window_ncolors != 0.\n" );
	    exit( 1 );
	    }
	if ( h10P->pixmap_format != XYFormat )
	    {
	    fprintf( stderr, "Can't handle X10 pixmap_format %d.\n",
		     h10P->pixmap_format );
	    exit( 1 );
	    }

	*colP = h10P->pixmap_width;
	*rowP = h10P->pixmap_height;
	*padrightP =
	    ( ( h10P->pixmap_width + 15 ) / 16 ) * 16 - h10P->pixmap_width;
	bits_per_item = 16;
	bits_used = bits_per_item;
	bit_order = LSBFirst;
	bit_invert = 1;
	}
    else
	{
	if ( h11P->file_version == X11WD_FILE_VERSION ||
	     bs_long( h11P->file_version ) == X11WD_FILE_VERSION )
	    {
	    if ( fread( &header[sizeof(*h10P)], sizeof(*h11P) - sizeof(*h10P), 1, file ) != 1 )
		{
		fprintf( stderr, "Couldn't read X11 XWD file header.\n" );
		exit( 1 );
		}
	    if ( h11P->file_version != X11WD_FILE_VERSION )
		{
		byte_swap = 1;
		h11P->header_size = bs_long( h11P->header_size );
		h11P->file_version = bs_long( h11P->file_version );
		h11P->pixmap_format = bs_long( h11P->pixmap_format );
		h11P->pixmap_depth = bs_long( h11P->pixmap_depth );
		h11P->pixmap_width = bs_long( h11P->pixmap_width );
		h11P->pixmap_height = bs_long( h11P->pixmap_height );
		h11P->xoffset = bs_long( h11P->xoffset );
		h11P->byte_order = bs_long( h11P->byte_order );
		h11P->bitmap_unit = bs_long( h11P->bitmap_unit );
		h11P->bitmap_bit_order = bs_long( h11P->bitmap_bit_order );
		h11P->bitmap_pad = bs_long( h11P->bitmap_pad );
		h11P->bits_per_pixel = bs_long( h11P->bits_per_pixel );
		h11P->bytes_per_line = bs_long( h11P->bytes_per_line );
		h11P->visual_class = bs_long( h11P->visual_class );
		h11P->red_mask = bs_long( h11P->red_mask );
		h11P->green_mask = bs_long( h11P->green_mask );
		h11P->blue_mask = bs_long( h11P->blue_mask );
		h11P->bits_per_rgb = bs_long( h11P->bits_per_rgb );
		h11P->colormap_entries = bs_long( h11P->colormap_entries );
		h11P->ncolors = bs_long( h11P->ncolors );
		h11P->window_width = bs_long( h11P->window_width );
		h11P->window_height = bs_long( h11P->window_height );
		h11P->window_x = bs_long( h11P->window_x );
		h11P->window_y = bs_long( h11P->window_y );
		h11P->window_bdrwidth = bs_long( h11P->window_bdrwidth );
		}
	    if ( fread( junk, h11P->header_size - sizeof(*h11P), 1, file ) != 1 )
		{
		fprintf( stderr, "Couldn't read rest of X11 XWD file header.\n" );
		exit( 1 );
		}
	    if ( fread( junk, sizeof(X11XColor), h11P->ncolors, file ) !=
		 h11P->ncolors )
		{
		fprintf( stderr, "Couldn't read X11 XWD colormap.\n" );
		exit( 1 );
		}

	    /* Check whether we can handle this dump. */
	    if ( h11P->pixmap_depth != 1 )
		{
		fprintf( stderr, "Can't handle X11 pixmap_depth != 1.\n" );
		exit( 1 );
		}
	    if ( h11P->pixmap_format != XYBitmap &&
		 h11P->pixmap_format != ZPixmap )
		{
		fprintf( stderr, "Can't handle X11 pixmap_format %d.\n",
			 h11P->pixmap_format );
		exit( 1 );
		}
	    if ( h11P->bitmap_unit != h11P->bitmap_pad )
		{
		fprintf(
		    stderr,
		    "X11 bitmap_unit (%d) != bitmap_pad (%d) - can't handle.\n",
		    h11P->bitmap_unit, h11P->bitmap_pad );
		exit( 1 );
		}
	    if ( h11P->bitmap_unit != 8 && h11P->bitmap_unit != 16 &&
		 h11P->bitmap_unit != 32 )
		{
		fprintf(
		    stderr,
		    "X11 bitmap_unit (%d) is non-standard - can't handle.\n",
		    h11P->bitmap_unit );
		exit( 1 );
		}

	    *colP = h11P->pixmap_width;
	    *rowP = h11P->pixmap_height;
	    *padrightP = h11P->bytes_per_line * 8 - h11P->pixmap_width;
	    bits_per_item = h11P->bitmap_unit;
	    bits_used = bits_per_item;
	    bit_order = h11P->bitmap_bit_order;
	    bit_invert = 0;
	    }
	else
	    {
	    fprintf( stderr, "Unknown XWD file version: %d.\n",
		     h11P->file_version );
	    exit( 1 );
	    }
	}

    byteP = (char *) buf;
    shortP = (short *) buf;
    longP = (long *) buf;
    }

bit
getbit( file )
FILE *file;
    {
    bit b;

    if ( bits_used == bits_per_item )
	{
	if ( fread( buf, bits_per_item / 8, 1, file ) != 1 )
	    {
	    fprintf( stderr, "Couldn't read bits.\n" );
	    exit( 1 );
	    }
	switch ( bits_per_item )
	    {
	    case 8:
	    break;

	    case 16:
	    if ( byte_swap )
		*shortP = bs_short( *shortP );
	    break;

	    case 32:
	    if ( byte_swap )
		*longP = bs_long( *longP );
	    break;
	    }
	bits_used = 0;

	if ( bit_order == MSBFirst )
	    bit_shift = bits_per_item - 1;
	else
	    bit_shift = 0;
	}

    switch ( bits_per_item )
	{
	case 8:
	b = ( *byteP >> bit_shift) & 1;
	break;

	case 16:
	b = ( *shortP >> bit_shift) & 1;
	break;

	case 32:
	b = ( *longP >> bit_shift) & 1;
	break;
	}

    if ( bit_invert )
	b = 1 - b;

    if ( bit_order == MSBFirst )
	bit_shift--;
    else
	bit_shift++;
    bits_used++;

    return b;
    }

short
bs_short( s )
short s;
    {
    short ss;
    unsigned char *bp, t;

    ss = s;
    bp = (unsigned char *) &ss;
    t = bp[0];
    bp[0] = bp[1];
    bp[1] = t;
    return ss;
    }

int
bs_int( i )
int i;
    {
    int ii;
    unsigned char *bp, t;

    ii = i;
    bp = (unsigned char *) &ii;
    t = bp[0];
    bp[0] = bp[3];
    bp[3] = t;
    t = bp[1];
    bp[1] = bp[2];
    bp[2] = t;
    return ii;
    }

long bs_long( l )
long l;
    {
    return bs_int( l );
    }
