/***********************************************************************
*  File:   xim.c
*  Author: Philip Thompson
*  $Date: 88/08/29 11:03:16 $
*  $Revision: 1.4 $
*  Purpose: To view a variety of images on X displays and make them
*       suitable for printing as bitmaps.  "Xim" displays images up to
*       24-bits deep with an 'ImageHeader' on 8-bit color and bitmap
*       displays.  Various dithering and halftoning techniques are used
*       to achieve this image portability.
*
*  Copyright (c) 1988  Philip R. Thompson
*                Computer Resource Laboratory (CRL)
*                Dept. of Architecture and Planning
*                M.I.T., Rm 9-526
*                Cambridge, MA  02139
*   This  software and its documentation may be used, copied, modified,
*   and distributed for any purpose without fee, provided:
*       --  The above copyright notice appears in all copies.
*       --  This disclaimer appears in all source code copies.
*       --  The names of M.I.T. and the CRL are not used in advertising
*           or publicity pertaining to distribution of the software
*           without prior specific written permission from me or CRL.
*   I provide this software freely as a public service.  It is NOT a
*   commercial product, and therefore is not subject to an an implied
*   warranty of merchantability or fitness for a particular purpose.  I
*   provide it as is, without warranty. This software was not sponsored,
*   developed or connected with any grants, funds, salaries, etc.
*
*   This software is furnished  only on the basis that any party who
*   receives it indemnifies and holds harmless the parties who furnish
*   it against any claims, demands, or liabilities connected with using
*   it, furnishing it to others, or providing it to a third party.
*
*   Philip R. Thompson (phils@athena.mit.edu)
***********************************************************************/
#ifndef LINT
static char xim_rcs_id[] =
    "$Header: xim.c,v 1.4 88/08/29 11:03:16 jim Exp $";
#endif LINT

#ifdef oldX
#include <strings.h>    /* MIT's Athena is a bit behind the times */
#else
#include <X11/Xos.h>
#endif
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>
#include <stdio.h>
#include "ImageHeader.h"

typedef unsigned char  byte;

int dm4[4][4] = {
     0,  8,  2, 10,
    12,  4, 14,  6,
     3, 11,  1,  9,
    15,  7, 13,  5
};
int dm8[8][8] = {
     0, 32,  8, 40,  2, 34, 10, 42,
    48, 16, 56, 24, 50, 18, 58, 26,
    12, 44,  4, 36, 14, 46,  6, 38,
    60, 28, 52, 20, 62, 30, 54, 22,
     3, 35, 11, 43,  1, 33,  9, 41,
    51, 19, 59, 27, 49, 17, 57, 25,
    15, 47,  7, 39, 13, 45,  5, 37,
    63, 31, 55, 23, 61, 29, 53, 21
};
int dm16[16][16];

char *progName;

int  dither_bw(), fs_bw(), mfs_bw();
int  (*bw)() = fs_bw;
int  ditherPeriod = 8;
int  threshold = 32767;
int  *dm = &(dm8[0][0]);
int  enhance = 0;
Bool  debug_flag = False;
Display  *dpy;
int  screen;
Window  root_win;
Visual  *visual = NULL;
u_long  blackpixel, whitepixel;

main(argc, argv)
int argc;
char **argv;
{
    register unsigned  i, j, k;
    register byte  *buffer, *red_buf, *grn_buf, *blu_buf, *icon_buf;
    unsigned  buf_size;
    int  icon_width, icon_height, iconfact;
    int  buf_width, buf_height, ncolors, nchannels; 
    int  modN[256], divN[256], levels;
    short xmouse, ymouse;
    char  *win_name = NULL, *display_name = NULL;
    char  *str_index, *calloc(), *malloc();
    Bool  inverse_flag = False, mono_flag = False;
    Bool  runlen_flag=False, newmap_flag = False, compress_flag = False;
    FILE  *in_file = stdin, *fopen();
    ImageHeader  p_head;

    XColor  colors[256], fore_color, back_color;
    Window  image_win, icon_win;
    Colormap  colormap, GetColormap();
    XEvent  event;
    XExposeEvent  *expose;
    XCrossingEvent  *xcrossing;
    GC  image_gc, icon_gc;
    XGCValues  gc_val;
    XSetWindowAttributes  xswa;
    XImage  *image = NULL, *icon_image = NULL;
    XSizeHints  sizehints;
    XWMHints  wmhints;

    progName = argv[0];
    for (i=1; i < argc; i++) {
        if (strncmp(argv[i], "-dis", 4) == 0) {
            display_name = argv[++i];
            continue;
        }
        if (strcmp(argv[i], "-in") == 0) {   /* compressed file ? */
            char *ptr = rindex(argv[++i], '.');
            if (ptr && strcmp(ptr, ".Z") == 0) {
                char popen_cmd[16];
                (void)strcpy(popen_cmd, "zcat ");
                (void)strcat(popen_cmd, argv[i]);
                in_file = popen(popen_cmd, "r");
                compress_flag = True;
            } else
                if ((in_file=fopen(argv[i], "r")) == NULL)
                    error("Can't open input file: %s ", argv[i]);
            win_name = argv[i];
            continue;
        }
        if (strncmp(argv[i], "-mf", 3) == 0) {
            bw = mfs_bw;
            mono_flag = True;
            if ((str_index=rindex(argv[i],'=')) != NULL)
                (void)sscanf(str_index+1,"%d", &threshold);
            continue;
        }
        if (strncmp(argv[i], "-fs", 3) == 0) {      /* default */
            bw = fs_bw;
            if ((str_index=rindex(argv[i],'=')) != NULL)
                (void)sscanf(str_index+1,"%d", &threshold);
            mono_flag = True;
            continue;
        }
        if (strncmp(argv[i], "-dit", 4) == 0) {
            char *ptr = rindex(argv[i], '=');
            if (ptr && strcmp(ptr, "=4") == 0) {
                ditherPeriod = 4;
                dm = &(dm4[0][0]);
            }
            bw = dither_bw;
            mono_flag = True;
            continue;
        }
        if (strncmp(argv[i], "-en", 3) == 0) {
            if ((str_index=rindex(argv[i],'=')) != NULL)
                (void)sscanf(str_index+1,"%d", &enhance);
            else
                enhance = 9;
            mono_flag = True;
            continue;
        }
        if (strncmp(argv[i], "-ru", 3) == 0) {
            runlen_flag = True;
            continue;
        }
        if (strncmp(argv[i], "-re", 3) == 0) {
            inverse_flag = True;
            continue;
        }
        if (strncmp(argv[i], "-mo", 3) == 0) {
            mono_flag = True;
            continue;
        }
        if (strncmp(argv[i], "-cm", 3) == 0) {
            newmap_flag = True;
            continue;
        }
        if (strncmp(argv[i], "-de", 3) == 0) {
            debug_flag = True;
            continue;
        }
        error("Usage: %s [-in <file>][-display <host>][-reverse][-cmap]\
    \n[-mono][-dither[=4] | -mfs][-enhance[=1-9]][-runlen][-debug]\n",
          progName);
    }

    /*  Open the display & set defualts */
    if ((dpy = XOpenDisplay(display_name)) == NULL)
        error("Can't open display '%s'", XDisplayName(display_name));
    screen = XDefaultScreen(dpy);
    root_win = XDefaultRootWindow(dpy, screen);
    visual = XDefaultVisual(dpy, screen);
    blackpixel = XBlackPixel(dpy, screen);
    whitepixel = XWhitePixel(dpy, screen);
    if (XDisplayPlanes(dpy, screen) == 1)
        mono_flag = True;

    /* Read header and verify image file formats */
    if (fread((char *)&p_head, sizeof(ImageHeader), 1, in_file) != 1)
        error("Unable to read file header", "\0");
    if (atoi(p_head.header_size) != sizeof(ImageHeader))
        error("Header size mismatch", "\0");
    if ((atoi(p_head.file_version)) != IMAGE_VERSION)
        error("Incorrect Image_file Version.","\0");
    buf_width = atoi(p_head.pixmap_width);
    buf_height = atoi(p_head.pixmap_height);
    if ((atoi(p_head.runlength)) == 1) {
        runlen_flag = True;
        if (compress_flag == True)
            error("Can't be runlength encoded AND compressed.","\0");
    }
    (void)fprintf(stderr,"Author: %s\nDate: %s\n", p_head.creator,
            p_head.date);
    (void)fprintf(stderr,"Size: %d x %d",buf_width, buf_height);
    (void)sscanf(p_head.num_colors, "%d", &ncolors);
    (void)sscanf(p_head.num_channels, "%d", &nchannels);
    buf_size = buf_width * buf_height;

    /* Get or make the color table.
    */
    if (nchannels >= 3) {
        (void)fprintf(stderr,"  nchannels %d\n", nchannels);
        if (mono_flag)          /* make map of grey values 0-65535 */
            for (i=0; i < ncolors; i++) {
                colors[i].pixel = (u_long)i;
                colors[i].red = colors[i].green =
                colors[i].blue = (u_short)(i * 257);
            }
        else {                /* Make color dither map */
            levels = 6;       /* assume for 8-bit display */
            ncolors = levels * levels * levels;
            make_dithermap(levels, colors, divN, modN, dm16);
        }
    } else {                    /* use provided colormap */
        (void)fprintf(stderr,"  ncolors %d\n", ncolors);
        for (i=0; i < ncolors; i++) {
            colors[i].pixel = (u_long)i;
            colors[i].red = (u_short)(p_head.c_map[i][0] * 257);
            colors[i].green = (u_short)(p_head.c_map[i][1] * 257);
            colors[i].blue = (u_short)(p_head.c_map[i][2] * 257);
            colors[i].flags = DoRed | DoGreen | DoBlue;
        }
    }

    /* malloc() and read the data buffer(s)
    */
    if ((buffer=(byte *)malloc((unsigned)buf_size)) == NULL)
        error("Can't malloc() image buffer.","\0");
    if (nchannels >= 3) {
        red_buf = (byte *)malloc(buf_size);
        grn_buf = (byte *)malloc(buf_size);
        blu_buf = (byte *)malloc(buf_size);
        if (red_buf == NULL || grn_buf == NULL || blu_buf == NULL)
            error("Can't malloc() channel buffers.","\0");
        Read_image_buf(in_file, red_buf, &buf_size, runlen_flag);
        Read_image_buf(in_file, grn_buf, &buf_size, runlen_flag);
        Read_image_buf(in_file, blu_buf, &buf_size, runlen_flag);
    } else
        Read_image_buf(in_file, buffer, &buf_size, runlen_flag);

    /*  Allocate the icon with max. dimension of 50 */
    iconfact = (buf_height/50) > (buf_width/50) ? (buf_height/50) 
        :(buf_width/50);
    if (iconfact == 0)
        iconfact = 1;
    if ((icon_width = buf_width / iconfact +1) % 2)
        icon_width -= 1;
    icon_height = buf_height / iconfact;
    if (debug_flag)
        fprintf(stderr,"icon width %d  height %d  factor %d\n",
            icon_width, icon_height, iconfact);
    icon_buf = (byte *)malloc((unsigned)icon_width*icon_height);
    if (icon_buf == NULL)
        error("Can't malloc() icon buffer", "\0");

    /* process and store the image and icon.
    */
    if (mono_flag) {               /* get a bitmap image */
        if (nchannels >= 3) {      /* convert to 8-bit grey values */
            for (i=0; i < buf_size; i++) 
                buffer[i] = (byte)GreyValue((u_short)red_buf[i],
                    (u_short)grn_buf[i], (u_short)blu_buf[i]);
        } else {
            MakeMapGrey(colors, &ncolors, buffer, buf_size);
        }
        if (bw == dither_bw)
            NormalizeDM();
        for (i=0, k=0; i < icon_height; i++) /* sample image for icon */
            for (j=0; j < icon_width; j++, k++)
                icon_buf[k]= buffer[(i*buf_width+j)*iconfact];
        if (enhance) {
            edge_enhance(buffer, buf_width, buf_height, buf_size);
            edge_enhance(icon_buf, icon_width, icon_height,
                (unsigned)icon_width*icon_height);
        }
        /* Translate pixmap into a bitmap for monochrome display. */
        Monochrome(buffer, buf_width, buf_height, colors);
        Monochrome(icon_buf, icon_width, icon_height, colors);
        colormap = XDefaultColormap(dpy,screen);
        icon_image = XCreateImage(dpy, visual, 1, XYBitmap, 0,
                (char *)icon_buf, icon_width, icon_height, 8, 0);
    } else {                      /* use color image */
        if (nchannels >= 3)
            rgb_dither(red_buf, grn_buf, blu_buf, buffer, buf_width,
                    buf_height, levels, divN, modN);
        if (ncolors > 250)          /* Don't bother trying to fit */
            newmap_flag = True;     /* into default map, faster too */
        colormap = GetColormap(colors, ncolors, &newmap_flag, buffer,
            buf_size);
        icon_image = XCreateImage(dpy, visual, 8, ZPixmap, 0,
                (char *)icon_buf, icon_width, icon_height, 8, 0);
        for (i=0; i < icon_height; i++)
            for (j=0; j < icon_width; j++)
                XPutPixel(icon_image, j, i,
                    buffer[(i*buf_width+j)*iconfact]);
    }
    image = XCreateImage(dpy, visual, (mono_flag ? 1 : 8),
            (mono_flag ? XYBitmap : ZPixmap), 0, (char *)buffer,
            buf_width, buf_height, 8, 0);
    if (debug_flag)
        fprintf(stderr,"processed.\n");

    /* Get window attributes */
    xswa.event_mask = ExposureMask |ButtonPressMask |ColormapChangeMask|
        LeaveWindowMask | EnterWindowMask;
    xswa.background_pixel = blackpixel;
    xswa.border_pixel = whitepixel;
    xswa.colormap = colormap;
    xswa.cursor = XCreateFontCursor(dpy, XC_gumby);
    image_win = XCreateWindow(dpy, root_win, 0, 0,
        buf_width, buf_height, 5, XDefaultDepth(dpy,screen),
        InputOutput, visual, CWBackPixel |CWEventMask |CWCursor |
        CWBorderPixel |CWColormap, &xswa);
    xswa.event_mask = ExposureMask;
    icon_win = XCreateWindow(dpy, root_win, 0, 0,
        icon_width, icon_height, 1, XDefaultDepth(dpy,screen),
        InputOutput, visual, CWBackPixel | CWBorderPixel, &xswa);

    /* set window manager hints */
    sizehints.flags = PPosition | PSize | PMinSize | PMaxSize;
    sizehints.width = sizehints.min_width = buf_width;
    sizehints.max_width = buf_width;
    sizehints.height = sizehints.min_height = buf_height;
    sizehints.max_height = buf_height;
    sizehints.x = 0;
    sizehints.y = 0;
    XSetStandardProperties(dpy, image_win, "X Imager", win_name,
            None, argv, argc, &sizehints);
    wmhints.flags = IconWindowHint | IconPositionHint;
    wmhints.icon_window = icon_win;
    wmhints.icon_x = XDisplayWidth(dpy,screen) - 200;
    wmhints.icon_y = 2;
    XSetWMHints(dpy, image_win, &wmhints);

    gc_val.function = GXcopy;
    gc_val.plane_mask = AllPlanes;
    if (inverse_flag) {
        gc_val.foreground = whitepixel;
        gc_val.background = blackpixel;
    } else {
        gc_val.foreground = blackpixel;
        gc_val.background = whitepixel;
    }
    image_gc = XCreateGC(dpy,image_win, GCFunction | GCPlaneMask |
        GCForeground | GCBackground, &gc_val);
    icon_gc = XCreateGC(dpy, icon_win, GCFunction | GCPlaneMask |
        GCForeground | GCBackground, &gc_val);

    XMapWindow(dpy, image_win);             /* Map the image window. */
    if ((newmap_flag) && (!mono_flag)) {
        XInstallColormap(dpy, colormap);
        if (ncolors > 254) {
            fore_color.red = colors[255].red;       /* force the last */
            fore_color.green = colors[255].green;   /* two colors and */
            fore_color.blue = colors[255].blue;   /* sacrifice cursor */
            back_color.red = colors[254].red;
            back_color.green = colors[254].green;
            back_color.blue = colors[254].blue;
            XRecolorCursor(dpy, xswa.cursor, &fore_color, &back_color);
        }
    }

    /* Select events to listen for  */
    XSelectInput(dpy, image_win, (ButtonPressMask | ColormapChangeMask |
        ExposureMask | LeaveWindowMask | EnterWindowMask));
    XSelectInput(dpy, icon_win, ExposureMask);

    if (debug_flag)
       fprintf(stderr,"While loop.\n");
    expose = (XExposeEvent *)&event;
    xcrossing = (XCrossingEvent *)&event;
    while (True) {          /* Set up a loop to maintain the image. */
        XNextEvent(dpy, &event);           /* Wait on input event. */
        switch((int)event.type) {
        int modulo;     /* Temporary var. for expose->x % 4 */
        case Expose:
            if (expose->window == icon_win) {
                XPutImage(dpy, icon_win, icon_gc, icon_image, 0, 0,
                    0, 0, icon_width, icon_height);
                break;
            }
            if (debug_flag)
                fprintf(stderr,
                "expose event x= %d y= %d width= %d height= %d\n",
                expose->x, expose->y, expose->width, expose->height);
            modulo = expose->x % 4;
            if ((modulo != 0) && (!mono_flag)) {
                expose->x -= modulo;
                expose->width += modulo;
            }
            if ((expose->width % 4 != 0) && (!mono_flag))
                expose->width += 4 - (expose->width % 4);
            XPutImage(dpy, image_win, image_gc, image,
                expose->x, expose->y, expose->x, expose->y,
                expose->width, expose->height);
            if (debug_flag)
                fprintf(stderr, "Actual expose: %d  %d  %d  %d\n",
                expose->x, expose->y, expose->width, expose->height);
            break;
        case ButtonPress:
            switch((int)event.xbutton.button) {
            case Button1:
                Read_image_buf(in_file,buffer,&buf_size,runlen_flag);
                if (mono_flag)
                   Monochrome(buffer,buf_width,buf_height,colors);
                XPutImage(dpy, image_win, image_gc, image, 0, 0, 0, 0,
                    image->width, image->height);
                break;
            case Button2:
                xmouse = event.xbutton.x;
                ymouse = event.xbutton.y;
                fprintf(stderr,"(x,y): %d %d\n", xmouse, ymouse);
                break;
            case Button3:
                if (compress_flag)
                    (void)pclose(in_file);
                else
                    (void)fclose(in_file);
                if (newmap_flag)
                    XInstallColormap(dpy, XDefaultColormap(dpy,screen));
                XDestroyWindow(dpy, image_win);
                XDestroyWindow(dpy, icon_win);
                XCloseDisplay(dpy);
                exit(0);
            }
        case LeaveNotify:
            if (newmap_flag && (xcrossing->mode != NotifyGrab))
                XInstallColormap(dpy, XDefaultColormap(dpy,screen));
            break;
        case EnterNotify:
            if (newmap_flag && (xcrossing->mode != NotifyUngrab))
                XInstallColormap(dpy, colormap);
            break;
        case ColormapNotify:
                /* Don't do anything for now */
            break;
        default:
             fprintf(stderr,"Bad X event.\n");
        }
    }
}  /* end main */


Colormap GetColormap(colors, ncolors, newmap_flag, buf, bufsize)
XColor  colors[];
int  ncolors;
Bool  *newmap_flag;
register byte  *buf;
unsigned  bufsize;
{
    register i;
    Colormap cmap, cmap2;
    XColor qcolor;
    u_long GetColorValue();
    
    if (ncolors > XDisplayCells(dpy,screen))    /* an X nonfeature */
        ncolors = XDisplayCells(dpy,screen);
    if (debug_flag)
        fprintf(stderr,"Colormap size %d\n", ncolors);

    if (*newmap_flag) {
        cmap = XCreateColormap(dpy, root_win, visual, AllocAll);
        XStoreColors(dpy, cmap, colors, ncolors);
    } else {
        cmap = XDefaultColormap(dpy, screen);
        for (i=0; i < ncolors; i++) {
            if (XAllocColor(dpy, cmap, &colors[i]) == 0) {
                fprintf(stderr,"Too many colors %d - new map made\n",i);
                cmap2 = XCopyColormapAndFree(dpy, cmap);
                *newmap_flag = True;
                for ( ; i < ncolors; i++)
                    XAllocColor(dpy, cmap2, &colors[i]);
                cmap = cmap2;
                break;
            }
        }
        for (i=0; i < bufsize; i++)
            buf[i] = (byte)colors[buf[i]].pixel;
    }
    if (*newmap_flag) {
        whitepixel = GetColorValue(cmap, ncolors, 255, 255, 255);
        blackpixel = GetColorValue(cmap, ncolors, 0, 0, 0);
    }
    if (debug_flag)
      for (i=0; i < ncolors; i++) {
         qcolor.pixel = (u_long)i;
         XQueryColor(dpy, cmap, &qcolor);
         fprintf(stderr,"color[%3d]: pix %3u r= %5u g= %5u b= %5u\n",i,
            qcolor.pixel, qcolor.red, qcolor.green, qcolor.blue);
      }
    return(cmap);
}


/* Find the the closest color in the colormap.
*/
u_long GetColorValue(cmap, ncolors, red, green, blue)
Colormap cmap;
int ncolors, red, green, blue;
{
    register i, red2, blue2, green2;
    XColor qcolor;
    u_long value;
    long dist, least = 1e5;
    
    for (i=0; i < ncolors; i++) {
        qcolor.pixel = (u_long)i;
        XQueryColor(dpy, cmap, &qcolor);
        red2 = (int)qcolor.red / 257;   
        green2 = (int)qcolor.green / 257;   
        blue2 = (int)qcolor.blue / 257; 
        dist = ((red2 - red) * (red2 - red)) +
               ((green2 - green) * (green2 - green)) +
               ((blue2 - blue) * (blue2 - blue));
        if (dist == 0)
            return(qcolor.pixel);
        else if (dist < least) {
            least = dist;
            value = qcolor.pixel;
        }
    }
    return(value);
}

/* This edge enhancing is taken from the ACM Transaction on Graphics
*  Vol. 6, No. 4, October 1987.  Dot diffusion is not implemented.
*/
edge_enhance(buf, width, height, bufsize)
register byte *buf;
int width, height;
unsigned bufsize;
{
    register i, x, y, lbyt, hbyt, tmp;
    register byte *tbuf;
    register float phi;
    char *malloc();

    if (debug_flag)
        fprintf(stderr,"enhancing... ");
    tbuf = (byte *)malloc(bufsize);
    bcopy((char *)buf, (char *)tbuf, (int)bufsize);

    lbyt = width-1;
    hbyt = width+1;
    i = lbyt;
    if (enhance == 9) {             /* is much faster, default */
        for (y=2; y < height; y++) {
            i += 2;
            for (x=2; x < width; x++,i++) {
                tmp = (9 * tbuf[i]) - (tbuf[i-hbyt] + tbuf[i-width]+
                 tbuf[i-lbyt] + tbuf[i-1] + tbuf[i+1] + tbuf[i+lbyt] +
                 tbuf[i+width] + tbuf[i+hbyt]);
                if (tmp > 255)
                    buf[i] = 255;
                else if (tmp < 0)
                    buf[i] = 0;
                else
                    buf[i] = (byte)tmp;
            }
        }
    } else {                        /* allows greater control */
        phi = enhance / 10.0;
        for (y=2; y < height; y++) {
            i += 2;
            for (x=2; x < width; x++,i++) {
                tmp = (tbuf[i-hbyt] + tbuf[i-width] + tbuf[i-lbyt] +
                    tbuf[i-1] + tbuf[i] + tbuf[i+1] +
                    tbuf[i+lbyt] + tbuf[i+width] + tbuf[i+hbyt]) / 9.0;
                tmp = (tbuf[i]-(phi*tmp)) / (1.0-phi) + 0.5;
                if (tmp > 255)
                    buf[i] = 255;
                else if (tmp < 0)
                    buf[i] = 0;
                else
                    buf[i] = (byte)tmp;
            }
        }
    }
    free((char *)tbuf);
}

int *error1, *error2;

/* Color to monochrome conversion. Performed in place - overwrites 
* the source pixmap and transforms it to a bitmap in the process.
*/
Monochrome(buf,width,height,map)
register byte  *buf;
int  width, height;
XColor  map[];
{
    register byte  *mbuffer, mvalue;   /* monochrome buffer */
    register byte  *mpbuffer;          /* monochrome pixel buffer */
    register  row, col, bit;
    char  *malloc(), *calloc();

    if (debug_flag)
        fprintf(stderr,"Monochrome... ");
    error1 = (int *)malloc((unsigned)(width+1) * sizeof(int));
    error2 = (int *)malloc((unsigned)(width+1) * sizeof(int));
    mbuffer= (byte *)calloc((unsigned)width*height/8, sizeof(byte));
    if ((error1 == NULL) || (error2 == NULL) || (mbuffer == NULL))
        error("calloc() in monochrome conversion", "\0");
    mpbuffer = buf;
    mbuffer = buf;

    /* do a monochrome translation on the pixmap.
    *  to a bitmap.
    */
    if (XBitmapBitOrder(dpy) == LSBFirst) {
        for (row=0; row < height; row++)
            for (col=0; col < width; ) {
                mvalue = 0x00;
                for (bit=0; (bit < 8) && (col < width); bit++,col++)
                    if ((*bw)(*mpbuffer++, map, col, row))
                        mvalue |= (0x01 << bit);    /*  for Vax */
                *mbuffer++ = mvalue;
            }
    } else {
        for (row=0; row < height; row++)
            for (col=0; col < width; ) {
                mvalue = 0x00;
                for (bit=0; (bit < 8) && (col < width); bit++,col++)
                    if ((*bw)(*mpbuffer++, map, col, row))
                        mvalue |= (0x80 >> bit);    /*  for RT, Sun  */
                *mbuffer++ = mvalue;
            }
    }
    free((char *)buf);
    buf = mbuffer;
    free((char *)error1);
    free((char *)error2);
}


/*************************
* code for dithering     *
*************************/
int dither_bw(pixel,map,count,line)
unsigned int pixel;
XColor map[]; 
register count, line;
{   
    if (map[pixel].red > dm[((line%ditherPeriod)*ditherPeriod) +
            (count%ditherPeriod)])
        return(0);
    else
        return(1);
}


/***************************************
* code for modified floyd steinberg    *
****************************************/
int mfs_bw(pixel,map,count,line)
unsigned int pixel;
XColor map[]; 
register count, line;
{
    int  onoff, *te;
    long  intensity, pixerr;

    if (count == 0) {
        te = error1;
        error1 = error2;
        error2 = te;
        error2[0] = 0;
    }  
    intensity = map[pixel].red + error1[count];
    if (intensity > 65535)
        intensity = 65535;
    else if (intensity < 0)
        intensity = 0;

    if (intensity < threshold) {
        onoff = 1;
        pixerr = threshold - intensity;
    }
    else {
        onoff = 0;
        pixerr = threshold - intensity;
    }
    error1[count+1] += (int)(3*pixerr)/8;
    error2[count+1] = (int)pixerr/4;
    error2[count] += (int)(3*pixerr)/8;
    return(onoff);
}


/*****************************
* code for floyd steinberg   *
*****************************/
int fs_bw(pixel, map, count, line)
unsigned int pixel;
XColor map[]; 
register count, line;
{
    int  onoff, *te; 
    long  intensity, pixerr;

    if (count == 0) {
        te = error1;
        error1 = error2;
        error2 = te;
        error2[0] = 0;
    }  
    intensity = map[pixel].red + error1[count];
    if (intensity > 65535)
        intensity = 65535;
    else
        if (intensity < 0)
            intensity = 0;
    if (intensity < threshold) {
        onoff = 1;
        pixerr = intensity;
    } else {
        onoff = 0;
        pixerr = intensity - 65535;
    }
    error1[count+1] += (int)(3*pixerr)/8;
    error2[count+1] = (int)pixerr/4;
    error2[count] += (int)(3*pixerr)/8;
    return(onoff);
}


/* Map rg&b channels to 8 bits through dithering.  This color dithering
* code has been adapted and greatly simplified from the original work by
*   author: Spencer W. Thomas
*           Computer Science Dept. (cs.utah.edu)
*           University of Utah
* Copyright (c) 1986, University of Utah
*/
rgb_dither(red, grn, blu, buf, width, height, levels, divN, modN)
register byte  *red, *grn, *blu, *buf;
int width, height, levels, divN[], modN[];
{
    register i, x, y, col=0, row=0, levelsq;

    levelsq = levels * levels;
    for (i=0, y=0; y < height; y++, row=y%16) {
        for (x=0; x < width; x++, col=x%16, i++) {
            buf[i] = (byte)(
                (modN[red[i]] > dm16[col][row] ? divN[red[i]]+1 :
                divN[red[i]]) +
                (modN[grn[i]] > dm16[col][row] ? divN[grn[i]]+1 :
                divN[grn[i]]) * levels +
                (modN[blu[i]] > dm16[col][row] ? divN[blu[i]]+1 :
                divN[blu[i]]) * levelsq);
        }
    }
    free((char *)red);
    free((char *)grn);
    free((char *)blu);
}


/* Create a color map and dithering matrix for the specified
*  intensity levels.
*/
make_dithermap(levels, colors, divN, modN, magic)
int levels, divN[256], modN[256], magic[16][16];
XColor colors[];
{
    float N, magicfact;
    register i, j, k, l, levelsq, ncolors;
    
    levelsq = levels*levels;    /* squared */
    ncolors = levels*levelsq;   /* and cubed */
    N = 255.0 / (levels-1);     /* Get size of each step */

    /* Set up the color map entries.  */
    for (i=0; i < ncolors; i++) {
        colors[i].pixel = (u_long)i;
        colors[i].red = (u_short)((int)((i%levels)*N+0.5)*257);
        colors[i].green=(u_short)((int)(((i/levels)%levels)*N+0.5)*257);
        colors[i].blue=(u_short)((int)(((i/levelsq)%levels)*N+0.5)*257);
        colors[i].flags = DoRed | DoGreen | DoBlue;
    }
    for (i=0; i < 256; i++) {
        divN[i] = (int)(i / N);
        modN[i] = i - (int)(N * divN[i]);
    }
    magicfact = (N - 2) / 16.0;
    for (i=0; i < 4; i++)
        for (j=0; j < 4; j++)
            for (k=0; k < 4; k++)
                for (l=0; l < 4; l++)
                    magic[4*k+i][4*l+j] = (int)(0.5 + dm4[i][j] *
                        magicfact + (dm4[k][l]/16.0) * magicfact);
    if (debug_flag) {
        fprintf(stderr,"Ncolors = %d   Levels = %d\n", ncolors, levels);
        for (i=0; i < 16; i++) {
            for (j=0; j < 16; j++)
                fprintf(stderr,"%4d", magic[i][j]);
            fprintf(stderr,"\n");
        }
        fprintf(stderr,"\n");
    }
}


/* Normalize dither matrix to 65535 maximum value.
*/
NormalizeDM()
{
    int i, normalValue, matsize;

    matsize = ditherPeriod * ditherPeriod;
    normalValue = 65536 / matsize;
    for (i=0; i < matsize; i++) {
        dm[i] = dm[i] * normalValue;
        if (debug_flag) {
            fprintf(stderr,"%8d", dm[i]);
            if  (((i+1) % ditherPeriod) == 0)
                fprintf(stderr,"\n");
        }
    }
}


/* Transformation from RGB to the Y (or luminence) factor from
*  YIQ encoding.
*/
int GreyValue(red, green, blue)
u_short red, green, blue;
{
    return((int)((float)red * 0.30 + (float)green * 0.55 +
            (float)blue * 0.15 + 0.5));
}


MakeMapGrey(colors, ncolors, buf, bufsize)
XColor colors[];
int *ncolors;
register byte  *buf;
unsigned  bufsize;
{
    register unsigned  i;
    byte  tval[256];

    /* determine b/w intensity level and place in pixel */
    if (debug_flag)
        fprintf(stderr,"making grey pixmap ...\n");
    for (i=0; i < *ncolors; i++) {
        tval[i] = GreyValue(colors[i].red, colors[i].green,
                colors[i].blue) / 257;
        if (debug_flag)
            fprintf(stderr,
            "color[%3d] pix: %3u  r: %5u  g: %5u  b: %5u  new: %3u\n",
            i, colors[i].pixel, colors[i].red, colors[i].green,
            colors[i].blue, tval[i]);
    }
    for (i=0; i < bufsize; i++)
        buf[i] = tval[buf[i]];
    *ncolors = 256;
    for (i=0; i < *ncolors; i++) {
        colors[i].pixel = (u_long)i;
        colors[i].red = colors[i].green = colors[i].blue =
                (u_short)(i * 257);
        colors[i].flags = DoRed | DoGreen | DoBlue;
    }
}


Read_image_buf(infile, buf, buf_size, encoded)
FILE *infile;
register byte  *buf;
unsigned  *buf_size;
int  encoded;
{
    register int  i, runlen, nbytes;
    register unsigned  j;
    register byte *line;
    long  marker;
    char *malloc();
    
    if (debug_flag)
        fprintf(stderr,"Reading pixmap... ");
    if (!encoded) {
        j = fread((char *)buf, 1, (int)*buf_size, infile);
    } else {
        if ((line=(byte *)malloc((unsigned)BUFSIZ)) == NULL)
            error("Can't malloc() fread string.","\0");
        /* Unrunlength encode data */
        marker = ftell(infile);
        j = 0;
        while (((nbytes=fread((char *)line, 1, BUFSIZ, infile)) > 0) &&
            (j < *buf_size)) {
            for (i=0; (i < nbytes) && (j < *buf_size); i++) {
                runlen = (int)line[i]+1;
                i++;
                while (runlen--)
                    buf[j++] = line[i];
            }
            marker += i;
        }
        /* return to the begining of the next image's bufffer */
        if (fseek(infile, marker, 0) == -1)
            error("Can't fseek to location in image buffer.","\0");
        free((char *)line);
    }
    if (j != *buf_size) {
        fprintf(stderr,"%cUnable to complete pixmap: %u / %u (%d%%)\n",
            7, j, *buf_size, (int)(j*100.0 / *buf_size));
        *buf_size = j;
    }
}


/*  A most tragic and fatal error.
*/
error(s1, s2)
char *s1, *s2;   /* Error description string. */
{
    extern int errno, sys_nerr;
    extern char *sys_errlist[];

    fprintf(stderr,"%c%s: Error =>\n%c", 7, progName, 7);
    fprintf(stderr, s1, s2);
    if ((errno > 0) && (errno < sys_nerr))
        fprintf(stderr, " (%s)", sys_errlist[errno]);
    fprintf(stderr, "\n");
    exit(1);
}


/* End of xim.c */
