/* Copyright 1985, Massachusetts Institute of Technology */

/* This is the main file for xperfmon.  This file should contain the necessary
   routines for linking the system section of this program to the graphing 
   section.  No actual work should be done here.  See the other source
   files for more information. */

/***************************************************************************
 *
 *  Modification History
 *
 *  This version was completed by Emanuel Jay Berkenbilt, MIT Project
 *  Athena on 1/21/1988.  There is no record of previous authors.
 *  Xperfmon contains the following files:
 *
 *    xperfmon.n
 *    icon
 *    all.h
 *    system.h
 *    window.h
 *    geometry.c
 *    system.c
 *    window.c
 *    xperfmon.c
 *
 *  2/11/1988: -n | -not comandline option added.  xperfmon.c and system.c
 *             were modified; xperfmon.n (the man file) was updated.  EJB
 *
 ***************************************************************************/

/* X Unix performance monitor.
   Simple graphical performance monitor for system-wide data. */

/****************** #defines *******************/
#define MILLION 1000000

/* These are passed to adjust_timeout() and represent numbers of
   microseconds. */
#define LARGE_ADD_TIME MILLION
#define LARGE_SUBTRACT_TIME -LARGE_ADD_TIME
#define SMALL_ADD_TIME LARGE_ADD_TIME/10
#define SMALL_SUBTRACT_TIME -SMALL_ADD_TIME
#define MAX_TIME 300*MILLION /* five minutes */

#define VALID_UPDATE(x) ((x <= MAX_TIME) && (x > 0))

/****************** #includes ******************/
#include "all.h"
#include "system.h"
#include "window.h"

#include <sys/types.h>

#include <stdio.h>
#include <X11/Xos.h
#include <sys/time.h>

#ifdef FD_SET
#define XFD_SET(n,p) FD_SET((n),(p))
#define XFD_ZERO(p) FD_ZERO((p))
#else
#include "fd.h"
#endif

/************ variables and functions *********/
/* external declarations */
extern char *malloc();
extern char *calloc();
extern double atof();

/* window declarations */
void graph();
void label_graph();
void reset_graph();
int win_setup();
short x_process_cmdline();
int display_fd;         /* Display file descriptor for select */


/* system declarations */
int sys_setup();
void update_stats();
short sys_process_cmdline();

int num_stats;          /* This must be called num_stats for FORALLSTATS */
int stat;               /* just a counter of stats */
extern stat_type stats; /* must be declared so that labels may be copied */
extern int poss_stats[NUM_POSSIBLE_STATS];

/* xperfmon delcarations */
struct timeval timeout;
/* This structure contains all information not having to do with system
   parameters of X that can be specified on the commandline. */
struct
{
  long update_interval;
} cmdline_params;


void adjust_timeout(delta)
int delta;
/* This routine changes the interval of time after which the graph should
   be updated. */
{
  long int time;
  
  /* time is the number of microseconds in the timeout interval */
  time = MILLION*timeout.tv_sec + timeout.tv_usec;

  while (!VALID_UPDATE(time+delta))
    delta/=2;
  time+=delta;
  timeout.tv_sec = time/MILLION;
  timeout.tv_usec = time%MILLION;
}

void initialize_cmdline()
/* This routine sees that all structures containing information that
   can be specified on the command line get initialized to the default
   values for each parameter. */
{
  set_x_defaults();
  set_sys_defaults();
  cmdline_params.update_interval = 1000000;
}

void initialize_timer()
/* This routine resets the timout value to the update interval specified
   on the commandline or to the default if none was specfied. */
{
  timeout.tv_sec = cmdline_params.update_interval/MILLION;
  timeout.tv_usec = cmdline_params.update_interval%MILLION;
}

short handle_key(ch)
char ch;
/* This routine interprets the key that was pressed on top of the window. 
   It returns TRUE if Quit has been selected, signaling that the program is
   done. */
{
  short done = FALSE;

  switch (ch)
    {
    case 'Q':
      done = TRUE;
      break;
    case 'q':
      exit(0);
      break;
    case 'R':
      reset_graph();
      initialize_timer();
      break;
    case 's':
      adjust_timeout(SMALL_ADD_TIME);
      break;
    case 'S':
      adjust_timeout(LARGE_ADD_TIME);
      break;
    case 'f':
      adjust_timeout(SMALL_SUBTRACT_TIME);
      break;
    case 'F':
      adjust_timeout(LARGE_SUBTRACT_TIME);
      break;
    case '?':
      puts("Q/q - Quit");
      puts("R - Reset graph and timer");
      puts("s - Decrease update interval (slower) by a small amount");
      puts("S - Decrease update interval (slower) by a large amount");
      puts("f - Increase update interval (faster) by a small amount");
      puts("F - Increase update interval (faster) by a large amount");
      puts("? - Help");
      break;
    default:
      break;
    }
  return(done);
}

void main_event_loop()
/* This routine checks for X events or timeout.  It uses select().  For
   X events, it uses the file descriptor returned by win_setup() in 
   window.c. */
{
  short done = FALSE;
  int nfound;
  char ch;
  int event_flag;
  fd_set display_fd_set;
  int *values;

  if ((values = (int *)calloc(num_stats,sizeof(int))) == NULL)
    {
      perror("Failure getting memory for values");
      exit(1);
    }

  while(!done)
    {
      /* Make sure that all X events are cleared before calling select().
	 I don't know why, but this needs to be done first. */
      while ((event_flag = check_win_events(&ch)) == WE_MORE_EVENTS)
	if (ch != 0)
	  done = handle_key(ch);

      if (event_flag == WE_FAILURE)
	done = TRUE;

      if (!done)
      {
	XFD_ZERO(&display_fd_set);
	XFD_SET(display_fd,&display_fd_set);
	if ((nfound = select(display_fd+1,&display_fd_set,0,0,&timeout)) == 0)
	  {
	    update_stats();
	    FORALLSTATS(stat)
	      values[stat] = stats[stat].value;
	    graph(values);
	  }
	else if (nfound > 0)
	  {
	    /* An X event occurred - this will be handled the next time through
	       the loop */
	  }
	else 
	  perror("select");
      }
    }
  free(values);
}

void process_cmdline(argc,argv)
int argc;
char **argv;
/* This routine calls the routines in system.c and window.c that process
   the command line arguments that are used by those sections of the program
   and processes all remaining arguments for its own use. */
{
  int i;
  short *args_found;

  initialize_cmdline();

  if ((args_found = (short *)calloc(argc,sizeof(short))) == NULL)
    {
      perror("Failure getting memory for args_found");
      exit(1);
    }

  /* These two routines fail if an improper syntax is given for one of the 
     options they check for. */
  if (!sys_process_cmdline(argc,argv,args_found))
    usage();
  if (!x_process_cmdline(argc,argv,args_found))
    usage();

  /* args_found has been passed to sys_process_cmdline() and 
     x_process_cmdline().  An elements of this array that has the
     value of one means that the corresponding element in argv
     has been picked up by either system.c or window.c and should
     be ignored by the remainder of this routine. */
  /* Note that calling usage() terminates the program with an error */
  for (i = 1; i < argc; i++)
    {
      if (ARGIS(i,"-u","-update"))
	{
	  if (++i >= argc)
	    usage();
	  if (VALID_UPDATE((long)(MILLION*atof(argv[i]))))
	    cmdline_params.update_interval = (long)(MILLION*atof(argv[i]));
	  else
	    fprintf(stderr,"Invalid update interval ignored\n");
	}
      else if (!args_found[i]) /* argv[i] has not been handled by x or sys */
	usage();
    }
  free((char *)args_found);
}

main(argc,argv)
int argc;
char **argv;
/* This is the way a main() should look. */
{
  process_cmdline(argc,argv);

  initialize_timer();

  /* Allow system.c to do whatever setup it needs and findout the number of
     stats that are actually being defined. */
  if ((num_stats = sys_setup()) == 0)
    {
      fprintf(stderr,"Sorry - no stats to display!\n");
      exit(1);
    }

  /* Allow window.c to setup the graph structures and initialize them with
     appropriate information from the stats variable. */
  graph_setup(num_stats,LINES_PER_LABEL);
  FORALLSTATS(stat)
    init_graph_list(stats[stat].label,stats[stat].min_val,stats[stat].max_val);

  /* Allow window.c to setup the window and display.  THis must be done
     after all other setup because win_setup uses information in the 
     graph data structures. */
  display_fd = win_setup(argc,argv);

  /* At last, process events until done. */
  main_event_loop();
}

usage()
{
  fprintf(stderr,"\nUsage: xperfmon option option option .....\n");
  fprintf(stderr,"options:\n");
  fprintf(stderr,"   =geometry               (to set window geometry)\n");
  fprintf(stderr,"   host:display            (to specify display)\n");
  fprintf(stderr,"   -rv | -reverse          (black on white)\n");
  fprintf(stderr,"   -fw | -forward          (white on black)\n");
  fprintf(stderr,"   -bw | -border n         (border width)\n");
  fprintf(stderr,"   -fn | -font font        (font)\n");
/*  fprintf(stderr,"   -bd | -color color      (border color)\n");  */
/*  fprintf(stderr,"   -fg | -foreground color (foreground color)\n");  */
/*  fprintf(stderr,"   -bg | -background color (background color)\n");  */
/*  fprintf(stderr,"   -hl | -highlight color  (hilight color)\n");  */
  fprintf(stderr,"   -u  | -update secs      (update interval)\n");
  fprintf(stderr,"   -st | -stepsize n       (stepsize in pixels)\n");
  fprintf(stderr,"   -n  | -not stat ...     (exclude a list of stats)\n");
  fprintf(stderr,"   stat stat stat ...      (list of stats to be shown)\n");
  fprintf(stderr,"Command line options are not mutually exclusive, but options that appear\n");
  fprintf(stderr,"later on the command line take precedence.  Specifying -n supercedes any\n");
  fprintf(stderr,"other stat specification you may have done and causes any subsequent stats\n");
  fprintf(stderr,"specified on the command line to be omitted instead of shown.\n\n");
  exit(1);
}
