/*
 * xshell - program for popping up windows
 *
 * $Source: /usr/expo/X/src/contrib/xshell/RCS/xshell.c,v $
 * $Header: xshell.c,v 1.3 88/02/12 15:15:06 jim Exp $
 *
 * Copyright 1988 Massachusetts Institute of Technology
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of M.I.T. not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  M.I.T. makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * Author          : Mark D. Baushke <silvlis!mdb@sun.com>
 * Created On      : Tue Feb  2 16:38:26 1988
 * Last Modified By: Mark D. Baushke
 * Last Modified On: Fri Feb  5 17:57:30 1988
 * Update Count    : 53
 * Developed Under : X.V11R1, SunOS 3.4
 * Status          : Beta test. Use at your own risk.
 *
 */

/*
 * xshell - raw Xlib version of xshell for X.V11R1.
 *
 * This program is based upon an X.V10R4 version written by Jim Fulton.  It
 * also makes use of code from his xlib-xbiff.
 *
 * Compile with:
 *
 *      cc -o xshell -O xshell.c -lX11
 */

#ifndef lint
static char *rcsid_xshell = "$Header: xshell.c,v 1.3 88/02/12 15:15:06 jim Exp $";
#endif

#include <stdio.h>                      /* general printing and NULL */
#include <X11/Xlib.h>                   /* X11 definintions */
#include <X11/Xutil.h>                  /* for XParseGeometry() */
#include <X11/cursorfont.h>             /* for cursor constants */
#include <X11/keysym.h>

#include <signal.h>
#include <ctype.h>
#include <strings.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/types.h>

#include "scallopshell.h"
#include "largescallopshell.h"

/*
 * Your mileage may vary... to 'int' or 'short'.
 */

typedef long    Pixel;

/*
 * Macros
 */

#define streq(a,b)  (strcmp ((a), (b)) == 0)
#define NO          0
#define YES         1
#define micro2centi (10000)             /* microsecond to 1/100 second */

#define min(a,b) ((a) < (b) ? (a) : (b))    /* not used but available */
#define max(a,b) ((a) > (b) ? (a) : (b))

/*
 * The X.V10R4 version of xshell allowed action.SHIFT action.META and
 * action.CONTROL, I would rather see this done by looking at the
 * modifiers of the event structure allowing action.META.<char> .
 * Currently this is not being done. If you *really* want to have the
 * action.SHIFT stuff, uncomment the next line.
 */

/* #define LOOKUP_MODIFIER_KEYS */

/*
 * For now, only have a table of 256. Eventually it would be nice to
 * add action.meta.<key> which would double the size of the table and
 * require a different indexing scheme that is why everything is done
 * in terms of the following macros. They could be re-written as
 * functions if a hash table of some kind were to be used.
 */

#define ACTION_TABLE_SIZE   256

#define ACTION_NEW          ((char) 0)
#define ACTION_FOUND        ((char) 1)
#define ACTION_NOT_FOUND    ((char) 2)

#define action_is_found(Key) \
    (action_flags[(short) (0xff & Key)] == ACTION_FOUND)

#define action_is_new(Key) \
    (action_flags[(short) (0xff & Key)] == ACTION_NEW)

#define set_action_is_new(Key) \
    action_flags[(short) (0xff & Key)] = ACTION_NEW;

#define set_action_is_found(Key) \
    action_flags[(short) (0xff & Key)] = ACTION_FOUND;

#define set_action_notfound(Key) \
    action_flags[(short) (0xff & Key)] = ACTION_NOT_FOUND;

#define set_action_notfound(Key) \
    action_flags[(short) (0xff & Key)] = ACTION_NOT_FOUND;

#define put_action(Key,Action)                          \
    {                                                   \
        action_vector[(short) (0xff & Key)] = Action;   \
        set_action_is_found(Key);                       \
    }

#define get_action(Key) (action_vector[(short) (0xff & Key)])

/*
 * Miscellaneous external function declarations
 */

extern char *calloc();
extern int  strlen();
extern char *strcpy();
extern int  bzero();

/*
 * Global definitions
 */

char            *ProgramName;               /* basename from argv[0] */
Display         *dpy = NULL;                /* the X connection */
int             screen = 0;                 /* the X screen to use */
Colormap        colormap;                   /* the X colormap to use */
Window          w;                          /* were to put the pixmap */
GC              gc;                         /* how to scribble */

int             PictureWidth  = largescallopshell_width;
int             PictureHeight = largescallopshell_height;
Pixmap          PictureIcon;

Pixmap          largescallopshell_pixmap;   /* for drawing */
Pixmap          scallopshell_pixmap;        /* for drawing */

int             window_width;               /* for drawing */
int             window_height;              /* for drawing */

Pixel           foreground_pixel;           /* for drawing */
Pixel           background_pixel;           /* for drawing */
Pixel           border_pixel;               /* for drawing */
int             border_width;
int             volume;                     /* how loud to ring bell */
int             num_flash;                  /* number of times to flash */
struct timeval  flash_delaytime;            /* delay between flashes */
Bool            bool_reverse_video;         /* true => reverse video */
Bool            bool_quiet;                 /* true => beeps are preformed */
Bool            use_small_scallopshell_icon;/* true => use small icon */

char            *displayname = NULL;        /* which display to use */


char            action_flags[ACTION_TABLE_SIZE];
char            **action_vector[ACTION_TABLE_SIZE];     /* command table */

char            *yes_on_truec[] =
                {
                    "yes", "on", "true", (char *) NULL
                };

int 	    	yes_on_truel[] =
    	    	{
		    1, 2, 1, (int) NULL
    	        };

char            *smallc[] =
                {
                    "small", "little", (char *) NULL
                };

int             smalll[] =
                {
                    1, 1, (int) NULL
                };


/*
 * xshell_shutdown -- exit cleanly
 */

void xshell_shutdown(status)
    int status;
{

    if (dpy)
    {
        if (w)
            XUnmapWindow(dpy, w);
        XCloseDisplay(dpy);
    }
    exit(status);
}

/*
 * New version of IsMemberOf to accept abbreviations.
 *  listc array: List of character strings to be a member of, case
 *  	    	 insensitive
 *  listl array: List of minimum abbreviations allowed for strings in listc.
 *  	         If abbreviations are not allowed for word, a number
 *  	    	 (<0) should be used. A number of 0 IS NOT ALLOWED.
 */

int IsMemberOf (listc, listl, string)
        register char 	**listc;
        register int   	*listl;
        register char 	*string;
{
    int	    good;
    int     done;
    char    *chp;
    char    *thp;

    for (; *listc; listl++, listc++)
    {
	/*
	 * chp = input string, thp = token string
	 */
	for (chp = string, thp = *listc, good = YES, done = NO;
	     (done == NO);
	     chp++, thp++)
	{
	    /* Hit end of token and not end of input? */
	    if (*thp == 0)
	    {
		done = YES;
		if (*chp != 0)
		{
		    good = NO;
		}
		break;
	    }
	    /* End of string input, always done */
	    if (*chp == 0)
	    {
		done = YES;
		break;
	    }
	    /* Both have chars - check match - case insensitive */
	    if ((islower(*chp) ? (toupper(*chp)) : (*chp)) !=
		(islower(*thp) ? (toupper(*thp)) : (*thp)))
	    {
		done = YES;
		good = NO;
		break;
	    }
	}
	if (!((good == NO) || ((thp - *listc) < *listl) ||
	      (((*chp != NULL) || (*thp != NULL)) && (*listl < 0))))
	{
	    return(YES);
	}
    }
    return(NO);
}

/*
 * Stolen directly from Jim Fulton's X.V10R4 version of xbiff.
 */

static void Error (msg, arg)
    char    *msg;
    char    *arg;
{
        (void) fprintf(stderr, "%s:  error with %s", msg);
        if (arg != (char *) NULL && *arg != '\0')
            (void) fprintf(stderr, ": %s\n", arg);
        else
            (void) fprintf(stderr, "\n");
        exit(1);
}

/*
 * make_picture - returns a pixmap made from the given bitmap and pixels;
 * (taken from xlib-xbiff.c written by Jim Fulton).
 */

Pixmap make_picture(bits, width, height, fg, bg)
    char    *bits;
    int     width;
    int     height;
    Pixel   fg;
    Pixel   bg;
{
    return (XCreatePixmapFromBitmapData (dpy, w, bits, width, height, fg, bg,
					 DefaultDepth (dpy, screen)));
}

static Pixel get_a_color(name, monodefault)
    register char   *name;
    register Pixel  monodefault;
{
    XColor  hard_col;
    XColor  exact_col;

    if (!name) return(monodefault);
    if (strcmp(name, "black") == 0) return(BlackPixel(dpy, screen));
    if (strcmp(name, "white") == 0) return(WhitePixel(dpy, screen));
    if (DisplayCells (dpy, screen) <= 2) return(monodefault);
    if (XAllocNamedColor(dpy, colormap, name, &hard_col, &exact_col))
          return(hard_col.pixel);
    return(monodefault);
}

/*
 * Tell the user what options this program expects.
 * NOTE: I did not implement any command line options for the "action"
 * entries. These must be specified through the ~/.Xdefaults file.
 */

static
void xshell_usage()
{
    (void) fprintf(stderr, "usage:  %s [options] [=geom] [host:dpy]\n",
                   ProgramName);
    (void) fprintf(stderr, "where options are:\n");
    (void) fprintf(stderr,
    "     -fg color       foreground color              (default black)\n");
    (void) fprintf(stderr,
    "     -bg color       background color              (default white)\n");
    (void) fprintf(stderr,
    "     -bd color       border color                  (default black)\n");
    (void) fprintf(stderr,
    "     -bw[idth]  n    border width in pixels        (default  0)\n");
    (void) fprintf(stderr,
    "     -v[olume]  n    bell volume in percent        (default 33)\n");
    (void) fprintf(stderr,
    "     -fl[ash]   n    number of times to flash icon (default  3)\n");
    (void) fprintf(stderr,
    "     -d[elay]   n    1/10ths of a second flashed   (default  5)\n");
    (void) fprintf(stderr,
    "     -r[everse]      reverse video\n");
    (void) fprintf(stderr,
    "     +r[everse]      normal video                  (default)\n");
    (void) fprintf(stderr,
    "     -q[uiet]        don\'t beep on errors\n");
    (void) fprintf(stderr,
    "     +q[uiet]        beep on errors                (default)\n");
    (void) fprintf(stderr,
    "     -s[mall]        use small icon not big one\n");
    (void) fprintf(stderr,
    "     +s[mall]        use big icon not small one    (default)\n");

    exit(1);
}

/*
 * This procedure largely stolen from Jim Fulton's xlib-xbiff program.
 * Any bugs introduced are my own.
 */

static
xshell_init(argc, argv)
    int     argc;
    char    **argv;
{
    int                     i;
    char                    *geom = NULL;
    XSizeHints              hints;
    XSetWindowAttributes    attrs;
    XClassHint              class_hints;
    int                     display_width;
    int                     display_height;
    int                     geom_result;
    XGCValues               gcv;
    char                    *fg_name = NULL;
    char                    *bg_name = NULL;
    char                    *bd_name = NULL;

    Bool                    borderWidth_given     = False;
    Bool                    volume_given          = False;
    Bool                    num_flash_given       = False;
    Bool                    flash_delaytime_given = False;
    Bool                    reverseVideo_given    = False;
    Bool                    quiet_given           = False;
    Bool                    smallIcon_given       = False;

    Pixel                   black;
    Pixel                   white;

    /*
     * Initialize the action flags and vector. Someday a kill -HUP
     * might mean to throw away all of the old data and re-read the
     * ~/.Xdefaults file ...
     */

    for (i = 0; i < ACTION_TABLE_SIZE; i++)
    {
        action_flags[i]  = ACTION_NEW;
        action_vector[i] = NULL;
    }

    /*
     * parse the command line arguments to find out what options
     * should not be read from the ~/.Xdefaults file.
     */

    for (i = 1; i < argc; i++)
    {
        char    *arg = argv[i];

        if (arg[0] == '=')
        {
            geom = arg;
        }
        else if (arg[0] == '-' || arg[0] == '+')
        {
            int negated = (arg[0] == '-');
            switch(arg[1])
            {
              case 'r':         /* -rv or -reverse */
                bool_reverse_video = negated;
                reverseVideo_given = True;
                break;
              case 'b':
                switch (arg[2])
                {
                  case 'w':     /* -bw pixels */
                    if (++i >= argc) xshell_usage();
                    border_width = atoi(argv[i]);
                    borderWidth_given = True;
                    break;
                  case 'd':     /* -bd color */
                    if (++i >= argc) xshell_usage();
                    bd_name = argv[i];
                    break;
                  case 'g':     /* -bg color */
                    if (++i >= argc) xshell_usage();
                    bg_name = argv[i];
                    break;
                  default:
                    xshell_usage();
                    break;
                }
                break;
              case 'v':         /* -volume percent */
                if (++i >= argc) xshell_usage();
                volume = atoi(argv[i]);
                volume_given = True;
                break;
              case 'f':
                switch (arg[2])
                {
                  case 'g':     /* -fg color */
                    if (++i >= argc) xshell_usage();
                    fg_name = argv[i];
                    break;
                  case 'l':     /* -fl[ash] */
                    if (++i >= argc) xshell_usage();
                    num_flash = atoi(argv[i]);
                    num_flash_given = True;
                    break;
                  default:
                    xshell_usage();
                    break;
                }
                break;
              case 'd':         /* -delay tenths_of_second */
                if (++i >= argc) xshell_usage();
                flash_delaytime.tv_sec = 0;
                flash_delaytime.tv_usec = atoi(argv[i]) * micro2centi;
                flash_delaytime_given = True;
                break;
              case 'q':         /* -quiet or +quiet */
                bool_quiet = negated;
                quiet_given = True;
                break;
              case 's':         /* -small or +small */
                use_small_scallopshell_icon = negated;
                smallIcon_given = True;
                break;
              default:
                xshell_usage();
                break;
            }
        }
        else
        {
            displayname = arg;
        }
    }

    /*
     * open a connection to the display
     */

    dpy = XOpenDisplay(displayname);
    if (!dpy)
    {
        (void) fprintf(stderr, "%s:  unable to open display \"%s\".\n",
                       ProgramName, XDisplayName(displayname));
        exit(1);
    }

    screen          = DefaultScreen(dpy);
    colormap        = DefaultColormap(dpy, screen);

    display_width   = DisplayWidth(dpy, screen);
    display_height  = DisplayHeight(dpy, screen);

    /*
     * get the resources, SHOULD USE RESOURCE MANAGER!  Actually, should use
     * toolkit.
     */

    if (!fg_name) fg_name = XGetDefault(dpy, ProgramName, "foreground");
    if (!bg_name) bg_name = XGetDefault(dpy, ProgramName, "background");
    if (!bd_name) bd_name = XGetDefault(dpy, ProgramName, "border");
    if (!borderWidth_given)
    {
        char    *cp = XGetDefault(dpy, ProgramName, "borderWidth");
        if (cp)
            border_width = atoi(cp);
        else
            border_width = 0;   /* default value */
    }
    if (!num_flash_given)
    {
        char    *cp = XGetDefault(dpy, ProgramName, "Flash");
        if (cp)
            num_flash = atoi(cp);
        else
            num_flash = 3;      /* default value */
    }
    if (!flash_delaytime_given)
    {
        char    *cp = XGetDefault(dpy, ProgramName, "Delay");
        flash_delaytime.tv_sec = 0;
        if (cp)
            flash_delaytime.tv_usec = atoi(cp) * micro2centi;
        else
            flash_delaytime.tv_usec = 5 * micro2centi; /* default value */
    }
    if (!reverseVideo_given)
    {
        char    *cp = XGetDefault(dpy, ProgramName, "reverseVideo");
        if (cp && IsMemberOf(yes_on_truec, yes_on_truel, cp))
            bool_reverse_video = True;
        else
            bool_reverse_video = False; /* default value */
    }
    if (!volume_given)
    {
        char    *cp = XGetDefault(dpy, ProgramName, "volume");
        if (cp)
            volume = atoi(cp);
        else
            volume = 33;        /* default value */
    }
    if (!quiet_given)
    {
        char    *cp = XGetDefault(dpy, ProgramName, "Quiet");
        if (cp && IsMemberOf(yes_on_truec, yes_on_truel, cp))
            bool_quiet = True;
        else
            bool_quiet = False; /* default value */
    }
    if (!smallIcon_given)
    {
        char    *cp = XGetDefault(dpy, ProgramName, "IconSize");
        if (cp && IsMemberOf(smallc, smalll, cp))
            use_small_scallopshell_icon = True;
        else
            use_small_scallopshell_icon = False; /* default value */
    }

    /*
     * Choose which icon to use.
     */

    if (use_small_scallopshell_icon)
    {
        PictureWidth   = scallopshell_width;
        PictureHeight  = scallopshell_height;
    }
    else
    {
        PictureWidth  = largescallopshell_width;
        PictureHeight = largescallopshell_height;
    }

    /*
     * get the necessary colors
     */

    black            = (Pixel) BlackPixel(dpy, screen);
    white            = (Pixel) WhitePixel(dpy, screen);
    foreground_pixel = get_a_color(fg_name, black);
    background_pixel = get_a_color(bg_name, white);
    border_pixel     = get_a_color(bd_name, black);

    if (bool_reverse_video)
    {
        Pixel   tmp;

        tmp              = background_pixel;
        background_pixel = foreground_pixel;
        foreground_pixel = tmp;
    }

    /*
     * parse the command line geometry flag; style taken from
     * clients/bitmap/bitmap.c and other applications
     */

    geom_result      = NoValue;
    hints.min_width  = PictureWidth;
    hints.min_height = PictureHeight;
    hints.flags      = PMinSize;

    if (geom) geom_result = XParseGeometry(geom, &hints.x, &hints.y,
                                           &hints.width, &hints.height);

    if ((geom_result & WidthValue) && (geom_result & HeightValue))
    {
        hints.width  = max(hints.width, hints.min_width);
        hints.height = max(hints.height, hints.min_height);
        hints.flags |= USSize;
    }

    if ((geom_result & XValue) && (geom_result & YValue))
    {
        hints.flags |= USPosition;
    }

    /*
     * supply program defaults in case user didn't provide info; some window
     * managers (like uwm) will ignore our information and force the user to
     * fully specify the position.  sigh.
     */

    if (!(hints.flags & USSize))
    {
        hints.width  = PictureWidth;
        hints.height = PictureHeight;
        hints.flags |= PSize;
    }

    if (!(hints.flags & USPosition))
    {
        hints.x      = hints.y = 0;
        hints.flags |= PPosition;
    }

    if (geom_result & XNegative)
    {
        /*
         * unlike X10, in X11 you give hints without borders
         */

        hints.x = display_width + hints.x - hints.width;
    }
    if (geom_result & YNegative)
    {
        /*
         * unlike X10, in X11 you give hints without borders
         */

        hints.y = display_height + hints.y - hints.height;
    }

    /*
     * build the window
     */

    attrs.background_pixel = background_pixel;
    attrs.border_pixel     = border_pixel;

    attrs.event_mask  = ExposureMask | ButtonPressMask | KeyPressMask |
        StructureNotifyMask;

    attrs.bit_gravity = ForgetGravity; /* use exposures instead */
    attrs.cursor      = XCreateFontCursor(dpy, XC_top_left_arrow);

    w = XCreateWindow(dpy, RootWindow(dpy, screen),
                      hints.x, hints.y, hints.width, hints.height,
                      border_width,
                      CopyFromParent, CopyFromParent, CopyFromParent,
                      CWBackPixel | CWBorderPixel | CWEventMask | CWCursor |
                      CWBitGravity,
                      &attrs);
    if (!w)
    {
        (void) fprintf(stderr, "%s:  unable to create window.\n", ProgramName);
        xshell_shutdown(1);
    }

    /*
     * set the hints in the right order
     */

    class_hints.res_name  = "xshell_command_window";
    class_hints.res_class = "xshell";
    XSetClassHint(dpy, w, &class_hints);

    window_width  = hints.width;        /* save for making pretty */
    window_height = hints.height;
    XSetStandardProperties(dpy, w, "xshell", "xshell", None,
                           argv, argc, &hints);

    gcv.function = GXcopy;
    gcv.graphics_exposures = False; /* don't want {Graphics,No}Expose */
    gc = XCreateGC(dpy, w, GCFunction | GCGraphicsExposures, &gcv);

    /*
     * build up the pixmaps that we'll put into the image
     */

    scallopshell_pixmap = make_picture((char *) scallopshell_bits,
                                       scallopshell_width, scallopshell_height,
                                       foreground_pixel, background_pixel);
    if (!scallopshell_pixmap)
    {
        (void) fprintf(stderr, "%s:  unable to make xshell pixmaps.\n",
                       ProgramName);
        xshell_shutdown(1);
    }

    largescallopshell_pixmap = make_picture((char *) largescallopshell_bits,
                                            largescallopshell_width,
                                            largescallopshell_height,
                                            foreground_pixel,
                                            background_pixel);
    if (!largescallopshell_pixmap)
    {
        (void) fprintf(stderr, "%s:  unable to make xshell pixmaps.\n",
                       ProgramName);
        xshell_shutdown(1);
    }

    if (use_small_scallopshell_icon)
    {
        PictureIcon    = scallopshell_pixmap;
        PictureWidth   = scallopshell_width;
        PictureHeight  = scallopshell_height;
    }
    else
    {
        PictureIcon   = largescallopshell_pixmap;
        PictureWidth  = largescallopshell_width;
        PictureHeight = largescallopshell_height;
    }

    /*
     * and let it rip
     */

    XMapWindow(dpy, w);
    XFlush(dpy);

    /*
     * we should now have ConfigureNotify, MapNotify, and Expose events
     * on their way.
     */
}

void redraw()
{
    int     x;
    int     y;
    Pixel   back;
    Pixel   border;

    /* center the picture in the window */

    x = (window_width - PictureWidth) / 2;
    x = max(x, 0);
    y = (window_height - PictureHeight) / 2;
    y = max(y, 0);

    back    = background_pixel;
    border  = foreground_pixel;

    XSetWindowBackground(dpy, w, back);
    XSetWindowBorder(dpy, w, border);
    XClearWindow(dpy, w);
    XCopyArea(dpy, PictureIcon, w, gc, 0, 0, PictureWidth, PictureHeight, x, y);
}

/*
 * Ring the bell...and maybe win a prize?
 */
void beep()
{
    XBell(dpy, volume);
}

/*
 * Flash the window
 */
void flash()
{
    register int    i;

    for (i = 0; i < num_flash; i++)
    {
        redraw();
        (void) select(0, (fd_set *) NULL, (fd_set *) NULL, (fd_set *) NULL,
                      &flash_delaytime);
    }
}

/*
 * stringFuncVal returns the appropriate string for a KeySym keycode.
 *
 * This code will be unnecessary when the XRebindKeySym stuff is
 * implemented. Then the call to XLookupString will be able to return
 * the text string for a function key. We will still need an
 * initialization step to map these strings to the keysyms.
 *
 * BUG: The following table assumes that the XK_R1 through XK_R15 keys
 * and XK_L1 through XK_L9 are preferred over the XK_F* functions as
 * far as the string names are concerned. If the XK_R1 keysym is given
 * there should be more than one XGetDefault call for the alternate
 * names by which this symbol could be called.
 *
 * This code should be re-written for X.V11R2.
 */
char *stringFuncVal(keycode)
    KeySym  keycode;
{
    switch ((int) keycode)
    {
        /* Cursor control & motion */

      case XK_Home:     return("Home");
      case XK_Left:     return("LeftArrow");
      case XK_Up:       return("UpArrow");
      case XK_Right:    return("RightArrow");
      case XK_Down:     return("DownArrow");
      case XK_Prior:    return("Prior");
      case XK_Next:     return("Next");
      case XK_End:      return("End");
      case XK_Begin:    return("Begin");

        /* Special Ascii characters */

      case XK_BackSpace:    return("BackSpace"); /* back space, back char,...*/
      case XK_Tab:          return("Tab");
      case XK_Clear:        return("Clear");
      case XK_Linefeed:     return("Linefeed"); /* Linefeed, LF */
      case XK_Return:       return("Enter");    /* Return, enter */
      case XK_Pause:        return("Pause");    /* Pause,hold,scroll lock,...*/
      case XK_Escape:       return("Escape");
      case XK_Delete:       return("Delete");   /* Delete <>, rubout */

        /* Misc Functions */

      case XK_Select:   return("Select");       /* Select, mark */
      case XK_Print:    return("Print");
      case XK_Execute:  return("Execute");      /* Execute, run, do */
      case XK_Insert:   return("Insert");       /* Insert, insert here */
      case XK_Undo:     return("Undo");         /* Undo, oops */
      case XK_Redo:     return("Redo");         /* redo, again */
      case XK_Menu:     return("Menu");
      case XK_Find:     return("Find");         /* Find, search */
      case XK_Cancel:   return("Cancel");       /* Cancel, stop, abort, exit */
      case XK_Help:     return("Help");         /* Help, ? */
      case XK_Break:    return("Break");        /* Sun3 "Alternate" key */
      case XK_script_switch:                    /* Alias for mode_switch */
        return("ScriptSwitch");
      case XK_Num_Lock: return("NumLock");

        /* Auxilliary Functions */

      case XK_L1:       return("L1");
      case XK_L2:       return("L2");
      case XK_L3:       return("L3");
      case XK_L4:       return("L4");
      case XK_L5:       return("L5");
      case XK_L6:       return("L6");
      case XK_L7:       return("L7");
      case XK_L8:       return("L8");
      case XK_L9:       return("L9");
      case XK_L10:      return("L10");

      case XK_R1:       return("R1");
      case XK_R2:       return("R2");
      case XK_R3:       return("R3");
      case XK_R4:       return("R4");
      case XK_R5:       return("R5");
      case XK_R6:       return("R6");
      case XK_R7:       return("R7");
      case XK_R8:       return("R8");
      case XK_R9:       return("R9");
      case XK_R10:      return("R10");
      case XK_R11:      return("R11");
      case XK_R12:      return("R12");
      case XK_R13:      return("R13");
      case XK_R14:      return("R14");
      case XK_R15:      return("R15");

      case XK_F1:       return("F1");
      case XK_F2:       return("F2");
      case XK_F3:       return("F3");
      case XK_F4:       return("F4");
      case XK_F5:       return("F5");
      case XK_F6:       return("F6");
      case XK_F7:       return("F7");
      case XK_F8:       return("F8");
      case XK_F9:       return("F9");

        /* Keypad Functions, keypad numbers cleverly chosen to map to ascii */

      case XK_KP_Space:         return("KeypadSpace");  /* space */
      case XK_KP_Tab:           return("KeypadTab");
      case XK_KP_Enter:         return("KeypadEnter");  /* enter */
      case XK_KP_F1:            return("Pf1");          /* PF1, KP_A, ... */
      case XK_KP_F2:            return("Pf2");
      case XK_KP_F3:            return("Pf3");
      case XK_KP_F4:            return("Pf4");
      case XK_KP_Equal:         return("Keypad=");      /* equals */
      case XK_KP_Multiply:      return("Keypad*");
      case XK_KP_Add:           return("Keypad+");
      case XK_KP_Separator:     return("Keypad,");      /* separator */
      case XK_KP_Subtract:      return("Keypad-");
      case XK_KP_Decimal:       return("Keypad.");
      case XK_KP_Divide:        return("Keypad/");
      case XK_KP_0:             return("Keypad0");
      case XK_KP_1:             return("Keypad1");
      case XK_KP_2:             return("Keypad2");
      case XK_KP_3:             return("Keypad3");
      case XK_KP_4:             return("Keypad4");
      case XK_KP_5:             return("Keypad5");
      case XK_KP_6:             return("Keypad6");
      case XK_KP_7:             return("Keypad7");
      case XK_KP_8:             return("Keypad8");
      case XK_KP_9:             return("Keypad9");

        /* Modifiers */

      case XK_Shift_L:                                  /* Left shift */
      case XK_Shift_R:                                  /* Right shift */
        return("Shift");

      case XK_Control_L:                                /* Left control */
      case XK_Control_R:                                /* Right control */
        return("Control");

      case XK_Caps_Lock:        return("CapsLock");     /* Caps lock */
      case XK_Shift_Lock:       return("ShiftLock");    /* Shift lock */

      case XK_Meta_L:                                   /* Left meta */
      case XK_Meta_R:                                   /* Right meta */
        return("Meta");
      case XK_Alt_L:                                    /* Left alt */
      case XK_Alt_R:                                    /* Right alt */
        return("Alt");

      case XK_Super_L:                                  /* Left super */
      case XK_Super_R:                                  /* Right super */
        return("Super");
      case XK_Hyper_L:                                  /* Left hyper */
      case XK_Hyper_R:                                  /* Right hyper */
        return("Hyper");

      default:
        return("");
    }

}

/*
 * Stolen from the X.V10R4 version of xshell.
 */
static parseaction (keycode, actionstring)
    KeySym  keycode;
    char    *actionstring;
{
        register char   *cp;
        register int    wc = 0;         /* word count */
        register int    inword;
        register char   **actionlist;
        register int    i;

        for (inword = 0,cp = actionstring; *cp; cp++)
        {
            /*
             * iterate over string
             */
            if (isspace(*cp))
            {
                if (inword)
                    inword = 0;        /* no longer in a word */
            }
            else
            {
                if (!inword)
                {                               /* weren't in word */
                    inword = 1;                 /* but now we are */
                    wc++;                       /* so increment counter */
                }
            }
        }
        /*
         *  wc now contains the number of separate words
         */
        actionlist = (char **) calloc((unsigned) (wc + 1),
                                      (unsigned) sizeof (char *));
        if (!actionlist)
            Error("allocating memory for command list", actionstring);

        for (i =0, inword = 0, cp = actionstring; *cp; cp++)
        {
            if (isspace(*cp))
            {
                if (inword)
                {                               /* were in a word */
                    inword = 0;                 /* but now we're not */
                }
                *cp = '\0';                     /* and null out space */
            }
            else
            {
                if (!inword)
                {                               /* weren't in a word */
                    inword = 1;                 /* but now we are */
                    actionlist [i++] = cp;      /* store pointer to start of word */
                }
            }
        }
        actionlist[wc] = (char *) NULL;         /* execv wants this */

        put_action(keycode, actionlist);        /* store the action */
}

/*
 * perform - This routine looks in its tables to see if it already has
 * a key definition, else it does an XGetDefault of the keyname.
 */
static
void perform(keyname, keycode)
    char    *keyname;
    KeySym  keycode;
{
    char    buf[32];
    char    *cp;

    if (action_is_new(keycode))
    {
        (void) strcpy(buf, "action.");
        (void) strcat(buf, keyname);
        cp = XGetDefault(dpy, ProgramName, buf);
        if (!cp)
        {
            set_action_notfound(keycode);
        }
        else
        {
            parseaction(keycode, cp);
        }
    }
    if (action_is_found(keycode))
    {
        if (vfork() == 0)       /* in child, start program */
        {
            char    **action;
            action = get_action(keycode);
            execvp(action[0], action);
        }
        else
            flash();
    }
    else
    {
        if (!bool_quiet)
            beep();
    }
}

/*
 * do_action -- process an input event and perform command.
 */
void do_action(event)
    XEvent      *event;
{
    int             nbytes;
    char            mapping_buf[32];
    XComposeStatus  status;
    KeySym          keysym;

    (void) bzero(mapping_buf, sizeof(mapping_buf));

    switch((int) event->type)
    {
      case KeyPress:
        nbytes = XLookupString(event, mapping_buf, sizeof(mapping_buf),
                               &keysym, &status);

        if (IsFunctionKey(keysym)       ||
            IsMiscFunctionKey(keysym)   ||
            IsKeypadKey(keysym)         ||
#ifdef LOOKUP_MODIFIER_KEYS
            IsModifierKey(keysym)       ||
#endif
            IsCursorKey(keysym))
        {
            (void) strcpy(mapping_buf, stringFuncVal(keysym));
            nbytes = strlen(mapping_buf);
        }
        else
        {
            if (nbytes == 1)
            {
                /*
                 * Special characters that might not parse well..
                 *
                 * This is a HACK!!!!!
                 *
                 * Be VERY careful that the second
                 * byte of a function key does not equal one of the
                 * control characters that you add to this switch
                 * statement!
                 */
                switch(*mapping_buf)
                {
                  case ' ':
                    (void) strcpy(mapping_buf, "Space");
                    nbytes = strlen(mapping_buf);
                    break;
                  case ':':
                    (void) strcpy(mapping_buf, "Colon");
                    nbytes = strlen(mapping_buf);
                    break;
                  case '\t':
                    (void) strcpy(mapping_buf, "Tab");
                    nbytes = strlen(mapping_buf);
                    break;
                  case '\n':
                    (void) strcpy(mapping_buf, "Newline");
                    nbytes = strlen(mapping_buf);
                    break;
                  case '\r':
                    (void) strcpy(mapping_buf, "Enter");
                    nbytes = strlen(mapping_buf);
                    break;
                  case '\f':
                    (void) strcpy(mapping_buf, "Formfeed");
                    nbytes = strlen(mapping_buf);
                    break;
                  default:
                    break;
                }
            }
            break;
        }
      case ButtonPress:
        /*
         * Warning: This assumes that no function key entry or
         * character entry will be available for ascii 0 through 5.
         */
        switch (event->xbutton.button)
        {
          case Button1:
            (void) strcpy(mapping_buf, "LeftButton");
            keysym = 1;
            break;
          case Button2:
            (void) strcpy(mapping_buf, "MiddleButton");
            keysym = 2;
            break;
          case Button3:
            (void) strcpy(mapping_buf, "RightButton");
            keysym = 3;
            break;
          case Button4:
            (void) strcpy(mapping_buf, "Button4");
            keysym = 4;
            break;
          case Button5:
            (void) strcpy(mapping_buf, "Button5");
            keysym = 5;
            break;
          default:
            (void) strcpy(mapping_buf, "SomeButton");
            keysym = 0;
            break;
        }
        nbytes = strlen(mapping_buf);
        break;
      default:
        break;
    }
    /*
     * Some events (like the modifier keys) will not have an action
     * string associated with them.
     */
    if (nbytes > 0)
        perform(mapping_buf, keysym);
}

/*
 * xshell_loop -- process events until completion.
 */

static
void xshell_loop()
{
    XEvent          event;
    XConfigureEvent *configure_event_p = (XConfigureEvent *) &event;

    while (1)
    {
        XNextEvent (dpy, &event);
        switch (event.type)
        {
          case ConfigureNotify:
            window_width = configure_event_p->width;  /* for making pretty */
            window_height = configure_event_p->height;
            break;

          case CirculateNotify:
          case DestroyNotify:
          case GravityNotify:
          case MapNotify:
          case ReparentNotify:
          case UnmapNotify:
            /* ignore */
            break;

          case Expose:
            redraw();
            break;

          case KeyPress:
          case ButtonPress:
            do_action(&event);
            break;

          default:
            (void) fprintf(stderr,
                    "%s:  unexpected event type %d from display \"%s\".\n",
                    ProgramName, event.type, DisplayString(dpy));
            break;
        }
    }
}

/*
 * This handler routine is called whenever there is a SIGINT or SIGHUP
 * received. Both of these conditions are currently defined to cause
 * xshell to shutdown.
 */
/*ARGSUSED*/
int quit(sig, code, scp)
    int                 sig;
    int                 code;
    struct sigcontext   *scp;
{
    xshell_shutdown(0);

    return(0);                  /* This statement never executed.  */
}

/*
 * Stolen from Jim Fulton's X.V10R4 version of xbiff.
 */
/*ARGSUSED*/
int reapchild (sig, code, scp)
    int                 sig;
    int                 code;
    struct sigcontext   *scp;
{
    union wait status;

    while (wait3 (&status, WNOHANG, (struct rusage *) 0) > 0) ;
    return(0);
}

/*
 * Main -- let's start at the very beginning...
 */
int main(argc, argv)
    int     argc;
    char    **argv;
{
    char    *cp;

    /*
     * Strip the program name to the basename component.
     */
    ProgramName = argv[0];
    cp = rindex(argv[0], '/');
    if(cp)
        ProgramName = ++cp;     /* strip off directory name */

    (void) signal(SIGHUP, quit);        /* hangup */
    (void) signal(SIGINT, quit);        /* interrup */
    (void) signal(SIGTERM, quit);       /* software termination signal */
    (void) signal(SIGCHLD, reapchild);  /* child status has changed */

    xshell_init(argc, argv);

    xshell_loop();

    xshell_shutdown(0);
}
