# include <X11/Intrinsic.h>
# include <sys/wait.h>
# include <sys/file.h>
# include <signal.h>
# include <fcntl.h>
# include <pwd.h>
# include <stdio.h>

# include "globs.h"

static int	pids[128];
static int	npids;
static int	dohang;

/*
 * client_start reads the .xtools file and runs each line.
 */
client_start() {
	extern char	*index();
	extern void	shutdown();
	extern void	cleanup();
	extern char	*getname();
	extern char	*getline();
	FILE		*fd;
	char		*cp;

	/*
	 * if getname can't file a .xtools file then just
	 * start an xterm console.
	 */
	if ((cp = getname()) == NULL) {
		if (execute("xterm -C &") == -1) {
			fprintf(stderr, "can't start console xterm\n");
			return(-1);
		}
		return(0);
	}

	if ((fd = fopen(cp, "r")) == NULL) {
		(void) fprintf(stderr, "open: ");
		perror(cp);
		return(-1);
	}

	if (fcntl(fileno(fd), F_SETFD, 1) == -1) {
		(void) fprintf(stderr, "fcntl: ");
		perror(cp);
		(void) fclose(fd);
		return(-1);
	}

	(void) signal(SIGINT, (int (*)()) shutdown);
	(void) signal(SIGCHLD, (int (*)()) cleanup);

	while ((cp = getline(fd)) != NULL) {
		if (execute(cp) == -1)
			(void) fprintf(stderr, "can't execute \"%s\"\n", cp);
	}
	dohang = 0;

	fclose(fd);

	return(0);
}

/*
 * getname returns the pathname to the user's .xtools file.
 */
static char *
getname() {
	extern char	*getenv();
	extern char	*rindex();
	struct passwd	*pw;
	char		*home, *cp;
	char		host[128];
	static char	path[256];

	if (file != NULL)
		return(file);

	if ((home = getenv("HOME")) == NULL) {
		if ((pw = getpwuid(getuid())) == NULL) {
			(void) fprintf(stderr, "who are you?\n");
			return(NULL);
		}
		home = pw->pw_dir;
		endpwent();
	}

	/*
	 * try finding a .xtools file of the form .xtools-hostname, where
	 * hostname can be the fully qualified host name or it can have
	 * the upper domains missing.  for example,
	 *	.xtools-velveeta.berkeley.edu
	 *	.xtools-velveeta.berkely
	 *	.xtools-velveeta
	 * will all work.
	 */
	if (gethostname(host, sizeof(host)) != -1) {
		sprintf(path, "%s/.xtools-%s", home, host);
		if (access(path, F_OK) == 0)
			return(path);

		/* try trimming off domain */
		while ((cp = rindex(host, '.')) != NULL) {
			*cp = NULL;
			sprintf(path, "%s/.xtools-%s", home, host);
			if (access(path, F_OK) == 0)
				return(path);
		}
	}

	sprintf(path, "%s/.xtools", home);

	if (access(path, F_OK) == 0)
		return(path);

	(void) strcpy(path, DEFAULT_STARTUP);

	if (access(path, F_OK) == 0)
		return(path);

	return(NULL);
}

/*
 * execute runs the line and remembers the pid of the process.
 */
static
execute(line)
	char		*line;
{
	extern char	**getargs();
	extern void	checkamp();
	char		**args;
	int		pid;
	int		i;

	if ((args = getargs(line)) == NULL)
		return(-1);

	checkamp(args);

	if (debug) {
		(void) printf("execvp(\"%s\"", args[0]);
		for (i = 1; args[i] != NULL; i++)
			(void) printf(", \"%s\"", args[i]);
		(void) printf(")\n");
		return(0);
	}

	if ((pid = fork()) == 0) {
		if (setpgrp(0, getpid()) == -1)
			perror("setpgrp(0)");

		execvp(args[0], &args[0]);

		(void) fprintf(stderr, "can't execvp(\"%s\"): ", args[0]);
		perror("execvp");
		exit(1);
	}
	else if (pid > 0) {
		for (i = 0; i < npids; i++) {
			if (pids[i] == 0)
				pids[i] = pid;
		}
		if (i == npids)
			pids[npids++] = pid;

		/* sleep(1); */	/* helps reduce thrashing */

		/*
		 * don't bother checking return status because
		 * some programs do their work and then exit.
		 */
		(void) setpgrp(pid, pid);
	}
	else	/* pid == -1, fork failed */
		return(-1);

	cleanup();

	return(0);
}

/*
 * checkamp checks for an ampersand as either
 * the last argument (i.e., it has a space in
 * front of it, or the last argument has an
 * ampersand appended to it.
 */
static void
checkamp(args)
	char	**args;
{
	int	i, amploc;
	char	*cp;

	dohang = 1;

	for (i = 0; args[i] != NULL; i++)
		continue;

	i--;	/* back up to last non-NULL arg */

	cp = args[i];
	if (strcmp(cp, "&") == 0) {
		args[i] = NULL;
		dohang = 0;
	}
	else if (cp[amploc = strlen(cp)-1] == '&') {
		cp[amploc] = NULL;
		dohang = 0;
	}
}

# define SIGS	(sizeof(sigs) / sizeof(sigs[0]))
int	sigs[] = { SIGTERM /* , SIGHUP, SIGKILL */ };

/*
 * shutdown is called when quitting.  it kills all outstanding
 * processes.
 */
static void
shutdown() {
	int		i, j;

	for (j = 0; j < SIGS; j++) {
		for (i = 0; i < npids; i++) {
			if (pids[i] == 0)
				continue;

			/*
			 * don't try to kill children that have
			 * already exited.
			 */
			if (getpgrp(pids[i]) == -1) {
				pids[i] = 0;
				continue;
			}

			if (killpg(pids[i], sigs[j]) == -1) {
				if (debug)
					(void) printf("can't kill pid %d\n",
						      pids[i]);
				continue;
			}
		}
	}

	dohang = 0;
	cleanup();

	exit(0);
}

/*
 * cleanup collectes dead children.
 */
static void
cleanup() {
	union wait	status;
	int		options;
	int		pid;
	int		i;

	options = (dohang == 1) ? 0 : WNOHANG;

	while ((pid = wait3(&status, options, (struct rusage *) NULL)) > 0) {
		for (i = 0; i < npids; i++) {
			if (pids[i] == pid) {
				if (debug)
					(void) printf("caught pid %d\n",
						      pids[i]);
				pids[i] = 0;
			}
		}

		/* pull up the tail */
		while (pids[npids-1] == 0) {
			npids--;
			if (npids == 0)
				return;
		}
	}
}
