/* 
 * pexclock.c - 3D clock
 * 
 * Copyright 1988
 * Center for Information Technology Integration (CITI)
 * Information Technology Division
 * University of Michigan
 * Ann Arbor, Michigan
 *
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the names of
 * CITI or THE UNIVERSITY OF MICHIGAN not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS." CITI AND THE UNIVERSITY OF
 * MICHIGAN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL CITI OR THE UNIVERSITY OF MICHIGAN BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

#include <math.h>
#include <time.h>
#include "PEXcolors.h"
#include "pexDICE.h"

extern long time();
extern int errno;

#define Forwards 0
#define Reverse 1

#define PI 3.14159265358979
/* 
 * Number of degrees a hand moves; big hand moves this distance once every
 * minute, while the little hand moves this distance once every 12 minutes.
 * and the second hand moves this every second.
 */
#define OneMinuteMove 0.10471976

/*
 * BigHandAngle takes the number of minutes as a parameter and returns the
 * angle the big hand should be at to point to that minute. I.e., at 15
 * minutes past the hour, we return -pi/2; at half past we return -pi, etc.
 */
#define BigHandAngle(min) -1.0*(((min)/60.0)*2*PI)
/*
 * LittleHandAngle
 */
#define LittleHandAngle(hour,min) BigHandAngle((hour*5)+(min/12))

/* face1 and face2 are the polygons and quadmeshes which
   make up the face2 of the clock face */

#define numsteps 36
#define diameter 1.0
#define borderspace 0.04
#define thickness 0.1

pexCoord3D face1[numsteps];
pexCoord3D border[numsteps+1][2];

void InitClockFaces()
{
    double th;
    int i;
    
    th = PI*2;
    border[numsteps][0].y = (float)diameter*sin(th);
    border[numsteps][0].x = (float)diameter*cos(th);
    border[numsteps][0].z = (float)thickness;
    
    border[numsteps][1].y = (float)diameter*sin(th);
    border[numsteps][1].x = (float)diameter*cos(th);
    border[numsteps][1].z = (float)-thickness;

    for (i = 0; i < numsteps; i++)
    {	face1[i].y = (float)(diameter*sin(th));
	face1[i].x = (float)(diameter*cos(th));
	face1[i].z = (float)-thickness;

	border[i][0].y = (float)diameter*sin(th);
	border[i][0].x = (float)diameter*cos(th);
	border[i][0].z = (float)thickness;

	border[i][1].y = (float)diameter*sin(th);
	border[i][1].x = (float)diameter*cos(th);
	border[i][1].z = (float)-thickness;

	th -= (double) (PI*2)/numsteps;
    }
}


/* bighand and littlehand are, of course, the big and little hands,
   at noon position (untransformed) */

#define depth .025



pexCoord3D secondhand[3][3] = {
    -0.001, -0.2, thickness+(2.*depth),    0.0, -0.2, thickness+(2.*depth),    0.001, -0.2, thickness+(2.*depth),
    0.05, 0.0, thickness+(2.*depth),  0.0, 0.0, thickness+(3.*depth),  -0.05, 0.0, thickness+(2.*depth),
    0.001, .99, thickness+(2.*depth),    0.0, .99, thickness+(2.*depth),    -0.001, .99, thickness+(2.*depth)
};
pexCoord3D bighand[3][3] = {
    -0.001, -.3, thickness+(0.*depth), 0.0, -.3, thickness+(0.*depth), 0.001, -.3, thickness+(0.*depth),
    .1, 0.0, thickness+(0.*depth), 0.0, 0.0, thickness+(1.*depth), -.1, 0.0, thickness+(0.*depth),
    .001, .9, thickness+(0.*depth),   0.0, .9, thickness+(0.*depth),  -0.001, .9, thickness+(0.*depth)
};
pexCoord3D littlehand[3][3] = {
    -0.001, -.2, thickness+(1.*depth),  0.0, -.2, thickness+(1.*depth),  0.001, -.2, thickness+(1.*depth), 
    .07, -.07, thickness+(1.*depth), 0.0, 0.0, thickness+(2.*depth),  -.07, -.07, thickness+(1.*depth),
    0.001, 0.56, thickness+(1.*depth), 0.0, 0.56, thickness+(1.*depth), -0.001, 0.56, thickness+(1.*depth)
};



void
DrawClock (pexi, tm)
    pexC *pexi;
    struct tm tm;
{
    Pint errchk;
    Pmatrix3 ltransmat;
    int i;
    pexCoord3D shit[4];

    PexSurfaceColorIndex (pexi, pexYellow);
    PexFillAreaWithData (pexi, Convex, 0, 0, numsteps,
			 face1, NULL, NULL, NULL, NULL);
    protatey (PI, &errchk, ltransmat);
    PexLocalTransform (pexi, Replace, ltransmat);
    PexFillAreaWithData (pexi, Convex, 0, 0, numsteps,
			 face1, NULL, NULL, NULL, NULL);
    
    PexSurfaceColorIndex(pexi, pexBlue);
    protatez (BigHandAngle(tm.tm_min), &errchk, ltransmat);
    PexLocalTransform (pexi, Replace, ltransmat);
    PexQuadrilateralMesh (pexi, Convex, 3, 3,
			 bighand, NULL, NULL, NULL, NULL);
    protatey (PI, &errchk, ltransmat);
    PexLocalTransform (pexi, PostConcatenate, ltransmat);
    PexQuadrilateralMesh (pexi, Convex, 3, 3,
			 bighand, NULL, NULL, NULL, NULL);

    PexSurfaceColorIndex(pexi, pexCyan);
    protatez ((LittleHandAngle(tm.tm_hour, tm.tm_min)),
	      &errchk, ltransmat);
    PexLocalTransform (pexi, Replace, ltransmat);
    PexQuadrilateralMesh (pexi, Convex, 3, 3,
			 littlehand, NULL, NULL, NULL, NULL);
    protatey (PI, &errchk, ltransmat);
    PexLocalTransform (pexi, PostConcatenate, ltransmat);
    PexQuadrilateralMesh (pexi, Convex, 3, 3,
			 littlehand, NULL, NULL, NULL, NULL);
    
    PexSurfaceColorIndex (pexi, pexRed);
    protatez (BigHandAngle(tm.tm_sec), &errchk, ltransmat);
    PexLocalTransform (pexi, Replace, ltransmat);
    PexQuadrilateralMesh (pexi, Convex, 3, 3,
			 secondhand, NULL, NULL, NULL, NULL);
    protatey (PI, &errchk, ltransmat);
    PexLocalTransform (pexi, PostConcatenate, ltransmat);
    PexQuadrilateralMesh (pexi, Convex, 3, 3,
			 secondhand, NULL, NULL, NULL, NULL);

    PexSurfaceColorIndex (pexi, pexMagenta);
    PexQuadrilateralMesh (pexi, Convex, numsteps+1, 2,
  			  border, NULL, NULL, NULL, NULL);

}



main(argc, argv)
    int argc;
    char *argv[];
{
    int i, turnsteps;
    int rotateCube = 0;
    int direction = 0;
    XEvent pe;
    XExposeEvent *ee;
    XButtonPressedEvent *be;
    XConfigureEvent *ce;
    
    struct tm tm, *localtime();
    long time_value;
    
    /* Phigs variables */
    Ppoint3 vrp;		/* view reference point	*/
    Pvector3 vpn, nvpn;		/* view plane normal	*/
    Pvector3 vup;		/* view up vector	*/
    Pint error;			/* error indicator	*/
    Pmatrix3 roty;		/* y rotation matrix	*/
    Pviewmapping3 map;		/* view mapping structure */
    Pviewrep3 viewrep;		/* Phigs view representation */
    
    pexC *pexi;

    solidflag = 1;
    cullflag = 1;

    InitClockFaces();
    pexi = popenphigs(0, 0, argc, argv);
    popenws(pexi, 0, 0, 0);
    
    /*
     * Set up X for MouseButton Events.
     */
    XSelectInput(pexi->phigsDisplay, pexi->phigsWindow, 
		 ExposureMask | StructureNotifyMask | ButtonPressMask );
    

    /* Set up the PHIGS view */
    vrp.x = 0.0; vrp.y = 0.0; vrp.z = 0.0;
    vpn.x = 0.0; vpn.y = 0.5; vpn.z = 3.0;
    vup.x = 0.0; vup.y = 1.0; vup.z = 0.0;
    
    pevalvieworientationmatrix3(&vrp, &vpn, &vup, 
		&error, viewrep.orientation_matrix);
    
    map.window.xmin = -1.1;	map.window.ymin = -1.1;
    map.window.xmax =  1.1;	map.window.ymax =  1.1;
    map.view_plane = 2.0;
    map.back_plane = -1.5;
    map.front_plane = 1.5;
    
    map.viewport.xmin= 0.0; map.viewport.ymin= 0.0; map.viewport.zmin= -1.0;
    map.viewport.xmax= 1.0; map.viewport.ymax= 1.0; map.viewport.zmax= 0.0;
    map.prp.x = 0.0; map.prp.y = 0.0; map.prp.z = 20.0;
    map.proj = PPERSPECTIVE; 
    
    pevalviewmappingmatrix3(&map, &error, viewrep.mapping_matrix);
    
    viewrep.clip_xy = PCLIP;
    viewrep.clip_back = PCLIP;
    viewrep.clip_front = PCLIP;
    viewrep.clip_limit.xmin = map.viewport.xmin;
    viewrep.clip_limit.xmax = map.viewport.xmax;
    viewrep.clip_limit.ymin = map.viewport.ymin;
    viewrep.clip_limit.ymax = map.viewport.ymax;
    viewrep.clip_limit.zmin = map.viewport.zmin;
    viewrep.clip_limit.zmax = map.viewport.zmax;
    
    psetviewrep3(pexi, 1, &viewrep );
    psetviewind(pexi, 1);

    SetPexSurfaceFlags(pexi);
    turnsteps = 0;

    while(1)
    {
	
	while (XPending(pexi->phigsDisplay))
	{
	    XNextEvent(pexi->phigsDisplay, &pe);
	    
	    switch( pe.type ) {
		
	    case Expose:
		ee = (XExposeEvent *) &pe;
		while (ee->count) {
		    XNextEvent(pexi->phigsDisplay, &pe);       
		    ee = (XExposeEvent *) &pe;
		}
		break;
		
	    case NoExpose:
		break;
		
	    case ConfigureNotify:
		ce = (XConfigureEvent *)&pe;
		pexi->winx = ce->x;
		pexi->winy = ce->y;
		pexi->winw = ce->width;
		pexi->winh = ce->height;
		break;
		
	    case ButtonPress:
		be = (XButtonPressedEvent *) &pe;
		switch (be->button)
		{
		case 1:
		    direction = False;
		    rotateCube = True;
		    break;
		case 2:
		    if (!rotateCube)
			turnsteps = 0;
		    rotateCube = False;
		    break;
		case 3:
		    direction = True;
		    rotateCube = True;
		    break;
		}
	    }
	}
	if (!rotateCube)
	    for (i=0; i<5; i++)
		if(!XPending(pexi->phigsDisplay))
		    sleep(1);

	(void) time (&time_value);
	tm = *localtime (&time_value);
	if (tm.tm_hour == 24)
	    tm.tm_hour = 0;
	else if (tm.tm_hour >= 12)
	    tm.tm_hour -= 12;
	
	/* Begin the PEX rendering */
	PexBeginRendering(pexi);
	
	PexBeginStructure(pexi, 0);

	PexRenderOutputCommands(pexi);
	
	if (rotateCube)
	{
	    if (direction)
		turnsteps++;
	    else
		turnsteps--;
	}

	protatey((10.0*PI/180.0)*turnsteps, &error, roty);
	PexGlobalTransform(pexi, roty);
	DrawClock (pexi, tm);
	
	PexEndRendComm(pexi);
	
	/* End the PEX rendering */
	PexEndStructure(pexi);
	PexEndRendering(pexi);
	
	/* Flush X and wait a bit. */
	XFlush(pexi->phigsDisplay);
	if(!XPending(pexi->phigsDisplay))
	    sleep(1);
    }
}
