/* 
 * polygon.c - machine independant dd layer for polygon drawing
 * 
 * Copyright 1988
 * Center for Information Technology Integration (CITI)
 * Information Technology Division
 * University of Michigan
 * Ann Arbor, Michigan
 *
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the names of
 * CITI or THE UNIVERSITY OF MICHIGAN not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS." CITI AND THE UNIVERSITY OF
 * MICHIGAN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL CITI OR THE UNIVERSITY OF MICHIGAN BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

#include <stdio.h>
#include "X.h"
#include "pixmap.h"
#include "gc.h"
#include "miscstruct.h"
#include "gcstruct.h"
#include "extnsionst.h"
#include "dix.h"
#include "dixstruct.h"
#include "resource.h"
#include "colormap.h"

#include "PEX.h"
#include "PEXproto.h"
#include "pubstr.h"
#include "fillarea.h"
#include "renderer.h"
#include "colortable.h"
#include "pexmath.h"
#include "mipex.h"

/*****************************************************************
 * TAG ( miPexPolygon3D )
 * 
 * Draws polygons through the graphics context. In this way, flat shaded
 * polygons are supported across machines with no PEX dd layer.
 * 
 * Inputs:
 * 	a ptr to a renderer;
 * 	a ptr of type pexFillAreaWithDataPtr which contains information for
 * 	  each vertex.
 * Outputs:
 * 	an integer error code or zero;
 * 	the polygon is displayed on the renderer's drawable if it is not
 * 	  clipped. 
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */

#define MaxVertCount 200

extern CARD32 PexClipPolygon ();


int
miPexPolygon3DWire(pRend, pStuff)
    pexRendererPtr pRend;
    pexFillAreaWithDataPtr pStuff;
{
    CARD32 i;			/* Loop counter */
    CARD32 vertsVis;            /* # of visible verts after clipping */
    pexCoord4D pts[MaxVertCount],     	/* Transformed points stored here */
       cpt[MaxVertCount];       /* clipped points stored here */
    register pexCoord4D * npt;
    DDXPointRec ddxpt[MaxVertCount];	/* GC coordinate endpoints */
    DrawablePtr pDraw;
    GCPtr pGC;
    Pixel index;
    int code, outcode = ~0, incode = 0;

    pDraw = pRend -> pDraw;
    pGC =  pRend -> pGC;

    for(i=0; i<(pStuff -> numVertices); i++)
    {
	/* Transform to NPC space */
	PexTransformPoint( pRend, PexFillAreaVertex (pStuff, i), &pts[i] );
	code = PexClipPoint( pts[i] );
	outcode &= code;
	incode |= code;		/* inverse of incode, really */
    }

    /* If all out on one side, return immediately */
    if ( outcode )
	return (Success);

    /* If all points in, don't clip */
    if ( incode )
    {
	vertsVis = PexClipPolygon (pStuff->numVertices, pts, MaxVertCount, cpt);
	npt = cpt;
    }
    else
    {
	vertsVis = pStuff->numVertices;
	npt = pts;
    }

    if (vertsVis)
    {
	for(i=0; i<vertsVis; i++)
	{
	    /* Do the perspective divide */
	    if( npt[i].w != 1.0 )
	    {
		npt[i].x /= npt[i].w;
		npt[i].y /= npt[i].w;
	    }
	    
	    /* Transform to physical device coordinates */
	    PexTransformNpcToDev( pRend, npt[i].x, npt[i].y, &ddxpt[i].x, &ddxpt[i].y );
	}

	if ((ddxpt[0].x != ddxpt[vertsVis-1].x) ||
	    (ddxpt[0].y != ddxpt[vertsVis-1].y))
	    ddxpt[vertsVis++] = ddxpt[0];

	miPexSetGCLineValue(pGC, pDraw,
			   pRend->pPC->surfaceColor.color.format.indexed.index,
			    index);

	/* Draw the polygon as wireframe on the display */
	(*GetGCValue(pGC, Polylines)) (pDraw, pGC, CoordModeOrigin,
					 vertsVis, ddxpt);
    }

    return (Success);
}
	    

int
miPexPolygon3DFlat(pRend, pStuff)
    pexRendererPtr pRend;
    pexFillAreaWithDataPtr pStuff;
{
    CARD32 i;			/* Loop counter */
    CARD32 vertsVis;            /* # of visible verts after clipping */
    pexCoord4D pts[MaxVertCount],     	/* Transformed points stored here */
       cpt[MaxVertCount];       /* clipped points stored here */
    register pexCoord4D * npt;
    DDXPointRec ddxpt[MaxVertCount];	/* GC coordinate endpoints */
    DrawablePtr pDraw;
    GCPtr pGC;
    Pixel index;
    pexVector3D normal;
    int shade, code, outcode = ~0, incode = 0;
    
    pDraw = pRend -> pDraw;
    pGC =  pRend -> pGC;

    for(i=0; i<(pStuff -> numVertices); i++)
    {
	/* Transform to NPC space */
	PexTransformPoint( pRend, PexFillAreaVertex (pStuff, i), &pts[i] );
	code = PexClipPoint( pts[i] );
	outcode &= code;
	incode |= code;		/* incode is not really an incode. */
    }

    /* If all out on one side, return immediately */
    if ( outcode )
	return (Success);

    /* Check to see if we're culling, and if we are, check to see if this
     * facet should be culled. If so, get out fast.
     */
    if ((pRend->pPC->cullMode == BackFaces) &&
	PexCullFacet(pts, pStuff->numVertices))
	return (Success);

    /* If all points in, don't clip */
    if ( incode )
    {
	vertsVis = PexClipPolygon (pStuff->numVertices, pts, MaxVertCount, cpt);
	npt = cpt;
    }
    else
    {
	vertsVis = pStuff->numVertices;
	npt = pts;
    }
 
    if (vertsVis)
    {
	for(i=0; i<vertsVis; i++)
	{
	    /* Do the perspective divide */
	    if( npt[i].w != 1.0 )
	    {
		npt[i].x /= npt[i].w;
		npt[i].y /= npt[i].w;
	    }
	    
	    /* Transform to physical device coordinates */
	    PexTransformNpcToDev( pRend, npt[i].x, npt[i].y, &ddxpt[i].x, &ddxpt[i].y );
	}
	if (pStuff->facetAttributes & GANormal)
	    normal = *PexFillAreaFacetNorm(pStuff);
	else
	    if (pStuff->vertexAttributes & GANormal)
		normal = *PexFillAreaVertexNorm(pStuff, 0); 
	    else
		PexAvrgVertexPoints(pStuff, &normal);
	
	PexTransformNorm(pRend, &normal, &normal);
	shade = PexCalcShade(pRend, &normal);
	
	miPexSetGCValues(pGC, pDraw,
			 pRend->pPC->surfaceColor.color.format.indexed.index,
			 shade, index);

	
	/* Draw the line on the display */
	(*GetGCValue(pGC, FillPolygon)) (pDraw, pGC, Complex, CoordModeOrigin,
					 vertsVis, ddxpt);
    }

    return (Success);
}

int
miPexPolygon3DSmooth(pRend, pStuff)
    pexRendererPtr pRend;
    pexFillAreaWithDataPtr pStuff;
{
    pexCoord4D pts[MaxVertCount];
    int i;
    int cornerShades[4];
    pexVector3D normal;

    if (!(pStuff->vertexAttributes & GANormal) || (pStuff->numVertices > 4))
	return miPexPolygon3DFlat(pRend, pStuff);

    /* Transform points to NPC space */
    for(i=0; i<(pStuff -> numVertices); i++)
	PexTransformPoint( pRend, PexFillAreaVertex (pStuff, i), &pts[i] );
    
    /* Check to see if we're culling, and if we are, check to see if this
     * facet should be culled. If so, get out fast.
     */
    if ((pRend->pPC->cullMode == BackFaces) &&
	PexCullFacet(pts, pStuff->numVertices))
	return (Success);
    
    /* Compute shades */
    for(i=0; i<pStuff->numVertices; i++)
    {
	PexTransformNorm(pRend, PexFillAreaVertexNorm(pStuff, i), &normal);
	cornerShades[i] = PexCalcShade(pRend, &normal);
    }
    
    if (!miPexPolyptSmooth(pts, cornerShades, pStuff->numVertices, pRend,
			   pRend->pPC->surfaceColor.color.format.indexed.index) )
	miPexPolygon3DFlat(pRend, pStuff);
    
    return Success;
}


#define perspDiv(pt)\
	    if( pt.w != 1.0 )\
	    {\
		pt.x /= pt.w;\
		pt.y /= pt.w;\
	    }

miPexPolyptSmooth( polypts, cornerShades, vertsVis, pRend, colorIndex )
pexCoord4D polypts[];
int cornerShades[];
int vertsVis;
pexRendererPtr pRend;
int colorIndex;
{
    int i, j, c, t, size;	     /* counters */
    pexCoord4D pts[MaxVertCount];
    pexCoord4D cpt[MaxVertCount];	/* clipped points stored here */
    register pexCoord4D *opts;
    DDXPointRec ddxpt[MaxVertCount]; /* GC coordinate endpoints */
    Bool facetClipped = FALSE;	     /* Does this facet get clipped? */

    DrawablePtr pDraw;
    GCPtr pGC;
    Pixel index;
    pexVector3D normal;
    int shade;
    int NumTriangles;
    int outcode, code;
    pexCoord4D triPts[2][3];
    int triShades[2][3];
    FLOAT diagonals;
				     /* Variables for SmoothShading!!! */
    pexCoord4D npcEdgePts[3][NUM_SHADES+1];
/* points along the 3 edges */
    int numddxPts[3];

    pDraw = pRend -> pDraw;
    pGC =  pRend -> pGC;

    if ((pRend->pPC->cullMode == BackFaces) &&
	PexCullFacet (polypts, vertsVis))
	return TRUE;

    /*
     * Check to see if the Polygon will be clipped at all.
     * without clipping it.
     */
    
    outcode = ~0;
    for (i = vertsVis-1; i>=0; i-- )
	if (code = PexClipPoint(polypts[i]))
	{
	    outcode &= code;
	    facetClipped = TRUE;
	}
	else
	    outcode = 0;
    /* Is polygon completely outside one plane? */
    if ( outcode != 0 )
	return TRUE;
    
    /* Case
     * 		Polygon was not clipped and is <4 vertices large.
     */

    if (vertsVis == 3)
    {
	triPts[0][0] = polypts[0];
	triPts[0][1] = polypts[1];
	triPts[0][2] = polypts[2];
	
	triShades[0][0] = cornerShades[0];
	triShades[0][1] = cornerShades[1];
	triShades[0][2] = cornerShades[2];
	NumTriangles = 1;
    }
    else
    {
	/* xProduct computes the determinant of the (2x2) matrix with p1 and
	 * p2 as rows.
	 * It is used to compute the vector triple product of three points,
	 * (one xProduct is a minor of the 3x3 determinant).
	 * We wish to determine whether the other two points of a
	 * quadrilateral are on the same side of the chosen diagonal (and thus
	 * if the quadrilateral is concave).
	 */
#define xProduct(p1, p2) (p1.x * p2.y - p1.y * p2.x)
	diagonals = xProduct(polypts[0],polypts[2]);
	NumTriangles = 2;

	if (sign(diagonals*polypts[1].w + xProduct(polypts[1],polypts[0])*polypts[2].w
		 + xProduct(polypts[2],polypts[1])*polypts[0].w) ==
	    sign(diagonals*polypts[3].w + xProduct(polypts[3],polypts[0])*polypts[2].w
		 + xProduct(polypts[2],polypts[3])*polypts[0].w))
	{
	    triPts[0][0] = polypts[0];
	    triPts[0][1] = polypts[1];
	    triPts[0][2] = polypts[2];
	    triPts[1][0] = polypts[0];
	    triPts[1][1] = polypts[3];
	    triPts[1][2] = polypts[2];
	    
	    triShades[0][0] = cornerShades[0];
	    triShades[0][1] = cornerShades[1];
	    triShades[0][2] = cornerShades[2];
	    triShades[1][0] = cornerShades[0];
	    triShades[1][1] = cornerShades[3];
	    triShades[1][2] = cornerShades[2];
	}
	else
	{
	    triPts[0][0] = polypts[1];
	    triPts[0][1] = polypts[0];
	    triPts[0][2] = polypts[3];
	    triPts[1][0] = polypts[1];
	    triPts[1][1] = polypts[2];
	    triPts[1][2] = polypts[3];
	    
	    triShades[0][0] = cornerShades[1];
	    triShades[0][1] = cornerShades[0];
	    triShades[0][2] = cornerShades[3];
	    triShades[1][0] = cornerShades[1];
	    triShades[1][1] = cornerShades[2];
	    triShades[1][2] = cornerShades[3];
	}
    }
    /* triPts now contains NumTriangles triangles which are splits of the
     * original quadrilateral
     *
     * The next 'little' loop will do both of them (slice and dice)
     */
	
    for (t = 0; t < NumTriangles; t++)
    {
	int MaxColorDiff = abs(triShades[t][0] - triShades[t][2]);
	int WheresMax = 0;
	int WheresMin = 0;
	int WheresOther = 0;

	int steps, current, spine, edge;
	pexCoord4D tpoint;
	
	for (i = 1; i < 3; i++)	/* the 0 case is covered by the start values */
	{
	    if (i && MaxColorDiff < abs(triShades[t][i] - triShades[t][i-1]))
		MaxColorDiff = abs(triShades[t][i] - triShades[t][i-1]);
	    
	    if (triShades[t][WheresMax] < triShades[t][i])
		WheresMax = i;

	    if (triShades[t][WheresMin] > triShades[t][i])
		WheresMin = i;
	}

	/* colors are all the same, flat shade */
	if (WheresMin == WheresMax)
	{
	    vertsVis = PexClipPolygon( 3, triPts[t], MaxVertCount, cpt );

	    if ( vertsVis )
	    {
		/* Do the perspective divide */
		for ( j = 0; j < vertsVis; j++ )
		{
		    perspDiv( cpt[j] );
		
		    PexTransformNpcToDev(pRend, cpt[j].x, cpt[j].y,
				&ddxpt[j].x,&ddxpt[j].y);

		}
		miPexSetGCValues(pGC, pDraw, colorIndex, triShades[t][0],
				 index);

		/* Draw the polygon on the display */
		(*GetGCValue(pGC, FillPolygon)) (pDraw, pGC, Complex,
						 CoordModeOrigin, vertsVis,
						 ddxpt);
	    }
	    continue;
	}

	/* Other is defined by max and min by the following table
	 *
	 * 	max	min	other
	 * 	0	1	2
	 * 	1	0	2
	 * 	0	2	1
	 * 	2	0	1
	 * 	1	2	0
	 * 	2	1	0
	 *
	 * 	This reduces to other = (min + max) xor 3
	 */
	WheresOther = (WheresMax + WheresMin) ^ 3;
	    
	/* Now WheresMax is the index into triPts which has the maximum shade.
	 * WheresMin has a similar kinda function.  The edge with the most
	 * color runs from WheresMax to WheresMin. This is the spine. 
	 */
	    
	/* We do some parametric line equation crap to interpolate the points
	 * along the edges.  We fill ddxPts[][] such that the ddxPts[0] is the
	 * spine edge. (that from WheresMin to WheresMax), ddxPts[1] is the
	 * edge from WheresMin to the third "middle corner" WheresOther.
	 */

#define interpolatePoint(p1, p2, t, s, p)\
	p.x = p1.x + (((FLOAT)t/s) * (p2.x - p1.x));\
	p.y = p1.y + (((FLOAT)t/s) * (p2.y - p1.y));\
	p.z = p1.z + (((FLOAT)t/s) * (p2.z - p1.z));\
	p.w = p1.w + (((FLOAT)t/s) * (p2.w - p1.w))

#define DoEdge(startPt, endPt, ddIndex)\
	steps = triShades[t][endPt] - triShades[t][startPt];\
	npcEdgePts[ddIndex][0] = triPts[t][startPt];\
	for(i = 1, c = 1; i <= steps; i++, c += 2 )\
	{\
	    interpolatePoint(triPts[t][startPt], triPts[t][endPt],\
			     c, (steps*2), npcEdgePts[ddIndex][i]);\
	}\
	npcEdgePts[ddIndex][i] = triPts[t][endPt];\
	numddxPts[ddIndex] = i;	/* Points from 0 to numddxPts */
	
	/* DO spine */
	DoEdge(WheresMin, WheresMax, 0);

	/* DO 1st Edge */
	DoEdge(WheresMin, WheresOther, 1);

	/* DO 2nd Edge */
	DoEdge(WheresOther, WheresMax, 2);

	spine = 0;
	edge = 1;
	shade = triShades[t][WheresMin];
	current = 0;

	for (i = 0; i <= MaxColorDiff; i++, shade++)
	{
	    size = 0;
	    
	    if ( i != 0 )
		pts[size++] = npcEdgePts[edge][current];

	    pts[size++] = npcEdgePts[spine][i];
	    pts[size++] = npcEdgePts[spine][i+1];
	    if ( edge == 1 || i != MaxColorDiff )
	    {
		pts[size++] = npcEdgePts[edge][++current];
	    }
	    if (edge == 1 && current == numddxPts[1] && numddxPts[2] != 1 )
	    {
		current = 1;
		edge++;
		pts[size] = pts[size-1];
		pts[(size++)-1] = npcEdgePts[edge][current];
	    }

	    /* polygon is in pts array, clip it */
	    if ( facetClipped )
	    {
		vertsVis = PexClipPolygon(size, pts, MaxVertCount, cpt);
		opts = cpt;
	    }
	    else
	    {
		vertsVis = size;
		opts = pts;
	    }

	/* Do the perspective divide */
	    for ( j = 0; j < vertsVis; j++ )
	    {
		perspDiv( opts[j] );
		
		PexTransformNpcToDev(pRend, opts[j].x, opts[j].y,
			    &ddxpt[j].x,&ddxpt[j].y);

	    }
	    if ( vertsVis )
	    {
		miPexSetGCValues(pGC, pDraw, colorIndex, shade, index);
	    
		/* Draw the polygon on the display */
		(*GetGCValue(pGC, FillPolygon)) (pDraw, pGC, Complex,
						 CoordModeOrigin, vertsVis,
						 ddxpt);
	    }
	}
    }
    
    
    return TRUE;
} /* Smooth */
