/* 
 * qmhide.h - Struct defn's for quadmesh hidden surface processing.
 * 
 * Copyright 1988
 * Center for Information Technology Integration (CITI)
 * Information Technology Division
 * University of Michigan
 * Ann Arbor, Michigan
 *
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the names of
 * CITI or THE UNIVERSITY OF MICHIGAN not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS." CITI AND THE UNIVERSITY OF
 * MICHIGAN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL CITI OR THE UNIVERSITY OF MICHIGAN BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */



/*****************************************************************
 * TAG( qm_facet )
 *
 * Data structure for a single quadmesh facet.  Contains information
 * necessary for hidden surface processing:
 * 	pln_eq:		Plane equation of this quad:
 * 			x*pln_eq[0] + y*pln_eq[1] + z*pln_eq[2] + pln_eq[3]
 * 	bbox:		Bounding box of the quad (e.g. [1][0] is ymin).
 *	arr:		Pointer to the array containing this facet.
 *	i, j:		Index into the quadmesh data structure.
 * 	n_behind:	Number of quads behind this one (up to MAX_BEHIND).
 * 	behind:		List of quads behind this one.  Used for
 * 			efficiency.  If there are actually more than
 * 			MAX_BEHIND, the extras are not included in the
 * 			list, but must be recomputed.
 * 	region:		Number of the region including this quad, if > 0.
 * 	rendered:	Set if this quad has been (or is being) rendered.
 * 	compared:	Set if this quad has already been compared to
 * 			all its neighbors.
 */

#define MAX_BEHIND	4

typedef struct _qm_facet {
    float	pln_eq[4];
    float	pln_min, pln_max;
    float	bbox[3][2];
    float	midpoint_z;
    struct _qm_facet_array * arr;
    int		i, j;
    int		n_behind;
    struct _qm_facet * behind[MAX_BEHIND];
    int		region;
    Bool	rendered:1;
    Bool	visible:1;	/* Is it in the viewport? */
} qm_facet;


/*****************************************************************
 * TAG( qm_facet_array )
 *
 * Basically an array of quadmesh facets.  Contains:
 * 	next:		Pointer to next in list for this window.
 * 	mesh:		Pointer to original pex quad mesh
 * 	m, n:		Size of mesh.
 * 	m_stride:	Stride in M direction.
 * 	n_stride:	Stride in N direction.
 * 	offset:		Offset to beginning of variable data.
 */

typedef struct _qm_facet_array {
    struct _qm_facet_array * next;
    pexQuadMeshPtr	mesh;
    pexColorSpecifier	color;
    CARD32		n, m;
    CARD32		m_stride;
    CARD32		n_stride;
    CARD32		offset;
    /* Variable data goes here */
} qm_facet_array;

/*****************************************************************
 * TAG( qmFacet qmFirst qmNextM qmNextN )
 *
 * Access functions:
 * 	qmFacet(pQM,m,n):	Return a pointer to the (m,n)th facet.
 * 	qmFirst(pQM)		Return a pointer to the first facet.
 * 	qmNextM(pQM,pQF)	Return a pointer to the next facet in
 * 				the same row.
 * 	qmNextN(pQM,pQF)	Return a pointer to the next facet in
 * 				the same column.
 * 				
 */
#define qmFacet(pQM,m,n)\
    ((qm_facet *)((char *)(pQM)+\
		  ((m) * ((pQM)->m_stride))+\
		  ((n) * ((pQM)->n_stride))+\
		  ((pQM)->offset)))


#define qmFirst(pQM)	((qm_facet *)((char *)(pQM) + (pQM)->offset))

#define qmNextM(pQM,pF)	((qm_facet *)((char *)(pF) + ((pQM)->m_stride)))
#define qmNextN(pQM,pF)	((qm_facet *)((char *)(pF) + ((pQM)->n_stride)))


/*****************************************************************
 * TAG( qm_square NSQUARE )
 *
 * Array of lists of facets, classified by position on screen.  Used
 * to make sorting more efficient.
 *
 * Use direct/indirect/double indirect scheme like Unix inodes.
 */

#define NSQUARE 20		/* Number of squares on a side */
#define NDIRECT 8		/* Number of direct facet pointers */
#define NINDIRECT 256		/* Number of facets in an indirect block */

typedef struct _qm_square {
    int		nfacet;
    qm_facet  * direct[NDIRECT];
    qm_facet ** indirect;
    qm_facet *** dbl_i;
} qm_square;

/*****************************************************************
 * TAG( facet_sqrs ins_sqr_facet get_sqr_facet qm_free_sqr )
 *
 * facet_sqrs - Return the range of squares covered by a facet.
 * ins_sqr_facet - Insert a facet into the squares it covers.  Return
 * 			FALSE if the facet is outside the window, TRUE
 * 			otherwise.
 * get_sqr_facet - Get the Nth facet from a square.
 * qm_free_sqr - Free all indirect or double indirect blocks allocated to
 * 		the square.  Also resets counts to 0.
 */

#ifdef __STDC__
extern Bool facet_sqrs( qm_facet *facet, int *minx_ret, int *maxx_ret,
			int *miny_ret, int *maxy_ret );

extern Bool ins_sqr_facet( qm_facet *facet );

extern qm_facet * get_sqr_facet( int x, int y, int n );
#else
extern Bool facet_sqrs();
extern Bool ins_sqr_facet();
extern qm_facet * get_sqr_facet();
#endif

extern void qm_free_sqr();
