/* 
 * qmhide3d.c - 3D display routine for hidden surface processing of quadmeshes.
 * 
 * Copyright 1988
 * Center for Information Technology Integration (CITI)
 * Information Technology Division
 * University of Michigan
 * Ann Arbor, Michigan
 *
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the names of
 * CITI or THE UNIVERSITY OF MICHIGAN not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS." CITI AND THE UNIVERSITY OF
 * MICHIGAN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL CITI OR THE UNIVERSITY OF MICHIGAN BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

#include "PEX.h"
#include "pubstr.h"
#include "PEXproto.h"
#include "quadmesh.h"
#include "renderer.h"
#include "qmhide.h"

/*****************************************************************
 * TAG( miPexQuadMesh3DHide )
 * 
 * 3D display routine for hidden surface processing of quadmeshes
 * Inputs:
 * 	pRD:	renderer to use
 * 	pQM:	Quad mesh to render.
 * Outputs:
 * 	Transforms Quad Mesh to 2d, adds it to hidden surface list
 * 	attached to deviceDep field of renderer.
 * Assumptions:
 * 	deviceDep field not used by other code.
 * Algorithm:
 *	[None]
 */

miPexQuadMesh3DHide(pRD, pQM)
pexRendererPtr pRD;
pexQuadMeshPtr pQM;
{
    pexQuadMeshPtr pQ2, PexTransformQuadmesh();

    pQ2 = PexTransformQuadmesh( pRD, pQM );

    if (!pQ2)
	return BadAlloc;

    qm_insert( pQ2, (qm_facet_array **)&pRD->deviceDep, pRD );
    return Success;
}


/*****************************************************************
 * TAG( PexTransformQuadmesh )
 * 
 * Transform a quadmesh to NPC using transforms in the given renderer.
 * Inputs:
 * 	pRD:	The renderer.
 * 	pQM:	The quadmesh.
 * Outputs:
 * 	Returns a quadmesh in NPC.  Most likely, the points will be Rational.
 * Assumptions:
 * 	shure.
 * Algorithm:
 * 	Create a new (Rational) quadmesh for output.
 * 	Map all the points in the quadmesh through the concatenation
 * 	of the local, global, orientation, and mapping transforms.
 * 	Map normals through the concatenation of the local & global
 * 	transforms (not strictly correct, but ok for now).
 * 	If there are no normals, add Facet normals.
 */
pexQuadMeshPtr
PexTransformQuadmesh(pRD, pQM)
pexRendererPtr pRD;
pexQuadMeshPtr pQM;
{
    pexQuadMeshPtr pQ2;
    pexCoord3D * ipt;
    pexCoord4D * opt;
    register int i, j;
    int mPts, nPts;
    CARD16 add_norms;
    pexArray1 a1, a2;

    mPts = pQM->mPoints;
    nPts = pQM->nPoints;

    add_norms = ((pQM->vertexAttributes | pQM->facetAttributes) & GANormal) ?
	0 : GANormal;

    pQ2 = NewPexQuadMesh( mPts, nPts, Rational,
			  pQM->facetAttributes | add_norms,
			  pQM->vertexAttributes );
    if (!pQ2)
	return NULL;

    InitPexQuadMesh( pQ2, mPts, nPts, Rational, pQM->colorType,
		     pQM->facetAttributes | add_norms, pQM->vertexAttributes );

    /* Transform the points */
    a1.n = mPts * nPts;
    a1.body = (char *)PexQuadMeshVertex(pQM, 0, 0);
    a1.stride = pQM->vertexNStride;
    a2.n = a1.n;
    a2.body = (char *)PexQuadMeshVertex(pQ2, 0, 0);
    a2.stride = pQ2->vertexNStride;
    PexTransformPoints(pRD, &a1, &a2);

    /* If we had to create facet normals, compute them */
    if ( add_norms )
    {
	pexCoord3D corners[2][2];
	pexVector3D diag[2], * nrml;

	for ( i = 0; i < mPts - 1; i++ )
	{
	    corners[0][1] = *PexQuadMeshVertex(pQM, i, 0);
	    corners[1][1] = *PexQuadMeshVertex(pQM, i+1, 0);
	    for ( j = 0; j < nPts - 1; j++ )
	    {
		corners[0][0] = corners[0][1];
		corners[1][0] = corners[1][1];
		corners[0][1] = *PexQuadMeshVertex(pQM, i, j+1);
		corners[1][1] = *PexQuadMeshVertex(pQM, i+1, j+1);
		diag[0].x = corners[1][1].x - corners[0][0].x;
		diag[0].y = corners[1][1].y - corners[0][0].y;
		diag[0].z = corners[1][1].z - corners[0][0].z;
		diag[1].x = corners[0][1].x - corners[1][0].x;
		diag[1].y = corners[0][1].y - corners[1][0].y;
		diag[1].z = corners[0][1].z - corners[1][0].z;
		nrml = PexQuadMeshFacetNorm(pQ2, i, j);
		nrml->x = diag[0].y * diag[1].z - diag[0].z * diag[1].y;
		nrml->y = diag[0].z * diag[1].x - diag[0].x * diag[1].z;
		nrml->z = diag[0].x * diag[1].y - diag[0].y * diag[1].x;
	    }
	}
	a1.n = (mPts - 1) * (nPts - 1);
	a1.body = (char *)PexQuadMeshFacetNorm(pQ2, 0, 0);
	a1.stride = pQ2->facetNormNStride;
	PexTransformNorms(pRD, &a1, &a1);
    }

    /* If facet normals are present, transform them */
    if ( pQM->facetAttributes & GANormal )
    {
	a1.n = (mPts - 1) * (nPts - 1);
	a1.body = (char *)PexQuadMeshFacetNorm(pQM, 0, 0);
	a1.stride = pQM->facetNormNStride;
	a2.n = (mPts - 1) * (nPts - 1);
	a2.body = (char *)PexQuadMeshFacetNorm(pQ2, 0, 0);
	a2.stride = pQ2->facetNormNStride;
	PexTransformNorms(pRD, &a1, &a2);
    }

    /* If vertex normals are present, transform them */
    if ( pQM->vertexAttributes & GANormal )
    {
	a1.n = mPts * nPts;
	a1.body = (char *)PexQuadMeshVertexNorm(pQM, 0, 0);
	a1.stride = pQM->vertexNormNStride;
	a2.n = mPts * nPts;
	a2.body = (char *)PexQuadMeshVertexNorm(pQ2, 0, 0);
	a2.stride = pQ2->vertexNormNStride;
	PexTransformNorms(pRD, &a1, &a2);
    }

    return pQ2;
}

/*****************************************************************
 * TAG( miPexKillQMList )
 * 
 * Kill the Quadmesh hidden surface list from a renderer.
 * Inputs:
 *	[None]
 * Outputs:
 *	[None]
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */
void
miPexKillQMList(pRD)
pexRendererPtr  pRD;
{
    qm_facet_array * pqa, * pqa1;

    for ( pqa = (qm_facet_array *)pRD->deviceDep; pqa; pqa = pqa1 )
    {
	pqa1 = pqa->next;
	FreePexQuadMesh(pqa->mesh);
	xfree(pqa);
    }
    pRD->deviceDep = NULL;
    qm_free_sqr();
}

/*****************************************************************
 * TAG( miPexFlushQMList )
 * 
 * Flush the list of quad meshes that are to be hidden surface processed.
 * Inputs:
 * 	pRD:	renderer pointer.  List of quad meshes is attached to
 * 		deviceDep field.
 * Outputs:
 * 	Flushes them to the output drawable and frees the list.
 * Assumptions:
 * 	renderer deviceDep field is indeed a pointer to a list of quad meshes.
 * Algorithm:
 *	[None]
 */
miPexFlushQMList(pRD)
pexRendererPtr pRD;
{
    if ( pRD->deviceDep )
    {
	qmhide((qm_facet_array *)pRD->deviceDep, pRD);
	miPexKillQMList(pRD);
    }

    return Success;
}

