/* 
 * nurbcurve.c - Routine to unpack and allocate a Nurb Curve object.
 * 
 * Copyright 1988
 * Center for Information Technology Integration (CITI)
 * Information Technology Division
 * University of Michigan
 * Ann Arbor, Michigan
 *
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the names of
 * CITI or THE UNIVERSITY OF MICHIGAN not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS." CITI AND THE UNIVERSITY OF
 * MICHIGAN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL CITI OR THE UNIVERSITY OF MICHIGAN BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/*
 * packet is assumed to be byte-swapped properly
 */
#include "PEX.h"
#include "swapmacros.h"
#include "nurb.h"


#define ExtraSpace(np, nk, ct)\
    (((np) * ((ct) == Rational ? sizeof(pexCoord4D) : sizeof(pexCoord3D))) + \
     ((nk) * sizeof(FLOAT)))

/*****************************************************************
 * TAG( NewPexNurbCurve )
 * 
 * allocates space for a Nurb Curve.
 * Inputs:
 * 	numPoints - Number of control points in the Nurb.
 * 	numKnots  - Number of Knots in the Nurb.
 * 		note: the difference of these two is the curve order.
 * 	coordType - specifies of the curve is Rational or not.
 * 		this effects the size of the control Points. (3D or 4D) 
 * Outputs:
 * 	A pointer to space big enough for that nurb curve.
 * Assumptions:
 * 	None.
 * Algorithm:
 * 	spit and grind.
 */
pexNurbCurvePtr 
NewPexNurbCurve(numPoints, numKnots, coordType)
    CARD32	 numPoints;
    CARD32	 numKnots;
    pexCoordType coordType;
    
{
    register pexNurbCurvePtr pNurb;

    pNurb = (pexNurbCurvePtr)	xalloc(sizeof(pexNurbCurveRec) +
				       ExtraSpace(numPoints,
						  numKnots,
						  coordType));

    pNurb->head.elementType = OCNurbCurve;
    pNurb->head.length = (CARD16)(sizeof(pexNurbCurveRec) +
				    ExtraSpace(numPoints,
					       numKnots,
					       coordType))>>2;
    return pNurb;
};

#define SWAP_FILE "nurbcurve.ci"
#include "scgen.ci"
#undef SWAP_FILE

/*****************************************************************
 * TAG( ErrorCheckPexNurbCurve )
 * 
 * This function checks a pexNurbCurvePtr for errors in composition
 * Inputs:
 * 	a pointer to a pexNurbCurveRec
 * Outputs:
 * 	bool true if an error exists.
 * Assumptions:
 * 	This function assumes that the pexNurbCurveRec has been formatted with
 *   UnpackPexNurbCurve.
 *
 * Algorithm:
 * 	for each constraint, we check.
 */
Bool
ErrorCheckPexNurbCurve(Nurb)
    pexNurbCurvePtr Nurb;
{
    FLOAT CurrentKnotValue, LastKnotValue;
    CARD32 CurrentKnot;
	
    /* numKnots must be == numPoints + spline order */
    if (Nurb->numKnots != (Nurb->numPoints + Nurb->curveOrder))
	return TRUE;

    /* numPoints must not be less that the order */
    if (Nurb->numPoints < Nurb->curveOrder)
	return TRUE;

    /* tmin must be less that tmax */
    if (Nurb->tMin >= Nurb->tMax)
	return TRUE;

    /* tmin must be less than or equal to the order'th knot value */
    if (Nurb->tMin >= *PexNurbCurveKnot(Nurb, Nurb->curveOrder))
	return TRUE;

    /* tmax must be greater than or equal to the (numKnots+1-order)'th knot */
    if (Nurb->tMax <= *PexNurbCurveKnot
	(Nurb, (Nurb->numKnots + 1 - Nurb->curveOrder)))
	return TRUE;

    /* the knot values must be a non-decreasing sequence */
    for(CurrentKnot = 0;CurrentKnot < Nurb->numKnots; CurrentKnot++)
    {
	CurrentKnotValue = *PexNurbCurveKnot(Nurb, CurrentKnot);
	if (CurrentKnot && (CurrentKnotValue > LastKnotValue))
	    return TRUE;
	LastKnotValue = CurrentKnotValue;
    };

    /* the data passed the test! */
    return FALSE;
};

void
FreePexNurbCurve(Nurb)
    pexNurbCurvePtr Nurb;
{
    xfree(Nurb);
};

