/* 
 * quadmesh.c - Object handling routines for a swapped quad mesh OC packet.
 * 
 * Copyright 1988
 * Center for Information Technology Integration (CITI)
 * Information Technology Division
 * University of Michigan
 * Ann Arbor, Michigan
 *
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the names of
 * CITI or THE UNIVERSITY OF MICHIGAN not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS." CITI AND THE UNIVERSITY OF
 * MICHIGAN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL CITI OR THE UNIVERSITY OF MICHIGAN BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

#include "PEX.h"
#include "PEXproto.h"
#include "swapmacros.h"
#include "quadmesh.h"


/*****************************************************************
 * TAG( NewPexQuadMesh )
 * 
 * allocate space for a quadrilateral mesh
 * 
 * Inputs:
 * 	mPts, nPts - dimensions of the quad mesh
 * 	pType - Type of points (Rational or NonRational)
 * 	facetAtt, vertexAtt - attributes present for the facets and vertices
 * Outputs:
 * 	a ptr to the required space
 * Assumptions:
 * 	WARNING: This code does not implement colors per facet or colors per
 * 	vertex. When adding these features, some code may need to be changed
 * 	(in particular, the ExtraSpace macro).
 * Algorithm:
 *	[None]
 */


#define ExtraSpace(m, n, pt, fa, va)\
    ((((pt) == Rational ? sizeof(pexCoord4D) : sizeof(pexCoord3D)) * m * n) +\
     ((fa == GANormal) ? (sizeof (pexVector3D) * (m-1) * (n-1)) : 0) +\
     ((va == GANormal) ? (sizeof (pexVector3D) * m * n) : 0))

pexQuadMeshPtr
NewPexQuadMesh (mPts, nPts, pType, facetAtt, vertexAtt)
    CARD32 mPts, nPts;
    pexCoordType pType;
    CARD16 facetAtt, vertexAtt;

{
    pexQuadMeshPtr NewArea;

    /* printf("Newquadmesh %dx%d points, ", mPts, nPts); */
    
    NewArea = (pexQuadMeshPtr)
	malloc(sizeof(pexQuadMeshRec) +
	       (ExtraSpace(mPts, nPts, pType, facetAtt, vertexAtt)));
    if (NewArea)
    {
	NewArea->head.length = (sizeof(pexQuadMeshRec) +
				ExtraSpace(mPts, nPts, pType,
					   facetAtt, vertexAtt) ) >> 2;
	NewArea->head.elementType = OCQuadrilateralMesh;
    }
    return (NewArea);
};

/*****************************************************************
 * TAG( InitPexQuadMesh  )
 * 
 * Initialize all the stride and offset fields of a quadmesh structure.
 * Inputs:
 * 	pq:		Pointer to the quadmesh.
 * 	mPts, nPts:	Size of the quadmesh.
 * 	pType:		Rational or NonRational -- type of points.
 * 	cType:		Indexed or Direct -- type of colors.
 * 	facetAtt:	Facet attribute bitmask
 * 	vertexAtt:	Vertex attribute bitmask
 * Outputs:
 * 	Fills in structure fields, including strides & offsets
 * 	according to parameter values.
 * Assumptions:
 * 	pq is allocated the right size.
 * Algorithm:
 * 	trivial.
 */
void
InitPexQuadMesh (pq, mPts, nPts, pType, cType, facetAtt, vertexAtt)
pexQuadMeshPtr pq;
CARD32 mPts, nPts;
pexCoordType pType;
pexColorType cType;
CARD16 facetAtt, vertexAtt;
{
    register int HaveFacetNormals, HaveVertexNormals;
    int HaveFacetColors, HaveVertexColors, pOffset;
    CARD32 numVertices, numFacets;

    numVertices = mPts*nPts;
    numFacets = (mPts-1)*(nPts-1);
    
#define colorSize 0		/* for now */
    HaveVertexColors = ((vertexAtt & GAColor) ? numVertices*colorSize : 0);
    HaveFacetColors = ((facetAtt & GAColor) ? colorSize : 0);
    HaveVertexNormals =((vertexAtt & GANormal) ?
			numVertices * sizeof(pexVector3D) : 0);
    HaveFacetNormals = ((facetAtt & GANormal) ?
			numFacets * sizeof(pexVector3D) : 0); 
    pOffset = pType == Rational ? sizeof(pexCoord4D) : sizeof(pexCoord3D);

    pq->facetAttributes = facetAtt;
    pq->vertexAttributes = vertexAtt;
    pq->mPoints = mPts;
    pq->nPoints = nPts;
    pq->pointType = pType;	/* always unpacking 3d coords */
    pq->colorType = cType;

#define	YourBasicOffset sizeof(pexQuadMeshRec)

    pq->vertexOffset = YourBasicOffset +
	HaveFacetNormals + HaveFacetColors;

    pq->vertexNormOffset = pq->vertexOffset + pOffset;
	    
    pq->vertexColorOffset = pq->vertexNormOffset + 
	((HaveVertexNormals) ? sizeof (pexVector3D) : 0);

    pq->vertexColorNStride =
	pq->vertexNormNStride =
	    pq->vertexNStride = pOffset +
		((HaveVertexNormals ? sizeof(pexVector3D) : 0) +
		 (HaveVertexColors ? colorSize : 0));
    pq->vertexColorMStride =
	pq->vertexNormMStride =
	    pq->vertexMStride = nPts * (pq->vertexNStride);

    pq->facetNormOffset = YourBasicOffset;

    pq->facetColorOffset = YourBasicOffset +
	(HaveFacetNormals ? sizeof(pexVector3D) : 0);

    pq->facetColorNStride =
	pq->facetNormNStride =
	    ((HaveFacetNormals ? sizeof (pexVector3D) : 0) +
	     (HaveFacetColors ? colorSize : 0));
    pq->facetColorMStride =
	pq->facetNormMStride = (nPts - 1) * pq->facetNormNStride;
}


/* Define UnpackPexQuadMesh functions */

#define SWAP_FILE "quadmesh.ci"
#include "scgen.ci"
#undef SWAP_FILE


/*****************************************************************
 * TAG( FreePexQuadMesh )
 * 
 * Inputs:
 *	pexQuadMeshPtr
 * Outputs:
 *	[None]
 * Assumptions:
 *	Area is byteSwapped properly.
 * Algorithm:
 *	[None]
 */

void
FreePexQuadMesh (pArea)
    pexQuadMeshPtr pArea;
{
    free(pArea);
};


/*****************************************************************
 * TAG( PrintPexQuadMesh )
 * 
 * Print a quadmesh to stdout
 * Inputs:
 * 	pq:	Pointer to the quad mesh.
 * Outputs:
 * 	Prints it.
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */
PrintPexQuadMesh(pq)
    pexQuadMeshPtr pq;
{
    char * comma = "";
    int i, j;
    pexVector3D * pv;
    pexCoord3D * pp;

    printf("Quad mesh at 0x%x, size %d by %d has", (int)pq,
	   pq->mPoints, pq->nPoints);
    if ( pq->vertexAttributes || pq->facetAttributes )
    {
	if ( pq->pointType == Rational )
	{
	    printf( "%s Rational coordinates", comma );
	    comma = ",";
	}
	if ( pq->facetAttributes & GANormal )
	{
	    printf( "%s facet normals", comma );
	    comma = ",";
	}
	if ( pq->facetAttributes & GAColor )
	{
	    printf( "%s facet colors", comma );
	    comma = ",";
	}
	if ( pq->vertexAttributes & GANormal )
	{
	    printf( "%s vertex normals", comma );
	    comma = ",";
	}
	if ( pq->vertexAttributes & GAColor )
	    printf( "%s vertex colors", comma );
	printf ( ".\n" );
    }
    else
	printf( " no attributes.\n" );

    if ( pq->facetAttributes )
    {
	for ( i = 0; i < pq->mPoints - 1; i++ )
	    for ( j = 0; j < pq->nPoints - 1; j++ )
	    {
		printf( "Facet (%3d,%3d)", i, j );
		if ( pq->facetAttributes & GANormal )
		{
		    pv = PexQuadMeshFacetNorm(pq, i, j);
		    printf( "\tNormal [%g %g %g]", pv->x, pv->y, pv->z );
		}
		if ( pq->facetAttributes & GAColor )
		{
		    /* nothing for now */
		}
		printf("\n");
	    }
    }

    for ( i = 0; i < pq->mPoints; i++ )
	for ( j = 0; j < pq->nPoints; j++ )
	{
	    printf( "Vertex (%3d,%3d)", i, j );
	    pp = PexQuadMeshVertex(pq, i , j);
	    if ( pq->pointType == Rational )
		printf( " [%g %g %g %g]", pp->x, pp->y, pp->z,
			((pexCoord4D *)pp)->w );
	    else
		printf( " [%g %g %g]", pp->x, pp->y, pp->z );
	    if ( pq->vertexAttributes & GANormal )
	    {
		pv = PexQuadMeshVertexNorm(pq, i, j);
		printf( "\tNormal [%g %g %g]", pv->x, pv->y, pv->z );
	    }
	    if ( pq->vertexAttributes & GAColor )
	    {
		/* Nothing */
	    }
	    printf("\n");
	}
}
