/* 
 * renderer.c - renderer object hiding functions
 * 
 * Copyright 1988
 * Center for Information Technology Integration (CITI)
 * Information Technology Division
 * University of Michigan
 * Ann Arbor, Michigan
 *
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the names of
 * CITI or THE UNIVERSITY OF MICHIGAN not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS." CITI AND THE UNIVERSITY OF
 * MICHIGAN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL CITI OR THE UNIVERSITY OF MICHIGAN BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

#include <gcstruct.h>
#include <PEX.h>
#include <PEXproto.h>
#include <renderer.h>
#include <pubstr.h>
#include <pexextensn.h>
#include <swapmacros.h>

/*****************************************************************
 * TAG( NewPexRenderer )
 * 
 * This function allocates space for a new renderer.
 * 
 * Inputs:
 * 	None; the renderer's size is static.
 * Outputs:
 * 	A pointer to the new renderer space.
 * Assumptions:
 * 	...
 * Algorithm:
 * 	...
 */

pexRendererPtr 
NewPexRenderer ()
{
    return (pexRendererPtr)Xalloc(sizeof(pexRendererRec));
};

/*****************************************************************
 * TAG( MakePexRenderer )
 * 
 * This function takes parameters which correspond to fields in the
 * renderer structure, gets space for a new structure, fills the
 * new info in, and returns a pointer to the new renderer.
 *
 * Right now this is not a full implementation. There are no pars
 * for the procedure vector. I doubt a function with that many pars
 * would have much value.  If it turns out I'm wrong, I'll do the
 * full implementation later.
 * 
 * Inputs:
 * 	(see below)
 * Outputs:
 * 	a pointer to the new renderer
 * Assumptions:
 * 	This currently doesn't work. It directly copies the pointers.
 * 	That's not good at all. I'll work on this more later.
 * Algorithm:
 * 	...
 */

pexRendererPtr 
MakePexRenderer (rID, pcID, pPC, rS,
		 viewT, clipM, colorT, depthCueT, lightT, colorApproxT,
		 patternT, hlhsrM, npcS, vP, drawID, pDraw, pGC, devDep)
    pexRenderer        rID;
    pexPC              pcID;
    pexPipelineContextPtr     pPC;
    CARD32             rS;
    pexViewEntry       viewT[];
    pexMatrix	       clipM[];
    pexRgbFloatColor   colorT[];
    pexLookupTable     depthCueT;
    pexLightEntry      lightT[];
    pexLookupTable     colorApproxT;
    pexLookupTable     patternT;
    pexNpcSubvolume    npcS;
    pexViewport        vP;
    Drawable           drawID;
    DrawablePtr        pDraw;
    GCPtr              pGC;
    CARD32             devDep;
{
    register CARD32 i;
    pexRendererPtr pRend;

    if (pRend = NewPexRenderer())
    {
	pRend->rID = rID;
	pRend->pcID = pcID;
	pRend->pPC = pPC;
	pRend->rendererState = rS;
	for (i=0; i<NUM_VIEWS; i++)
	{
	    pRend->viewTable[i] = viewT[i];
	    bcopy( clipM[i], pRend->clipMatrix[i], sizeof (pexMatrix));
	}
	for (i=0; i<NumColors; i++)
	    pRend->colorTable[i] = colorT[i];
	pRend->depthCueTable = depthCueT;
	for (i=0; i<NUM_LIGHT_SOURCES; i++)
	    pRend->lightTable[i] = lightT[i];
	pRend->colorApproxTable = colorApproxT;
	pRend->patternTable = patternT;
	pRend->HLHSRMode = hlhsrM;
	pRend->npcSubvolume = npcS;
	pRend->viewport = vP;
	pRend->drawID = drawID;
	pRend->pDraw = pDraw;
	pRend->pGC = pGC;
	pRend->deviceDep = devDep;

	pRend->Polygon3D = NULL;
	pRend->Polygon2DFlat = NULL;
	pRend->Polygon2DSmoothh = NULL;
	pRend->Polyline3D = NULL;
	pRend->Polyline2D = NULL;
	pRend->Polyline2DZ = NULL;
	pRend->AnnotationText3D = NULL;
	pRend->AnnotationText2D = NULL;
	pRend->QuadMesh3D = NULL;
	pRend->QuadMesh2DFlat = NULL;
	pRend->QuadMesh2DSmooth = NULL;
	pRend->NurbSurface = NULL;
	pRend->NurbCurve = NULL;
	pRend->Pixel8 = NULL;
	pRend->PixelZ8 = NULL;
	pRend->PixelZRGB = NULL;
	pRend->BeginStructure = NULL;
	pRend->EndStructure = NULL;
	pRend->LocalTransform3D = NULL;
	pRend->GlobalTransform3D = NULL;
	pRend->BeginRenderOutput = NULL;
	pRend->EndRenderOutput = NULL;
    }
    return (pRend);
};
                                                    
/*****************************************************************
 * TAG( GetDefaultRenderer )
 * 
 * Supplies a renderer with default values filled in.
 *
 * Inputs:
 * 	None.
 * Outputs:
 * 	A pointer to a pex renderer, with default values supplied. The
 * 	procedure vector is not filled in (see assumptions).
 * Assumptions:
 * 	I assume the output will eventually go through the validate function.
 * 	It is up to validate to supply default values for the procedure
 * 	vector if the hardware platform is unknown.
 * Algorithm:
 * 	...
 */
pexRendererPtr
GetDefaultRenderer()
{
    pexViewEntry viewTable[NUM_VIEWS];
    pexRgbFloatColor colorTable[NumColors];
    pexLightEntry lightTable[NUM_LIGHT_SOURCES];
    pexMatrix clippingMatrices[NUM_VIEWS];
    pexViewport viewport;
    pexNpcSubvolume npcSubvol;
    
    /* Initialize */
    InitPexViewTable (viewTable);
    InitPexColorTable (colorTable);
    InitPexClippingMatrices (clippingMatrices);
    InitPexLightTable (lightTable);
    
    npcSubvol.minval.x = 0.0;
    npcSubvol.minval.y = 0.0;
    npcSubvol.minval.z = 0.0;
    npcSubvol.maxval.x = 1.0;
    npcSubvol.maxval.y = 1.0;
    npcSubvol.maxval.z = 1.0;

    viewport.minval.x = 0;    /* since useDrawable defaults to true, */
    viewport.minval.y = 0;    /* x and y values are meaningless */
    viewport.minval.z = 1.0;
    viewport.maxval.x = 0;
    viewport.maxval.y = 0;
    viewport.maxval.z = 0.0;
    viewport.useDrawable = TRUE;

    return (MakePexRenderer (0, 0, GetDefaultPC(), Idle,
			     viewTable, clippingMatrices, colorTable, NULL,
			     lightTable, NULL, NULL, 1, npcSubvol,
			     viewport, 0, NULL, NULL, NULL));
}


/*****************************************************************
 * TAG( FreePexRenderer )
 * 
 * Frees the renderer pointed to in the parameter.
 * 
 * Inputs:
 * 	a pointer to a renderer which should be disposed of.
 * Outputs:
 * 	void
 * Assumptions:
 * 	...
 * Algorithm:
 *      ...
 */

void 
FreePexRenderer (pRend)
    pexRendererPtr pRend;
{
    pexPipelineContextPtr pc;
    /*
     * first get rid of the pipeline context (stack) and the graphics context.
     */
    if ( !pRend )
	return;

    /*
     * do an implicit EndRendering here if necessary.
     */
    if (pRend->rendererState == Rendering)
        pexEndRenderingVAL (pRend);

    DDPEXFreeRenderer(pRend);	

    while ( pRend->pPC )
    {
	pc = pRend->pPC;
	pRend->pPC = pRend->pPC->OldPC;
	Xfree (pc);
    }
    if ( pRend->pGC )
    {
	pRend->pGC->tile = (PixmapPtr)NULL;
	FreeGC (pRend->pGC);
    }

    Xfree (pRend);
};


/*****************************************************************
 * TAG( CopyPexRenderer )
 * 
 * Copy one renderer to another.
 * 
 * Inputs:
 * 	pointer to an original renderer.
 * Outputs:
 * 	pointer to a new renderer, a duplicate of the first.
 * Assumptions:
 * 	...
 * Problems:
 * 	No way will this work right in its present state.
 */

pexRendererPtr 
CopyPexRenderer (origR)
    pexRendererPtr origR;
{
    pexRendererPtr pRend;

    pRend = NewPexRenderer ();
    *pRend = *origR;
    return (pRend);
};


#define SWAP_FILE "renderer.ci"
#include "scgen.ci"

