/* 
 * looktables.c - Initializes renderer tables
 * 
 * Copyright 1988
 * Center for Information Technology Integration (CITI)
 * Information Technology Division
 * University of Michigan
 * Ann Arbor, Michigan
 *
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the names of
 * CITI or THE UNIVERSITY OF MICHIGAN not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS." CITI AND THE UNIVERSITY OF
 * MICHIGAN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL CITI OR THE UNIVERSITY OF MICHIGAN BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

#include <math.h>
#include "PEXproto.h"
#include "PEX.h"
#include "renderer.h"


#include "X.h"
#include "pixmap.h"
#include "misc.h"
#include "dix.h"
#include "dixstruct.h"
#include "resource.h"


#define  ErrorCheck    0.00000001

/*****************************************************************
 * TAG( PexUnitVector )
 * 
 *   normalize a 3D vector
 * Inputs:
 *      *pexVector3D
 *      *pexVector3D
 * Outputs:
 *      returns normalized vector in second parameter
 *      returns vector (0,0,0) if the vector has no length
 * Assumptions:
 *      [None]
 * Algorithm:
 *      [None]
 */

void
static localunitvector( invector, outvector )
    pexVector3D  *invector, *outvector;
{
    FLOAT   length;

    length = sqrt(  (invector->x * invector->x)
                  + (invector->y * invector->y)
                  + (invector->z * invector->z) );

    if ( (length < ErrorCheck) &&
         (length > -ErrorCheck) )
    {
        outvector->x = outvector->y = outvector->z = 0.0;
    }
    else
    {
        outvector->x = invector->x / length;
        outvector->y = invector->y / length;
        outvector->z = invector->z / length;
    }
}

/*****************************************************************
 * TAG( InitPexViewTable )
 * 
 * Initializes view table
 * 
 * Inputs:
 * 	a pexViewRep element
 * Outputs:
 * 	above view rep has been initialized
 *      this func always succeeds
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */

void
InitPexViewTable(viewTable)
    pexViewEntry viewTable[NUM_VIEWS];
{
    register int i;

    for (i=0; i<NUM_VIEWS; i++)
    {
        viewTable[i].clipFlags = PEXClippingOn;

	identm44( viewTable[i].orientation );
	identm44( viewTable[i].mapping );

	viewTable[i].clipLimits.minval.x = 0.0;
	viewTable[i].clipLimits.minval.y = 0.0;
	viewTable[i].clipLimits.minval.z = 0.0;

	viewTable[i].clipLimits.maxval.x = 1.0;
	viewTable[i].clipLimits.maxval.y = 1.0;
	viewTable[i].clipLimits.maxval.z = 1.0;

    }
}


/*****************************************************************
 * TAG( InitPexClippingMatrices )
 * 
 * Initializes clipping matrices
 * 
 * Inputs:
 * 	an array of pexMatrix.
 * Outputs:
 * 	[None]
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */

void
InitPexClippingMatrices (clippingMatrices)
    pexMatrix clippingMatrices[NUM_VIEWS];
{
    int i;
    
    for (i=0; i<NUM_VIEWS; i++)
    {
	/* Set the default clipmatrix. This matrix transforms the NPC subvolume
	 * to the -1->1, -1->1, 0->1 clipping cube.
	 */
	identm44(clippingMatrices[i]);
	clippingMatrices[i][0][0] = 2.0;
	clippingMatrices[i][1][1] = 2.0;
	clippingMatrices[i][2][2] = -1.0;
	clippingMatrices[i][3][0] = -1.0;
	clippingMatrices[i][3][1] = -1.0;
	clippingMatrices[i][3][2] = 0.0;
    }
}

/*****************************************************************
 * TAG( InitPexColorTable )
 * 
 * Initializes color table
 * 
 * Inputs:
 * 	a pexRgbFloatColor element
 * Outputs:
 * 	above color has been initialized
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */

void
InitPexColorTable(colorTable)
    pexRgbFloatColor colorTable[NumColors];
{
    register int i;

    for (i=0; i<NumColors; i++)
    {
	colorTable[i].red = i/NumColors;
	colorTable[i].green = i/NumColors;
	colorTable[i].blue = i/NumColors;
    }
}


/*****************************************************************
 * TAG( InitPexLightTable )
 * 
 * Initializes light table
 * 
 * Inputs:
 * 	a pexLightEntry element
 * Outputs:
 * 	above light has been initialized
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */

void
InitPexLightTable(lightTable)
    pexLightEntry lightTable[NUM_LIGHT_SOURCES];
{
    register int i;

    for (i=0; i<NUM_LIGHT_SOURCES; i++)
    {
	lightTable[i].lightType = LightAmbient;
	lightTable[i].pad = 0;
	/*
	 * total kluge to get the second light (there are two) to be different
	 * that the others.
	 */
	lightTable[i].direction.x = 1.0 - (2 * (FLOAT)i);
	lightTable[i].direction.y = 1.0 - (FLOAT)(.5 * (FLOAT)i);
	lightTable[i].direction.z = 1.0 + (FLOAT)i;
	lightTable[i].point.x = 0.0;
	lightTable[i].point.y = 0.0;
	lightTable[i].point.y = 0.0;

	localunitvector(&(lightTable[i].direction),
			&(lightTable[i].direction));

	lightTable[i].concentration = 0;
	lightTable[i].spreadAngle = 0;
	lightTable[i].attenuation1 = 0;
	lightTable[i].attenuation2 = 0;
	
	lightTable[i].lightColor.colorType = Indexed;
	lightTable[i].lightColor.pad = 0;
	lightTable[i].lightColor.color.format.indexed.index = 1;
	lightTable[i].lightColor.color.format.indexed.pad = 0;
    }
}
