/* 
 * mipolyclip.c - Code to do reentrant polyline clipping
 * 
 * Copyright 1988
 * Center for Information Technology Integration (CITI)
 * Information Technology Division
 * University of Michigan
 * Ann Arbor, Michigan
 *
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the names of
 * CITI or THE UNIVERSITY OF MICHIGAN not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS." CITI AND THE UNIVERSITY OF
 * MICHIGAN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL CITI OR THE UNIVERSITY OF MICHIGAN BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */
#include    <stdio.h>
#include    "polyclip.h"


/*------------------------------------------------------------------------+
| Global storage associated with polygon clipping routines.		 |
+------------------------------------------------------------------------*/

static	int	    out_count;	  /* Storage for number of vertices clipped. */
static	pexCoord4D  *out_ptr;      /* Next clipped vertex to store. */
static	int    vertices_left; /* Storage for vertices left in out_vertices. */

/*
 * planes0-5 = six values for clipping planes (xl, xr, yb, yt, zh, zy)
 * plane_cbs = global storage for planes to clip to and shit. 
 */
static	pexPolyclip_Cb plane_cbs[6] =
{
{0.,0.,0.,0.},1,0,-1.0,{0.,0.,0.,0.},	   /* x left (min) */
{0.,0.,0.,0.},1,0, 1.0,{0.,0.,0.,0.},	   /* x right (max) */
{0.,0.,0.,0.},1,0,-1.0,{0.,0.,0.,0.},	   /* y bottom (min) */
{0.,0.,0.,0.},1,0, 1.0,{0.,0.,0.,0.},	   /* y top (max) */
{0.,0.,0.,0.},1,0, 0.0,{0.,0.,0.,0.},	   /* z hither (min) */
{0.,0.,0.,0.},1,0, 1.0,{0.,0.,0.,0.}	   /* z yon (max) */
};


/******************************************************************************
* +------------------------------------------------------------------------+ *
* | "pexpolyclip_out"							   | *
* |									   | *
* |    Purpose:								   | *
* |	   Store away a clipped vertex.					   | *
* |									   | *
* |    Parameters:							   | *
* |	   cb	   -> clipping plane information			   | *
* |	   current -> polygon vertex to be stored			   | *
* |									   | *
* |    Globals:								   | *
* |	   out_count	 =  count of vertices placed in out_vertices	   | *
* |	   out_ptr	 =  cursor to run across out_vertices when storing vertices| *
* |	   vertices_left =  global storage for down counter for max_out_count| *
* |									   | *
* +------------------------------------------------------------------------+ *
*****************************************************************************/

pexpolyclip_out(cb, current)
   pexPolyclip_Cb	   *cb;
   pexCoord4D	   *current;
{
   extern  int	       out_count; /* Storage for number of vertices clipped. */
   extern  pexCoord4D  *out_ptr; /* Next clipped vertex to store. */
   extern  int	       vertices_left; /* Storage for vertices left in out_vertices. */

   if (--vertices_left >= 0)
   {
       out_ptr->x = current->x;
       out_ptr->y = current->y;
       out_ptr->z = current->z;
       out_ptr->w = current->w;

       out_ptr++;
       out_count++;
   }
}


/******************************************************************************
* +------------------------------------------------------------------------+ *
* | "pexpolyclip_xl"							   | *
* |									   | *
* |    Purpose:								   | *
* |	   Clip a polygon vertex to the left x clipping plane.		   | *
* |									   | *
* |    Parameters:							   | *
* |	   cb	   -> clipping plane information for left % clipping plane | *
* |	   current -> polygon vertex to be processed			   | *
* |									   | *
* +------------------------------------------------------------------------+ *
*****************************************************************************/

pexpolyclip_xl(cb, current)
   pexPolyclip_Cb	   *cb;
   pexCoord4D	   *current;
{
   FLOAT	   alpha;	   /* Ratio of line on each */
				   /*	side of clipping plane. */
   FLOAT	   delta;	   /* Distance in % direction between */
				   /*	current point and clip plane. */
   pexCoord4D	   intersection;   /* Storage for point which is */
				   /*	intersection of edge and plane. */


   /*------------------------------------------------------------------------+
    | Processing  first	 point. Save its value in previous and first storage |
    | locations of cb. Clear first point flag.				     |
    +------------------------------------------------------------------------*/

   if (cb->first_vertex)
   {
       cb->first.x = current->x;
       cb->previous.x = current->x;
       cb->first.y = current->y;
       cb->previous.y = current->y;
       cb->first.z = current->z;
       cb->previous.z = current->z;
       cb->first.w = current->w;
       cb->previous.w = current->w;

       cb->first_vertex = 0;
   }

   /*------------------------------------------------------------------------+
    | This is not the  first  point  of	 the  polygon,	and  the  edge	from |
    | previous	to  current  does  not	cross the clipping boundary. So just |
    | save the current point as the \ previous point.			     |
    +------------------------------------------------------------------------*/

   else if ( ( (current->x - (current->w * cb->plane)) *
	  (cb->previous.x - (cb->previous.w * cb->plane)) )
	  >= 0.0 )
   {
       cb->previous.x = current->x;
       cb->previous.y = current->y;
       cb->previous.z = current->z;
       cb->previous.w = current->w;
   }

   /*------------------------------------------------------------------------+
    | This is not the first point, and the edge	 from  current	to  previous |
    | crosses  the  clipping  plane.  So  calculate  the  intersection point |
    | between the edge and the plane and output	 it  to	 the  next  clipping |
    | stage.  Then  set	 the  previous	vertex	to the values of the current |
    | vertex.								     |
    +------------------------------------------------------------------------*/

   else
   {
       delta = current->x - (current->w * cb->plane);
       alpha = delta / ( delta - (cb->previous.x - (cb->previous.w * cb->plane)) );
       intersection.x = current->x + alpha*(cb->previous.x - current->x);
       intersection.y = current->y + alpha*(cb->previous.y - current->y);
       intersection.z = current->z + alpha*(cb->previous.z - current->z);
       intersection.w = current->w + alpha*(cb->previous.w - current->w);

       cb->output_occurred = 1;
       pexpolyclip_xr(cb+1, &intersection);

       cb->previous.x = current->x;
       cb->previous.y = current->y;
       cb->previous.z = current->z;
       cb->previous.w = current->w;
   }

   /*------------------------------------------------------------------------+
    | Test to see if we should output the vertex now stored in previous.  If |
    | it is visible we should send it to the next clipping stage.	     |
    +------------------------------------------------------------------------*/

   if ( cb->previous.x >= (cb->previous.w * cb->plane) )
   {
       cb->output_occurred = 1;
       pexpolyclip_xr(cb+1, &cb->previous);
   }

   return 0;
}


/******************************************************************************
* +------------------------------------------------------------------------+ *
* | "pexpolyclip_xr"							   | *
* |									   | *
* |    Purpose:								   | *
* |	   Clip a polygon vertex to the right x clipping plane.		   | *
* |									   | *
* |    Parameters:							   | *
* |	   cb	   -> clipping plane information for right x clipping plane| *
* |	   current -> polygon vertex to be processed			   | *
* |									   | *
* +------------------------------------------------------------------------+ *
*****************************************************************************/

pexpolyclip_xr(cb, current)
   pexPolyclip_Cb	   *cb;
   pexCoord4D	   *current;
{
   FLOAT	   alpha;	   /* Ratio of line on each */
				   /*	side of clipping plane. */
   FLOAT	   delta;	   /* Distance in % direction between */
				   /*	current point and clip plane. */
   pexCoord4D	   intersection;   /* Storage for point which is */
				   /*	intersection of edge and plane. */

   /*------------------------------------------------------------------------+
    | Processing  first	 point. Save its value in previous and first storage |
    | locations of cb. Clear first point flag.				     |
    +------------------------------------------------------------------------*/

   if (cb->first_vertex)
   {
       cb->first.x = current->x;
       cb->previous.x = current->x;
       cb->first.y = current->y;
       cb->previous.y = current->y;
       cb->first.z = current->z;
       cb->previous.z = current->z;
       cb->first.w = current->w;
       cb->previous.w = current->w;

       cb->first_vertex = 0;
   }

   /*------------------------------------------------------------------------+
    | This is not the  first  point  of	 the  polygon,	and  the  edge	from |
    | previous	to  current  does  not	cross the clipping boundary. So just |
    | save the current point as the \ previous point.			     |
    +------------------------------------------------------------------------*/

   else if ( ( (current->x - (current->w * cb->plane)) *
	  (cb->previous.x - (cb->previous.w * cb->plane)) )
	  >= 0.0 )
   {
       cb->previous.x = current->x;
       cb->previous.y = current->y;
       cb->previous.z = current->z;
       cb->previous.w = current->w;
   }

   /*------------------------------------------------------------------------+
    | This is not the first point, and the edge	 from  current	to  previous |
    | crosses  the  clipping  plane.  So  calculate  the  intersection point |
    | between the edge and the plane and output	 it  to	 the  next  clipping |
    | stage.  Then  set	 the  previous	vertex	to the values of the current |
    | vertex.								     |
    +------------------------------------------------------------------------*/

   else
   {
       delta = current->x - (current->w * cb->plane);
       alpha = delta / ( delta - (cb->previous.x - (cb->previous.w * cb->plane)) );
       intersection.x = current->x + alpha*(cb->previous.x - current->x);
       intersection.y = current->y + alpha*(cb->previous.y - current->y);
       intersection.z = current->z + alpha*(cb->previous.z - current->z);
       intersection.w = current->w + alpha*(cb->previous.w - current->w);

       cb->output_occurred = 1;
       pexpolyclip_yb(cb+1, &intersection);

       cb->previous.x = current->x;
       cb->previous.y = current->y;
       cb->previous.z = current->z;
       cb->previous.w = current->w;
   }

   /*------------------------------------------------------------------------+
    | Test to see if we should output the vertex now stored in previous.  If |
    | it is visible we should send it to the next clipping stage.	     |
    +------------------------------------------------------------------------*/

   if ( cb->previous.x <= (cb->previous.w * cb->plane) )
   {
       cb->output_occurred = 1;
       pexpolyclip_yb(cb+1, &cb->previous);
   }

   return 0;
}

/******************************************************************************
* +------------------------------------------------------------------------+ *
* | "pexpolyclip_yb"							   | *
* |									   | *
* |    Purpose:								   | *
* |	   Clip a polygon vertex to the bottom y clipping plane.	   | *
* |									   | *
* |    Parameters:							   | *
* |	   cb	   -> clipping plane information for bottum y clipping plane| *
* |	   current -> polygon vertex to be processed			   | *
* |									   | *
* +------------------------------------------------------------------------+ *
*****************************************************************************/

pexpolyclip_yb(cb, current)
   pexPolyclip_Cb	   *cb;
   pexCoord4D	   *current;
{
   FLOAT	   alpha;	   /* Ratio of line on each */
				   /*	side of clipping plane. */
   FLOAT	   delta;	   /* Distance in y direction between */
				   /*	current point and clip plane. */
   pexCoord4D	   intersection;   /* Storage for point which is */
				   /*	intersection of edge and plane. */

   /*------------------------------------------------------------------------+
    | Processing  first	 point. Save its value in previous and first storage |
    | locations of cb. Clear first point flag.				     |
    +------------------------------------------------------------------------*/

   if (cb->first_vertex)
   {
       cb->first.x = current->x;
       cb->previous.x = current->x;
       cb->first.y = current->y;
       cb->previous.y = current->y;
       cb->first.z = current->z;
       cb->previous.z = current->z;
       cb->first.w = current->w;
       cb->previous.w = current->w;

       cb->first_vertex = 0;
   }

   /*------------------------------------------------------------------------+
    | This is not the  first  point  of	 the  polygon,	and  the  edge	from |
    | previous	to  current  does  not	cross the clipping boundary. So just |
    | save the current point as the \ previous point.			     |
    +------------------------------------------------------------------------*/

   else if ( ( (current->y - (current->w * cb->plane)) *
	  (cb->previous.y - (cb->previous.w * cb->plane)) )
	  >= 0.0 )
   {
       cb->previous.x = current->x;
       cb->previous.y = current->y;
       cb->previous.z = current->z;
       cb->previous.w = current->w;
   }

   /*------------------------------------------------------------------------+
    | This is not the first point, and the edge	 from  current	to  previous |
    | crosses  the  clipping  plane.  So  calculate  the  intersection point |
    | between the edge and the plane and output	 it  to	 the  next  clipping |
    | stage.  Then  set	 the  previous	vertex	to the values of the current |
    | vertex.								     |
    +------------------------------------------------------------------------*/

   else
   {
       delta = current->y - (current->w * cb->plane);
       alpha = delta / ( delta - (cb->previous.y - (cb->previous.w * cb->plane)) );
       intersection.x = current->x + alpha*(cb->previous.x - current->x);
       intersection.y = current->y + alpha*(cb->previous.y - current->y);
       intersection.z = current->z + alpha*(cb->previous.z - current->z);
       intersection.w = current->w + alpha*(cb->previous.w - current->w);

       cb->output_occurred = 1;
       pexpolyclip_yt(cb+1, &intersection);

       cb->previous.x = current->x;
       cb->previous.y = current->y;
       cb->previous.z = current->z;
       cb->previous.w = current->w;
   }

   /*------------------------------------------------------------------------+
    | Test to see if we should output the vertex now stored in previous.  If |
    | it is visible we should send it to the next clipping stage.	     |
    +------------------------------------------------------------------------*/

   if ( cb->previous.y >= (cb->previous.w * cb->plane) )
   {
       cb->output_occurred = 1;
       pexpolyclip_yt(cb+1, &cb->previous);
   }

   return 0;
}

/******************************************************************************
* +------------------------------------------------------------------------+ *
* | "pexpolyclip_yt"							   | *
* |									   | *
* |    Purpose:								   | *
* |	   Clip a polygon vertex to the top y clipping plane.		   | *
* |									   | *
* |    Parameters:							   | *
* |	   cb	   -> clipping plane information for top y clipping plane  | *
* |	   current -> polygon vertex to be processed			   | *
* |									   | *
* +------------------------------------------------------------------------+ *
*****************************************************************************/

pexpolyclip_yt(cb, current)
   pexPolyclip_Cb	   *cb;
   pexCoord4D	   *current;
{
   FLOAT	   alpha;	   /* Ratio of line on each */
				   /*	side of clipping plane. */
   FLOAT	   delta;	   /* Distance in y direction between */
				   /*	current point and clip plane. */
   pexCoord4D	   intersection;   /* Storage for point which is */
				   /*	intersection of edge and plane. */

   /*------------------------------------------------------------------------+
    | Processing  first	 point. Save its value in previous and first storage |
    | locations of cb. Clear first point flag.				     |
    +------------------------------------------------------------------------*/

   if (cb->first_vertex)
   {
       cb->first.x = current->x;
       cb->previous.x = current->x;
       cb->first.y = current->y;
       cb->previous.y = current->y;
       cb->first.z = current->z;
       cb->previous.z = current->z;
       cb->first.w = current->w;
       cb->previous.w = current->w;

       cb->first_vertex = 0;
   }

   /*------------------------------------------------------------------------+
    | This is not the  first  point  of	 the  polygon,	and  the  edge	from |
    | previous	to  current  does  not	cross the clipping boundary. So just |
    | save the current point as the \ previous point.			     |
    +------------------------------------------------------------------------*/

   else if ( ( (current->y - (current->w * cb->plane)) *
	  (cb->previous.y - (cb->previous.w * cb->plane)) )
	  >= 0.0 )
   {
       cb->previous.x = current->x;
       cb->previous.y = current->y;
       cb->previous.z = current->z;
       cb->previous.w = current->w;
   }

   /*------------------------------------------------------------------------+
    | This is not the first point, and the edge	 from  current	to  previous |
    | crosses  the  clipping  plane.  So  calculate  the  intersection point |
    | between the edge and the plane and output	 it  to	 the  next  clipping |
    | stage.  Then  set	 the  previous	vertex	to the values of the current |
    | vertex.								     |
    +------------------------------------------------------------------------*/

   else
   {
       delta = current->y - (current->w * cb->plane);
       alpha = delta / ( delta - (cb->previous.y - (cb->previous.w * cb->plane)) );
       intersection.x = current->x + alpha*(cb->previous.x - current->x);
       intersection.y = current->y + alpha*(cb->previous.y - current->y);
       intersection.z = current->z + alpha*(cb->previous.z - current->z);
       intersection.w = current->w + alpha*(cb->previous.w - current->w);

       cb->output_occurred = 1;
       pexpolyclip_zh(cb+1, &intersection);

       cb->previous.x = current->x;
       cb->previous.y = current->y;
       cb->previous.z = current->z;
       cb->previous.w = current->w;
   }

   /*------------------------------------------------------------------------+
    | Test to see if we should output the vertex now stored in previous.  If |
    | it is visible we should send it to the next clipping stage.	     |
    +------------------------------------------------------------------------*/

   if ( cb->previous.y <= (cb->previous.w * cb->plane) )
   {
       cb->output_occurred = 1;
       pexpolyclip_zh(cb+1, &cb->previous);
   }

   return 0;
}

/******************************************************************************
* +------------------------------------------------------------------------+ *
* | "pexpolyclip_zh"							   | *
* |									   | *
* |    Purpose:								   | *
* |	   Clip a polygon vertex to the z hither clipping plane.	   | *
* |									   | *
* |    Parameters:							   | *
* |	   cb	   -> clipping plane information for z hither clipping plane| *
* |	   current -> polygon vertex to be processed			   | *
* |									   | *
* +------------------------------------------------------------------------+ *
*****************************************************************************/

pexpolyclip_zh(cb, current)
   pexPolyclip_Cb	   *cb;
   pexCoord4D	   *current;
{
   FLOAT	   alpha;	   /* Ratio of line on each */
				   /*	side of clipping plane. */
   FLOAT	   delta;	   /* Distance in y direction between */
				   /*	current point and clip plane. */
   pexCoord4D	   intersection;   /* Storage for point which is */
				   /*	intersection of edge and plane. */

   /*------------------------------------------------------------------------+
    | Processing  first	 point. Save its value in previous and first storage |
    | locations of cb. Clear first point flag.				     |
    +------------------------------------------------------------------------*/

   if (cb->first_vertex)
   {
       cb->first.x = current->x;
       cb->previous.x = current->x;
       cb->first.y = current->y;
       cb->previous.y = current->y;
       cb->first.z = current->z;
       cb->previous.z = current->z;
       cb->first.w = current->w;
       cb->previous.w = current->w;

       cb->first_vertex = 0;
   }

   /*------------------------------------------------------------------------+
    | This is not the  first  point  of	 the  polygon,	and  the  edge	from |
    | previous	to  current  does  not	cross the clipping boundary. So just |
    | save the current point as the \ previous point.			     |
    +------------------------------------------------------------------------*/

   else if ( ( (current->z - (current->w * cb->plane)) *
	  (cb->previous.z - (cb->previous.w * cb->plane)) )
	  >= 0.0 )
   {
       cb->previous.x = current->x;
       cb->previous.y = current->y;
       cb->previous.z = current->z;
       cb->previous.w = current->w;
   }

   /*------------------------------------------------------------------------+
    | This is not the first point, and the edge	 from  current	to  previous |
    | crosses  the  clipping  plane.  So  calculate  the  intersection point |
    | between the edge and the plane and output	 it  to	 the  next  clipping |
    | stage.  Then  set	 the  previous	vertex	to the values of the current |
    | vertex.								     |
    +------------------------------------------------------------------------*/

   else
   {
       delta = current->z - (current->w * cb->plane);
       alpha = delta / ( delta - (cb->previous.z - (cb->previous.w * cb->plane)) );
       intersection.x = current->x + alpha*(cb->previous.x - current->x);
       intersection.y = current->y + alpha*(cb->previous.y - current->y);
       intersection.z = current->z + alpha*(cb->previous.z - current->z);
       intersection.w = current->w + alpha*(cb->previous.w - current->w);

       cb->output_occurred = 1;
       pexpolyclip_zy(cb+1, &intersection);

       cb->previous.x = current->x;
       cb->previous.y = current->y;
       cb->previous.z = current->z;
       cb->previous.w = current->w;
   }

   /*------------------------------------------------------------------------+
    | Test to see if we should output the vertex now stored in previous.  If |
    | it is visible we should send it to the next clipping stage.	     |
    +------------------------------------------------------------------------*/

   if ( cb->previous.z >= (cb->previous.w * cb->plane) )
   {
       cb->output_occurred = 1;
       pexpolyclip_zy(cb+1, &cb->previous);
   }

   return 0;
}

/******************************************************************************
* +------------------------------------------------------------------------+ *
* | "pexpolyclip_zy"							   | *
* |									   | *
* |    Purpose:								   | *
* |	   Clip a polygon vertex to the yon z clipping plane.		   | *
* |									   | *
* |    Parameters:							   | *
* |	   cb	   -> clipping plane information for left % clipping plane | *
* |	   current -> polygon vertex to be processed			   | *
* |									   | *
* +------------------------------------------------------------------------+ *
*****************************************************************************/

pexpolyclip_zy(cb, current)
   pexPolyclip_Cb	   *cb;
   pexCoord4D	   *current;
{
   FLOAT	   alpha;	   /* Ratio of line on each */
				   /*	side of clipping plane. */
   FLOAT	   delta;	   /* Distance in y direction between */
				   /*	current point and clip plane. */
   pexCoord4D	   intersection;   /* Storage for point which is */
				   /*	intersection of edge and plane. */

   /*------------------------------------------------------------------------+
    | Processing  first	 point. Save its value in previous and first storage |
    | locations of cb. Clear first point flag.				     |
    +------------------------------------------------------------------------*/

   if (cb->first_vertex)
   {
       cb->first.x = current->x;
       cb->previous.x = current->x;
       cb->first.y = current->y;
       cb->previous.y = current->y;
       cb->first.z = current->z;
       cb->previous.z = current->z;
       cb->first.w = current->w;
       cb->previous.w = current->w;

       cb->first_vertex = 0;
   }

   /*------------------------------------------------------------------------+
    | This is not the  first  point  of	 the  polygon,	and  the  edge	from |
    | previous	to  current  does  not	cross the clipping boundary. So just |
    | save the current point as the \ previous point.			     |
    +------------------------------------------------------------------------*/

   else if ( ( (current->z - (current->w * cb->plane)) *
	  (cb->previous.z - (cb->previous.w * cb->plane)) )
	  >= 0.0 )
   {
       cb->previous.x = current->x;
       cb->previous.y = current->y;
       cb->previous.z = current->z;
       cb->previous.w = current->w;
   }

   /*------------------------------------------------------------------------+
    | This is not the first point, and the edge	 from  current	to  previous |
    | crosses  the  clipping  plane.  So  calculate  the  intersection point |
    | between the edge and the plane and output	 it  to	 the  next  clipping |
    | stage.  Then  set	 the  previous	vertex	to the values of the current |
    | vertex.								     |
    +------------------------------------------------------------------------*/

   else
   {
       delta = current->z - (current->w * cb->plane);
       alpha = delta / ( delta - (cb->previous.z - (cb->previous.w * cb->plane)) );
       intersection.x = current->x + alpha*(cb->previous.x - current->x);
       intersection.y = current->y + alpha*(cb->previous.y - current->y);
       intersection.z = current->z + alpha*(cb->previous.z - current->z);
       intersection.w = current->w + alpha*(cb->previous.w - current->w);

       cb->output_occurred = 1;
       pexpolyclip_out(cb+1, &intersection);

       cb->previous.x = current->x;
       cb->previous.y = current->y;
       cb->previous.z = current->z;
       cb->previous.w = current->w;
   }

   /*------------------------------------------------------------------------+
    | Test to see if we should output the vertex now stored in previous.  If |
    | it is visible we should send it to the next clipping stage.	     |
    +------------------------------------------------------------------------*/

   if ( cb->previous.z <= (cb->previous.w * cb->plane) )
   {
       cb->output_occurred = 1;
       pexpolyclip_out(cb+1, &cb->previous);
   }

   return 0;
}

/******************************************************************************
* +------------------------------------------------------------------------+ *
* | "pexpolyclose_xl"							   | *
* |									   | *
* |    Purpose:								   | *
* |	   Close the x left clipping stage for a polygon.		   | *
* |									   | *
* |    Parameters:							   | *
* |	   cb	   -> clipping plane information for left x clipping plane | *
* |									   | *
* +------------------------------------------------------------------------+ *
*****************************************************************************/

pexpolyclose_xl(cb)
   pexPolyclip_Cb	   *cb;
{
   FLOAT	   alpha;	   /* Ratio of line on each side of */
				   /*	clipping plane. */
   FLOAT	   delta;	   /* Distance in % direction between */
				   /*	current point and clip plane. */
   pexCoord4D	   intersection;   /* Storage for point which is */
				   /*	intersection of edge and plane. */

   /*------------------------------------------------------------------------+
    | Output  occurred.	 So  close  the polygon back to its first vertex. If |
    | the edge from  the  last	vertex	to  the	 first	vertex	crosses	 the |
    | clipping plane, generate the intersection point and output it.	     |
    +------------------------------------------------------------------------*/

   if (cb->output_occurred)
   {
       if ( ( (cb->previous.x - (cb->previous.w * cb->plane)) *
	      (cb->first.x - (cb->first.w * cb->plane)) )
	      < 0.0 )
       {
	   delta = cb->first.x - (cb->first.w * cb->plane);
	   alpha = delta / ( delta - (cb->previous.x -
				      (cb->previous.w * cb->plane)) );
	   intersection.x = cb->first.x + alpha*(cb->previous.x - cb->first.x);
	   intersection.y = cb->first.y + alpha*(cb->previous.y - cb->first.y);
	   intersection.z = cb->first.z + alpha*(cb->previous.z - cb->first.z);
	   intersection.w = cb->first.w + alpha*(cb->previous.w - cb->first.w);

	   pexpolyclip_xr(cb+1, &intersection);
       }
   }

   /*------------------------------------------------------------------------+
    | Reset flags and close off next stage.				     |
    +------------------------------------------------------------------------*/

   cb->output_occurred = 0;
   cb->first_vertex = 1;
   pexpolyclose_xr(cb+1);

   return 0;
}

/******************************************************************************
* +------------------------------------------------------------------------+ *
* | "pexpolyclose_xr"							   | *
* |									   | *
* |    Purpose:								   | *
* |	   Close the x right clipping stage for a polygon.		   | *
* |									   | *
* |    Parameters:							   | *
* |	   cb	   -> clipping plane information for right x clipping plane| *
* |									   | *
* +------------------------------------------------------------------------+ *
*****************************************************************************/

pexpolyclose_xr(cb)
   pexPolyclip_Cb	   *cb;
{
   FLOAT	   alpha;	   /* Ratio of line on each side of */
				   /*	clipping plane. */
   FLOAT	   delta;	   /* Distance in % direction between */
				   /*	current point and clip plane. */
   pexCoord4D	   intersection;   /* Storage for point which is */
				   /*	intersection of edge and plane. */

   /*------------------------------------------------------------------------+
    | Output  occurred.	 So  close  the polygon back to its first vertex. If |
    | the edge from  the  last	vertex	to  the	 first	vertex	crosses	 the |
    | clipping plane, generate the intersection point and output it.	     |
    +------------------------------------------------------------------------*/

   if (cb->output_occurred)
   {
       if ( ( (cb->previous.x - (cb->previous.w * cb->plane)) *
	      (cb->first.x - (cb->first.w * cb->plane)) )
	      < 0.0 )
       {
	   delta = cb->first.x - (cb->first.w * cb->plane);
	   alpha = delta / ( delta - (cb->previous.x -
				      (cb->previous.w * cb->plane)) );
	   intersection.x = cb->first.x + alpha*(cb->previous.x - cb->first.x);
	   intersection.y = cb->first.y + alpha*(cb->previous.y - cb->first.y);
	   intersection.z = cb->first.z + alpha*(cb->previous.z - cb->first.z);
	   intersection.w = cb->first.w + alpha*(cb->previous.w - cb->first.w);

	   pexpolyclip_yb(cb+1, &intersection);
       }
   }

   /*------------------------------------------------------------------------+
    | Reset flags and close off next stage.				     |
    +------------------------------------------------------------------------*/

   cb->output_occurred = 0;
   cb->first_vertex = 1;
   pexpolyclose_yb(cb+1);

   return 0;
}

/******************************************************************************
* +------------------------------------------------------------------------+ *
* | "pexpolyclose_yb"							   | *
* |									   | *
* |    Purpose:								   | *
* |	   Close the y bottom clipping stage for a polygon.		   | *
* |									   | *
* |    Parameters:							   | *
* |	   cb	   -> clipping plane information for bottom y clipping plane| *
* |									   | *
* +------------------------------------------------------------------------+ *
*****************************************************************************/

pexpolyclose_yb(cb)
   pexPolyclip_Cb	   *cb;
{
   FLOAT	   alpha;	   /* Ratio of line on each side of */
				   /*	clipping plane. */
   FLOAT	   delta;	   /* Distance in % direction between */
				   /*	current point and clip plane. */
   pexCoord4D	   intersection;   /* Storage for point which is */
				   /*	intersection of edge and plane. */

   /*------------------------------------------------------------------------+
    | Output  occurred.	 So  close  the polygon back to its first vertex. If |
    | the edge from  the  last	vertex	to  the	 first	vertex	crosses	 the |
    | clipping plane, generate the intersection point and output it.	     |
    +------------------------------------------------------------------------*/

   if (cb->output_occurred)
   {
       if ( ( (cb->previous.y - (cb->previous.w * cb->plane)) *
	      (cb->first.y - (cb->first.w * cb->plane)) )
	      < 0.0 )
       {
	   delta = cb->first.y - (cb->first.w * cb->plane);
	   alpha = delta / ( delta - (cb->previous.y -
				      (cb->previous.w * cb->plane)) );
	   intersection.x = cb->first.x + alpha*(cb->previous.x - cb->first.x);
	   intersection.y = cb->first.y + alpha*(cb->previous.y - cb->first.y);
	   intersection.z = cb->first.z + alpha*(cb->previous.z - cb->first.z);
	   intersection.w = cb->first.w + alpha*(cb->previous.w - cb->first.w);

	   pexpolyclip_yt(cb+1, &intersection);
       }
   }

   /*------------------------------------------------------------------------+
    | Reset flags and close off next stage.				     |
    +------------------------------------------------------------------------*/

   cb->output_occurred = 0;
   cb->first_vertex = 1;
   pexpolyclose_yt(cb+1);

   return 0;
}

/******************************************************************************
* +------------------------------------------------------------------------+ *
* | "pexpolyclose_yt"							   | *
* |									   | *
* |    Purpose:								   | *
* |	   Close the y top clipping stage for a polygon.		   | *
* |									   | *
* |    Parameters:							   | *
* |	   cb	   -> clipping plane information for top y clipping plane  | *
* |									   | *
* +------------------------------------------------------------------------+ *
*****************************************************************************/

pexpolyclose_yt(cb)
   pexPolyclip_Cb	   *cb;
{
   FLOAT	   alpha;	   /* Ratio of line on each */
				   /*	side of clipping plane. */
   FLOAT	   delta;	   /* Distance in % direction between */
				   /*	current point and clip plane. */
   pexCoord4D	   intersection;   /* Storage for point which is */
				   /*	intersection of edge and plane. */

   /*------------------------------------------------------------------------+
    | Output  occurred.	 So  close  the polygon back to its first vertex. If |
    | the edge from  the  last	vertex	to  the	 first	vertex	crosses	 the |
    | clipping plane, generate the intersection point and output it.	     |
    +------------------------------------------------------------------------*/

   if (cb->output_occurred)
   {
       if ( ( (cb->previous.y - (cb->previous.w * cb->plane)) *
	      (cb->first.y - (cb->first.w * cb->plane)) )
	      < 0.0 )
       {
	   delta = cb->first.y - (cb->first.w * cb->plane);
	   alpha = delta / ( delta - (cb->previous.y -
				      (cb->previous.w * cb->plane)) );
	   intersection.x = cb->first.x + alpha*(cb->previous.x - cb->first.x);
	   intersection.y = cb->first.y + alpha*(cb->previous.y - cb->first.y);
	   intersection.z = cb->first.z + alpha*(cb->previous.z - cb->first.z);
	   intersection.w = cb->first.w + alpha*(cb->previous.w - cb->first.w);

	   pexpolyclip_zh(cb+1, &intersection);
       }
   }

   /*------------------------------------------------------------------------+
    | Reset flags and close off next stage.				     |
    +------------------------------------------------------------------------*/

   cb->output_occurred = 0;
   cb->first_vertex = 1;
   pexpolyclose_zh(cb+1);

   return 0;
}

/******************************************************************************
* +------------------------------------------------------------------------+ *
* | "pexpolyclose_zh"							   | *
* |									   | *
* |    Purpose:								   | *
* |	   Close the z hither clipping stage for a polygon.		   | *
* |									   | *
* |    Parameters:							   | *
* |	   cb	   -> clipping plane information for hither z clipping plane| *
* |									   | *
* +------------------------------------------------------------------------+ *
*****************************************************************************/

pexpolyclose_zh(cb)
   pexPolyclip_Cb	   *cb;
{
   FLOAT	   alpha;	   /* Ratio of line on each */
				   /*	side of clipping plane. */
   FLOAT	   delta;	   /* Distance in % direction between */
				   /*	current point and clip plane. */
   pexCoord4D	   intersection;   /* Storage for point which */
				   /*	is intersection of edge and plane. */

   /*------------------------------------------------------------------------+
    | Output  occurred.	 So  close  the polygon back to its first vertex. If |
    | the edge from  the  last	vertex	to  the	 first	vertex	crosses	 the |
    | clipping plane, generate the intersection point and output it.	     |
    +------------------------------------------------------------------------*/

   if (cb->output_occurred)
   {
       if ( ( (cb->previous.z - (cb->previous.w * cb->plane)) *
	      (cb->first.z - (cb->first.w * cb->plane)) )
	     < 0.0 )
       {
	   delta = cb->first.z - (cb->first.w * cb->plane);
	   alpha = delta / ( delta - (cb->previous.z -
				      (cb->previous.w * cb->plane)) );
	   intersection.x = cb->first.x + alpha*(cb->previous.x - cb->first.x);
	   intersection.y = cb->first.y + alpha*(cb->previous.y - cb->first.y);
	   intersection.z = cb->first.z + alpha*(cb->previous.z - cb->first.z);
	   intersection.w = cb->first.w + alpha*(cb->previous.w - cb->first.w);

	   pexpolyclip_zy(cb+1, &intersection);
       }
   }

   /*------------------------------------------------------------------------+
    | Reset flags and close off next stage.				     |
    +------------------------------------------------------------------------*/

   cb->output_occurred = 0;
   cb->first_vertex = 1;
   pexpolyclose_zy(cb+1);

   return 0;
}

/******************************************************************************
* +------------------------------------------------------------------------+ *
* | "pexpolyclose_zy"							   | *
* |									   | *
* |    Purpose:								   | *
* |	   Close the z yon clipping stage for a polygon.		   | *
* |									   | *
* |    Parameters:							   | *
* |	   cb	   -> clipping plane information for yon z clipping plane  | *
* |									   | *
* +------------------------------------------------------------------------+ *
*****************************************************************************/

pexpolyclose_zy(cb)
   pexPolyclip_Cb	   *cb;
{
   FLOAT	   alpha;	   /* Ratio of line on each side */
				   /*	of clipping plane. */
   FLOAT	   delta;	   /* Distance in % direction between */
				   /*	current point and clip plane. */
   pexCoord4D	   intersection;   /* Storage for point which is */
				   /*	intersection of edge and plane. */

   /*------------------------------------------------------------------------+
    | Output  occurred.	 So  close  the polygon back to its first vertex. If |
    | the edge from  the  last	vertex	to  the	 first	vertex	crosses	 the |
    | clipping plane, generate the intersection point and output it.	     |
    +------------------------------------------------------------------------*/

   if (cb->output_occurred)
   {
       if ( ( (cb->previous.z - (cb->previous.w * cb->plane)) *
	      (cb->first.z - (cb->first.w * cb->plane)) )
	      < 0 )
       {
	   delta = cb->first.z - (cb->first.w * cb->plane);
	   alpha = delta / ( delta - (cb->previous.z -
				      (cb->previous.w * cb->plane)) );
	   intersection.x = cb->first.x + alpha*(cb->previous.x - cb->first.x);
	   intersection.y = cb->first.y + alpha*(cb->previous.y - cb->first.y);
	   intersection.z = cb->first.z + alpha*(cb->previous.z - cb->first.z);
	   intersection.w = cb->first.w + alpha*(cb->previous.w - cb->first.w);

	   pexpolyclip_out(cb+1, &intersection);
       }
   }

   /*------------------------------------------------------------------------+
    | Reset flags.							     |
    +------------------------------------------------------------------------*/

   cb->output_occurred = 0;
   cb->first_vertex = 1;

   return 0;
}


/******************************************************************************
* +------------------------------------------------------------------------+ *
* | "PexClipPolygon"								   | *
* |									   | *
* |    Purpose:								   | *
* |	   Clip a polygon against a viewport and return resultant polygon  | *
* |									   | *
* |    Parameters:							   | *
* |	   in_count	 =  number of vertices in polygon to be clipped	   | *
* |	   in_vertices	 -> array of vertices for polygon to be clipped	   | *
* |	   max_out_count =  maximum number of vertices in out_vertices	   | *
* |	   out_vertices	 -> array to store clipped vertices in		   | *
* |									   | *
* |    Globals:								   | *
* |	   out_count	 =  count of vertices placed in out_vertices	   | *
* |	   out_ptr	 =  runs across out_vertices when storing vertices | *
* |	   plane_cbs	 =  for planes to clip to ( by pexpolyclip_init call)| *
* |	   vertices_left =  for down counter for max_out_count		   | *
* |									   | *
* |    Function value:							   | *
* |	   if > 0 it indicates number of vertices in output polygon	   | *
* |	   if = 0 polygon was clipped					   | *
* |	   if < 0 then clipped polygon has more than max_out_count vertices| *
* |									   | *
* +------------------------------------------------------------------------+ *
*****************************************************************************/

CARD32
PexClipPolygon(in_count, in_vertices, max_out_count, out_vertices)
   int		       in_count;
   pexCoord4D	       *in_vertices;
   int		       max_out_count;
   pexCoord4D	       *out_vertices;
{
   pexCoord4D	       *current;     /* Cursor for processing in_vertices. */
   extern  int	       out_count;    /* Storage for number of vrtcs clipped. */
   extern  pexCoord4D  *out_ptr;     /* Next clipped vertex to store. */
   extern  pexPolyclip_Cb plane_cbs[6]; /* Storage for planes to clip to. */
   extern  int	       vertices_left; /* Storage for vertices left out_vert. */
   static int		initialized = 0;

   /*------------------------------------------------------------------------+
    | Initialize  global  storage  locations associated with our parameters. |
    | Then set indices for processing in_vertices.			     |
    +------------------------------------------------------------------------*/

   vertices_left = max_out_count;
   out_count = 0;
   out_ptr = out_vertices;
   current = in_vertices;

   /*------------------------------------------------------------------------+
    | Loop through input array clipping the vertices based on  the  clipping |
    | planes in "planes".						     |
    +------------------------------------------------------------------------*/

   while (--in_count >= 0)
   {
       pexpolyclip_xl(plane_cbs, current);
       current++;
   }

   /*------------------------------------------------------------------------+
    | If  we  exceeded	the size of out_vertices, send back a -1 indication. |
    | Otherwise return the number of vertices processed.		     |
    +------------------------------------------------------------------------*/

   if (vertices_left < 0)
   {
       fprintf(stderr, "Exceeded out_vertices size in the Pex polygon clipping code.\n");
       return -out_count;
   }

   pexpolyclose_xl(plane_cbs);
   return out_count;
}



