/*
 * xfed - a little font editor
 *
 * Copyright 1988 Siemens
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Siemens not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  Siemens makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * Author:  Claus Gittinger, Siemens
 *
 *
 * a little font editor by claus
 * usage: xfed [-fg color] [-bg color] 
 *             [-bd color] [-bw number] [-nogrid] [-psize number] fontname.bdf
 */

#include "defs.h"
#include "X11/Xlib.h"
#include "X11/Xatom.h"
#include "X11/Xutil.h"
#include "X11/cursorfont.h"
#include <stdio.h>
#include <errno.h>

#include "icon.bit"
#include "next.bit"
#include "prev.bit"
#include "save.bit"
#include "quit.bit"

Window XCreateWindow();
Window mainWin, charinfoWin, infoWin, editWin, nextWin, prevWin, quitWin, saveWin;
Display *dpy;
Colormap cmap;
GC gc;
int screen;
int winW, winH;
char *borderColor = "black";
char *foregroundColor = "black";
char *backgroundColor = "white";
int borderWidth = 1;
long foregroundPixel, backgroundPixel, borderPixel;
char *fontfilename = NULL;
Pixmap saveButton, nextButton, prevButton, quitButton;
struct character *charp;
int charIndex = 0;
int firstSave = 1;

#define SPACE		5
int PIXSIZE = 16;
int grid=1;

#define info_height	20
#define charinfo_height	20

int info_x, info_y;
int info_width;
int charinfo_x, charinfo_y;
int charinfo_width;
int edit_x, edit_y;
int edit_width, edit_height;
int next_x, next_y;
int prev_x, prev_y;
int quit_x, quit_y;
int save_x, save_y;

#define SETBIT		0
#define CLEARBIT	1
#define COMBIT		2

main(argc, argv)
char *argv[];
{
	int i,r,g,b;
	char **ap;
	char *cp;

	for (ap=argv; *ap != NULL; ap++) {
	    cp = *ap;
	    if (strcmp(cp, "-bd") == 0)
		borderColor = *++ap;
	    else if (strcmp(cp, "-nogrid") == 0)
		grid = 0;
	    else if (strcmp(cp, "-fg") == 0)
		foregroundColor = *++ap;
	    else if (strcmp(cp, "-bg") == 0)
		backgroundColor = *++ap;
	    else if (strcmp(cp, "-psize") == 0) 
		sscanf(*++ap, "%d", &PIXSIZE);
	    else if (strcmp(cp, "-bw") == 0) 
		sscanf(*++ap, "%d", &borderWidth);
	    else
		fontfilename = *ap;
	}


	if (fontfilename == NULL) usage(argv, argc);

	if (freopen(fontfilename, "r", stdin) == NULL) {
	    fprintf(stderr, "cannot open %s\n", fontfilename);
	    exit(1);
	}
	yyparse();
	setupWindow(argc, argv);
	charp = font.characters; charIndex = 0;
	doEvents();
}

usage(argc, argv) 
char *argv[];
{
	fprintf(stderr, "usage: %s [-fg color] [-bg color] [-bw n] [-bd color]\n", argv[0]);
	fprintf(stderr, "          [-nogrid] [-psize number] file\n");
	exit(1);
}

setupWindow(argc, argv) 
char *argv[];
{
    int amount, i, j ;
    char *geom = NULL;
    int winx, winy;
    register int xdir, ydir;
    register int xoff, yoff;
    register int centerX, centerY;
    XGCValues xgcv;
    XSetWindowAttributes xswa;
    Window root;
    int x, y, w, h;
    unsigned int d;
    char *display;
    Visual visual;
    XSizeHints szhint;
    XColor def, exact;

    display = (char *)getenv("DISPLAY");
    if (!(dpy = XOpenDisplay(display)))
    {
	perror("Cannot open display\n");
	exit(-1);
    }

    screen = DefaultScreen(dpy);
    cmap = DefaultColormap(dpy, screen);

    XAllocNamedColor(dpy, cmap, borderColor, &def, &exact);
    borderPixel = def.pixel;
    XAllocNamedColor(dpy, cmap, foregroundColor, &def, &exact);
    foregroundPixel = def.pixel;
    XAllocNamedColor(dpy, cmap, backgroundColor, &def, &exact);
    backgroundPixel = def.pixel;

    info_x = info_y = 0;
    charinfo_x = 0;
    charinfo_y = info_y + SPACE + info_height;
    edit_x = 0;
    edit_y = charinfo_y + SPACE + charinfo_height;
    edit_width = (PIXSIZE+grid)*font.maxbbx.w;
    edit_height = (PIXSIZE+grid)*font.maxbbx.h;
    prev_x = edit_width + SPACE;
    prev_y = edit_y;
    next_x = prev_x;
    next_y = prev_y + SPACE + prev_height;
    save_x = prev_x;
    save_y = next_y + SPACE + next_height;
    quit_x = prev_x;
    quit_y = save_y + SPACE + save_height;

    winx = 0;
    winy = 0;
    winW = edit_width + SPACE + prev_width + SPACE;
    winH = info_height + SPACE + charinfo_height + SPACE + edit_height + SPACE;
    if (winH < (quit_y+quit_height+SPACE))
	winH = quit_y + quit_height + SPACE;

    info_width = winW;
    charinfo_width = winW;

    szhint.flags = PPosition | PSize;
    szhint.x = 0;
    szhint.y = 0;
    szhint.width = winW;
    szhint.height = winH;

    xswa.backing_store = NotUseful;
    xswa.event_mask = ExposureMask;
    xswa.background_pixel = backgroundPixel;
    xswa.border_pixel = borderPixel;
    visual.visualid = CopyFromParent;
    mainWin = XCreateWindow(dpy,
		RootWindow(dpy, DefaultScreen(dpy)),
		winx, winy, szhint.width, szhint.height, borderWidth,
		DefaultDepth(dpy, 0), InputOutput,
		&visual, 
	        CWEventMask | CWBackingStore | CWBorderPixel | CWBackPixel, 
		&xswa);

    XSetStandardProperties(dpy, mainWin, "fed", "fed",
				XCreateBitmapFromData(dpy, mainWin,
						     icon_bits,
						     icon_width, icon_height),
				argv, argc, &szhint);

    XSelectInput(dpy, mainWin, ButtonPressMask|ExposureMask);
    XMapWindow(dpy, mainWin);

    gc = XCreateGC(dpy, mainWin, 0L, &xgcv);
    XSetForeground(dpy, gc, foregroundPixel);
    XSetBackground(dpy, gc, backgroundPixel);

    infoWin = XCreateSimpleWindow(dpy, mainWin,
			    info_x, info_y, info_width, info_height,
			    1, borderPixel, backgroundPixel);
    XSelectInput(dpy, infoWin, ExposureMask);

    charinfoWin = XCreateSimpleWindow(dpy, mainWin,
			    charinfo_x, charinfo_y, charinfo_width, charinfo_height,
			    1, borderPixel, backgroundPixel);
    XSelectInput(dpy, charinfoWin, ExposureMask);

    editWin = XCreateSimpleWindow(dpy, mainWin,
			    edit_x, edit_y, edit_width, edit_height,
			    1, borderPixel, backgroundPixel);
    XSelectInput(dpy, editWin, ExposureMask | ButtonPressMask |
			       Button1MotionMask | Button2MotionMask |
			       Button3MotionMask);
    xswa.cursor = XCreateFontCursor(dpy, XC_cross);
    XChangeWindowAttributes(dpy, editWin, CWCursor, &xswa);

    prevWin = XCreateSimpleWindow(dpy, mainWin,
			    prev_x, prev_y, prev_width, prev_height,
			    1, borderPixel, backgroundPixel);
    XSelectInput(dpy, prevWin, ExposureMask | ButtonPressMask);
    prevButton = XCreateBitmapFromData(dpy, mainWin,
			       prev_bits, prev_width, prev_height);

    nextWin = XCreateSimpleWindow(dpy, mainWin,
			    next_x, next_y, next_width, next_height,
			    1, borderPixel, backgroundPixel);
    XSelectInput(dpy, nextWin, ExposureMask | ButtonPressMask);
    nextButton = XCreateBitmapFromData(dpy, mainWin,
			       next_bits, next_width, next_height);

    saveWin = XCreateSimpleWindow(dpy, mainWin,
			    save_x, save_y, save_width, save_height,
			    1, borderPixel, backgroundPixel);
    XSelectInput(dpy, saveWin, ExposureMask | ButtonPressMask);
    saveButton = XCreateBitmapFromData(dpy, mainWin,
			       save_bits, save_width, save_height);

    quitWin = XCreateSimpleWindow(dpy, mainWin,
			    quit_x, quit_y, quit_width, quit_height,
			    1, borderPixel, backgroundPixel);
    XSelectInput(dpy, quitWin, ExposureMask | ButtonPressMask);
    quitButton = XCreateBitmapFromData(dpy, mainWin,
			       quit_bits, quit_width, quit_height);

    XMapWindow(dpy, infoWin);
    XMapWindow(dpy, charinfoWin);
    XMapWindow(dpy, editWin);
    XMapWindow(dpy, prevWin);
    XMapWindow(dpy, nextWin);
    XMapWindow(dpy, saveWin);
    XMapWindow(dpy, quitWin);
}

doEvents() {
    XEvent pe;
    XExposeEvent *ee = (XExposeEvent *) &pe;
    XConfigureEvent *ce = (XConfigureEvent *)&pe;
    XButtonPressedEvent *be = (XButtonPressedEvent *)&pe;
    XMotionEvent *me = (XMotionEvent *)&pe;
    int last_row, last_col;
    int this_row, this_col;

    while(1) {
	XNextEvent(dpy, &pe);       /* this should get first exposure event */
	switch (pe.type) {
	    case Expose:
		if (ee->window == editWin)
		    redrawEditWindow(ee->x, ee->y, ee->width, ee->height);
		else if (ee->window == prevWin)
		    redrawPrevWindow();
		else if (ee->window == nextWin)
		    redrawNextWindow();
		else if (ee->window == infoWin)
		    redrawInfoWindow();
		else if (ee->window == charinfoWin)
		    redrawCharinfoWindow();
		else if (ee->window == quitWin)
		    redrawQuitWindow();
		else if (ee->window == saveWin)
		    redrawSaveWindow();
		break;

	    case ButtonPress:
		if (be->window == editWin) {
		    whatSquare(be->x, be->y, &last_row, &last_col);
		    switch (be->button) {
			case Button1:	/* left button */ 
		            bitOp(SETBIT, last_row, last_col);
			    break;
			case Button2:	/* middle button */ 
		            bitOp(COMBIT, last_row, last_col);
			    break;
			case Button3:	/* right button */ 
		            bitOp(CLEARBIT, last_row, last_col);
			    break;
		    }
		} else if (be->window == prevWin)
		    prevCharacter();
		else if (be->window == nextWin)
		    nextCharacter();
		else if (be->window == saveWin) {
		    char cmdbuff[132];

		    if (firstSave) {
			firstSave = 0;
		        sprintf(cmdbuff, "cp %s %s~", fontfilename, fontfilename);
		        system(cmdbuff);
		    }
		    output(&font, fontfilename);
		} else if (be->window == quitWin)
		    return;
		break;

    	    case MotionNotify:
		if (me->window != editWin) break;
      		if (whatSquare (me->x, me->y, &this_row, &this_col))
        	    break;  
      		if ((this_row != last_row) || (this_col != last_col)) {
       	  	    switch (me->state) {
	    		case Button1Mask:	/* left button down */
	      		    bitOp(SETBIT, this_row, this_col);
	      		    break;
	    		case Button2Mask:	/* middle button down */
	      		    bitOp(COMBIT, this_row, this_col);
	      		    break;
	    		case Button3Mask:	/* right button down */
	      		    bitOp(CLEARBIT, this_row, this_col);
	      		    break;
		    }
      		    last_row = this_row;
      		    last_col = this_col;
		}
      		break;
  
	    default:
		break;
	}
    }
}

/*
 * convert mouse x/y to row/col and check for pixel in bounding box
 * return -1 if illegal.
 */
whatSquare(x, y, prow, pcol)
int *prow, *pcol;
{
	int row, col;

	row = y / (PIXSIZE+grid);
	if (row >= charp->bbx.h) return -1;
	col = x / (PIXSIZE+grid);
	if (col >= charp->bbx.h) return -1;
	*prow = row;
	*pcol = col;
	return 0;
}

/*
 * set/clear/complement a pixel in current character - row/col must be checked
 * elsewhere
 */
int bitmask[] = { 8, 4, 2, 1 };

bitOp(what, row, col) {
	char *rowbits;
	int bits;
	char fourbits[2];

	rowbits = charp->rows[row];
	/*
	 * must we extend bitmap-line ?
	 */
	if (col >= strlen(rowbits)*4) {
	    char *tmp;

	    tmp = malloc(col/4 + 2);
	    strcpy(tmp, rowbits);
	    while (strlen(tmp) < (col/4 + 1))
		strcat(tmp, "0");
	    charp->rows[row] = tmp;
	    free(rowbits);
	    rowbits = tmp;
	};
	rowbits += col/4;
	fourbits[0] = *rowbits;
	fourbits[1] = '\0';
	sscanf(fourbits, "%x", &bits);
	switch (what) {
	    case SETBIT:
		bits |= bitmask[col % 4];
		break;
	    case CLEARBIT:
		bits &= ~bitmask[col % 4];
		break;
	    case COMBIT:
		bits ^= bitmask[col % 4];
		break;
	}
	sprintf(fourbits, "%x", bits);
	*rowbits = fourbits[0];
	if (bits & bitmask[col % 4]) 
	    XFillRectangle(dpy, editWin, gc, 
			col*(PIXSIZE+grid), row*(PIXSIZE+grid), 
			PIXSIZE, PIXSIZE);
	else
	    XClearArea(dpy, editWin, 
			col*(PIXSIZE+grid), row*(PIXSIZE+grid), 
			PIXSIZE, PIXSIZE, 0);

	/*
	 * redraw bounding box marker if scratched
	 */
	if (row == charp->bbx.h)
	    XDrawLine(dpy, editWin, gc,
		        0, row*(PIXSIZE+grid), 
		        edit_width, row*(PIXSIZE+grid));

	if (col == charp->bbx.w)
            XDrawLine(dpy, editWin, gc,
		        col*(PIXSIZE+grid), 0,
		        col*(PIXSIZE+grid), edit_height);
}

nextCharacter() {
	if ((charIndex+1) < font.nchars) {
	    charp++;
	    charIndex++;
	    redrawEditWindow();
	    redrawCharinfoWindow();
	}
}

prevCharacter() {
	if (charIndex > 0) {
	    charp--;
	    charIndex--;
	    redrawEditWindow();
	    redrawCharinfoWindow();
	}
}

redrawEditWindow() {
	int row, col;
	char *rowbits;
	char fourbits[2];
	int bits;

	XClearWindow(dpy, editWin);
	if (grid) {
	    XSetForeground(dpy, gc, borderPixel);
	    for (row=1; row<font.maxbbx.h; row++) {
	        XDrawLine(dpy, editWin, gc,
			    0, row*(PIXSIZE+grid)-1, 
			    edit_width, row*(PIXSIZE+grid)-1);
		if (row == charp->bbx.h)
	            XDrawLine(dpy, editWin, gc,
			    0, row*(PIXSIZE+grid), 
			    edit_width, row*(PIXSIZE+grid));
	    }

	    for (col=1; col<font.maxbbx.w; col++) {
	        XDrawLine(dpy, editWin, gc,
			    col*(PIXSIZE+grid)-1, 0,
			    col*(PIXSIZE+grid)-1, edit_height);
		if (col == charp->bbx.w)
	            XDrawLine(dpy, editWin, gc,
			        col*(PIXSIZE+grid), 0,
			        col*(PIXSIZE+grid), edit_height);
	    }

	    XSetForeground(dpy, gc, foregroundPixel);
	}

	fourbits[1] = '\0';
	for (row=0; row<charp->nrows; row++) {
	    rowbits = charp->rows[row];
	    for (col=0; rowbits[0] || (col % 4); col++) {
		if (col % 4 == 0) {
	            fourbits[0] = rowbits[0];
	            sscanf(fourbits, "%x", &bits);
		    rowbits++;
		}
		if (bits & bitmask[col % 4]) 
		    XFillRectangle(dpy, editWin, gc, 
				col*(PIXSIZE+grid), row*(PIXSIZE+grid), 
				PIXSIZE, PIXSIZE);
	    }
	}
}

redrawPrevWindow() {
	redrawButtonInWindow(prevButton, prevWin, prev_width, prev_height);
}

redrawNextWindow() {
	redrawButtonInWindow(nextButton, nextWin, next_width, next_height);
}

redrawSaveWindow() {
	redrawButtonInWindow(saveButton, saveWin, save_width, save_height);
}

redrawQuitWindow() {
	redrawButtonInWindow(quitButton, quitWin, quit_width, quit_height);
}

redrawButtonInWindow(pixmap, win, w, h)
Pixmap pixmap;
Window win;
{
	XClearWindow(dpy, win);
	XCopyPlane(dpy, pixmap, win, gc, 0, 0, w, h, 0, 0, 1);
}

redrawInfoWindow() {
	char buffer[200];

	sprintf(buffer, "File: %s Name: '%s' %d Characters                       ", 
					fontfilename, font.fontname, font.nchars);
	XDrawImageString(dpy, infoWin, gc,
			  5, 13,
			  buffer, strlen(buffer));

}

redrawCharinfoWindow() {
	char buffer[200];

	sprintf(buffer, "Character: '%s' Code: %x                        ", 
					charp->charId, charp->encoding);
	XDrawImageString(dpy, charinfoWin, gc,
			  5, 13,
			  buffer, strlen(buffer));

}

