/*
 * Dclock.c -- a digital clock widget.
 * Author: Dan Heller <island!argv@sun.com>
 */
#include <stdio.h>
#include <X11/IntrinsicP.h>
#include <X11/Xos.h>
#include <X11/StringDefs.h>
#include "DclockP.h"

static void
    Initialize(), Resize(), Realize(), Destroy(), Redisplay(),
    timeout(), toggle_bell(), toggle_reverse_video(), toggle_scroll(),
    toggle_seconds(), show_dialog_box(), GetGC(), make_number(), show_time(),
    show_date(), scroll_time();

#define BORDER		5
#define CLOCK_WIDTH	256
#define CLOCK_HEIGHT	80
#define when		break;case
#define otherwise	break;default

static Boolean SetValues();
static int winwidth = CLOCK_WIDTH;
static int winheight = CLOCK_HEIGHT;
static Boolean false = False;
static Boolean true = True;
static double x_ratio, y_ratio;
static Pixmap old_pix[4];
static struct tm before;

extern int exit();

static char defaultTranslations[] =
    "<Key>b:		toggle-bell()		\n\
     <Key>j:		toggle-scroll()		\n\
     <Key>r:		toggle-reverse-video()	\n\
     <Key>s:		toggle-seconds()	\n\
     <Btn3Down>:	dialog-box()";

static XtActionsRec actionsList[] = {
    { "toggle-bell",		toggle_bell		},
    { "toggle-scroll",		toggle_scroll		},
    { "toggle-reverse-video",	toggle_reverse_video	},
    { "toggle-seconds",		toggle_seconds		},
    { "dialog-box",		show_dialog_box		},
};

static XtResource resources[] = {
    { XtNwidth, XtCWidth, XtRInt, sizeof(int),
	XtOffset(Widget,core.width), XtRInt, (caddr_t)&winwidth },
    { XtNheight, XtCHeight, XtRInt, sizeof(int),
	XtOffset(Widget,core.height), XtRInt, (caddr_t)&winheight },
    { XtNforeground, XtCForeground, XtRPixel, sizeof(Pixel),
        XtOffset(DclockWidget,dclock.foreground), XtRString, "Black"},
    { XtNreverseVideo, XtCReverseVideo, XtRBoolean, sizeof (Boolean),
	XtOffset(DclockWidget,dclock.reverse), XtRBoolean, (caddr_t)&false},
    { XtNscroll, XtCBoolean, XtRBoolean, sizeof (Boolean),
	XtOffset(DclockWidget,dclock.scroll), XtRBoolean, (caddr_t)&true},
    { XtNbell, XtCBoolean, XtRBoolean, sizeof (Boolean),
	XtOffset(DclockWidget,dclock.bell), XtRBoolean, (caddr_t)&false},
    { XtNseconds, XtCBoolean, XtRBoolean, sizeof (Boolean),
	XtOffset(DclockWidget,dclock.seconds), XtRBoolean, (caddr_t)&false},
    { XtNdate, XtCString, XtRString, sizeof (String),
	XtOffset(DclockWidget,dclock.date_fmt), XtRString, NULL},
    { XtNfont, XtCFont, XtRFontStruct, sizeof (XFontStruct *),
	XtOffset(DclockWidget,dclock.font), XtRString, "Fixed"},
};

DclockClassRec dclockClassRec = {
    { /* core fields */
    /* superclass		*/	&widgetClassRec,
    /* class_name		*/	"Dclock",
    /* widget_size		*/	sizeof(DclockRec),
    /* class_initialize		*/	NULL,
    /* class_part_initialize	*/	NULL,
    /* class_inited		*/	FALSE,
    /* initialize		*/	Initialize,
    /* initialize_hook		*/	NULL,
    /* realize			*/	Realize,
    /* actions			*/	actionsList,
    /* num_actions		*/	XtNumber(actionsList),
    /* resources		*/	resources,
    /* resource_count		*/	XtNumber(resources),
    /* xrm_class		*/	NULL,
    /* compress_motion		*/	TRUE,
    /* compress_exposure	*/	TRUE,
    /* compress_enterleave	*/	TRUE,
    /* visible_interest		*/	FALSE,
    /* destroy			*/	Destroy,
    /* resize			*/	Resize,
    /* expose			*/	Redisplay,
    /* set_values		*/	SetValues,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus		*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private		*/	NULL,
    /* tm_table			*/	defaultTranslations,
    /* query_geometry		*/	NULL,
    }
};

WidgetClass dclockWidgetClass = (WidgetClass) &dclockClassRec;

/* ARGSUSED */
static void
Initialize (request, new)
DclockWidget   request;
DclockWidget   new;
{
    GetGC(new);
}

static void
GetGC(w)
DclockWidget w;
{
    XGCValues  	xgcv;
    XtGCMask	gc_mask =
		    GCGraphicsExposures | GCFont | GCForeground | GCBackground;

#ifdef NOT_NOW
    if (w->dclock.reverse) {
	Pixel fg = w->dclock.foreground;
	Pixel bg = w->core.background_pixel;

	if (w->core.border_pixel == fg)
	    w->core.border_pixel = bg;
	w->dclock.foreground = bg;
	w->core.background_pixel = fg;
	xgcv.function = GXcopyInverted;        gc_mask |= GCFunction;
    } else
	xgcv.function = GXcopy;        gc_mask |= GCFunction;
#endif /* NOT_NOW */

    xgcv.font = w->dclock.font->fid;
    /*
    xgcv.foreground = BlackPixelOfScreen(XtScreen(w)));
    xgcv.background = WhitePixelOfScreen(XtScreen(w)));
    */
    xgcv.foreground = w->dclock.foreground;
    xgcv.background = w->core.background_pixel;
    xgcv.graphics_exposures = FALSE;

    w->dclock.foreGC = XtGetGC ((Widget) w, gc_mask, &xgcv);
    xgcv.foreground = w->core.background_pixel;
    xgcv.background = w->dclock.foreground;
    w->dclock.backGC = XtGetGC ((Widget) w, gc_mask, &xgcv);
}

static void
Realize (w, valueMask, attrs)
Widget w;
XtValueMask *valueMask;
XSetWindowAttributes *attrs;
{
    *valueMask |= CWBitGravity;
    attrs->bit_gravity = ForgetGravity;
    XtCreateWindow (w, InputOutput, (Visual *) CopyFromParent,
		    *valueMask, attrs);
    Resize(w);
}

static void
Destroy (w)
DclockWidget w;
{
    int n;

    XtRemoveTimeOut(w->dclock.interval_id);
    XtDestroyGC (w->dclock.foreGC);
    XtDestroyGC (w->dclock.backGC);
    for (n = 0; n < 10; n++) {
	XFreePixmap(XtDisplay(w), w->dclock.digits[n]);
	XFreePixmap(XtDisplay(w), w->dclock.tiny_digits[n]);
    }
    if (w->dclock.colon[0])
	XFreePixmap(XtDisplay(w), w->dclock.colon[0]);
    if (w->dclock.colon[1])
	XFreePixmap(XtDisplay(w), w->dclock.colon[1]);
}
 
/* ARGSUSED */
static void
Resize  (w)
DclockWidget    w;
{
    int i, digit_w, digit_h;
    Pixmap pix;
    GC gc;

    winwidth = w->core.width;
    winheight = w->core.height;

    x_ratio = (double)winwidth / CLOCK_WIDTH;
    y_ratio =  (double)winheight / CLOCK_HEIGHT;

    if (w->dclock.date_fmt)
	/* make win temporarily shorter so digits will fit on top of date */
	winheight -= w->dclock.font->ascent + w->dclock.font->descent;

    digit_w = winwidth/(4 - !w->dclock.seconds) - (int)(x_ratio*BORDER*5);
    digit_h = winheight - (int)(y_ratio * BORDER*2);
    w->dclock.digit_w = digit_w;
    w->dclock.digit_h = digit_h;

    gc = w->dclock.reverse? w->dclock.backGC : w->dclock.foreGC;

    for (i = 0; i < 10; i++) {
	/* Make the big digit */
	if (w->dclock.digits[i])
	    XFreePixmap(XtDisplay(w), w->dclock.digits[i]);
	w->dclock.digits[i] =
	      XCreatePixmap(XtDisplay(w), XtWindow(w), digit_w, digit_h,
			    DefaultDepthOfScreen(XtScreen(w)));
	make_number(w, w->dclock.digits[i], gc, i, digit_w, digit_h);

	/* make smaller version of this digit for use by "seconds" */
	if (w->dclock.tiny_digits[i])
	    XFreePixmap(XtDisplay(w), w->dclock.tiny_digits[i]);
	w->dclock.tiny_digits[i] =
	      XCreatePixmap(XtDisplay(w), XtWindow(w), digit_w/2, digit_h/2,
			    DefaultDepthOfScreen(XtScreen(w)));
	make_number(w, w->dclock.tiny_digits[i], gc, i, digit_w/2, digit_h/2);
    }
    /* The colon[0] area is blank */
    if (w->dclock.colon[0])
	XFreePixmap(XtDisplay(w), w->dclock.colon[0]);
    w->dclock.colon[0] =
	XCreatePixmap(XtDisplay(w), XtWindow(w), digit_w, digit_h,
		      DefaultDepthOfScreen(XtScreen(w)));
    if (w->dclock.reverse)
	XFillRectangle(XtDisplay(w), w->dclock.colon[0], w->dclock.foreGC,
		       0, 0, digit_w,digit_h);

    /* colon[1] area has two squares */
    if (w->dclock.colon[1])
	XFreePixmap(XtDisplay(w), w->dclock.colon[1]);
    w->dclock.colon[1] = XCreatePixmap(XtDisplay(w), XtWindow(w),
				       (int)(30*x_ratio), digit_h,
				       DefaultDepthOfScreen(XtScreen(w)));
    if (w->dclock.reverse)
	/* black background with white squres */
        XFillRectangle(XtDisplay(w), w->dclock.colon[1], w->dclock.foreGC,
		       0, 0, (int)(30*x_ratio), digit_h);
    XFillArc(XtDisplay(w), w->dclock.colon[1], gc,
	    (int)(15*x_ratio), digit_h/3, digit_w/7, digit_w/7,
	    0, 360 * 64);
    XFillArc(XtDisplay(w), w->dclock.colon[1], gc,
	    (int)(15*x_ratio), (2*digit_h)/3, digit_w/7, digit_w/7,
	    0, 360 * 64);

    /* to optimize scrolling information (see scroll_time()) */
    old_pix[0] = w->dclock.digits[0];
    old_pix[1] = old_pix[2] = old_pix[3] = 0;

    if (w->dclock.date_fmt)
	/* restore size */
	winheight += w->dclock.font->ascent + w->dclock.font->descent;
}

/* Defines to draw the (simulated) LED bars for each light in the digit */
#define TOP    (pts[0].x = 2, pts[0].y = pts[1].y = 0, pts[1].x = w-2, \
                pts[3].x = 2+6*x_ratio, pts[3].y = pts[2].y = 6*y_ratio, \
		pts[2].x = w - pts[3].x, \
		XFillPolygon(dpy, pix, gc, pts, 4, Convex, CoordModeOrigin));

#define MIDDLE (pts[0].x = 2, pts[0].y = h/2 - 1, \
		pts[1].x = 6*x_ratio, pts[1].y = h/2 - 3*y_ratio, \
                pts[2].x = w-pts[1].x, pts[2].y = pts[1].y, \
                pts[3].x = w-2, pts[3].y = h/2 - 1, \
		pts[4].x = pts[2].x, pts[4].y = h/2 + 3*y_ratio, \
		pts[5].x = pts[1].x, pts[5].y = pts[4].y, \
		XFillPolygon(dpy, pix, gc, pts, 6, Convex, CoordModeOrigin));

#define BOTTOM (pts[0].x = 2, pts[0].y = pts[1].y = h, pts[1].x = w-2, \
                pts[3].x = 6*x_ratio, pts[3].y = pts[2].y = h - 6*y_ratio, \
		pts[2].x = w - pts[3].x, \
		XFillPolygon(dpy, pix, gc, pts, 4, Convex, CoordModeOrigin));

#define T_LEFT (pts[0].x = pts[1].x = 0, pts[0].y = 2, pts[1].y = h/2-2, \
                pts[2].x = pts[3].x = 6*x_ratio, \
		pts[2].y = h/2 - 5*y_ratio, pts[3].y = 8*y_ratio, \
		XFillPolygon(dpy, pix, gc, pts, 4, Convex, CoordModeOrigin));

#define B_LEFT (pts[0].x = pts[1].x = 0, pts[0].y = h/2, pts[1].y = h, \
                pts[2].x = pts[3].x = 6*x_ratio, \
		pts[3].y = h/2 + 5*y_ratio, pts[2].y = h - 8*y_ratio, \
		XFillPolygon(dpy, pix, gc, pts, 4, Convex, CoordModeOrigin));

#define T_RIGHT (pts[0].x = pts[1].x = w, pts[0].y = 2, pts[1].y = h/2-2, \
                 pts[2].x = pts[3].x = w-6*x_ratio, \
		 pts[2].y = h/2 - 5*y_ratio, pts[3].y = 8*y_ratio, \
		 XFillPolygon(dpy, pix, gc, pts, 4, Convex, CoordModeOrigin));

#define B_RIGHT (pts[0].x = pts[1].x = w, pts[0].y = h/2, pts[1].y = h, \
                 pts[2].x = pts[3].x = w-6*x_ratio, \
		 pts[3].y = h/2 + 5*y_ratio, pts[2].y = h - 8*y_ratio, \
		 XFillPolygon(dpy, pix, gc, pts, 4, Convex, CoordModeOrigin));

static void
make_number(dw, pix, gc, n, w, h)
DclockWidget dw;
Pixmap pix;
GC gc;
int n, w, h;
{
    XPoint pts[6];
    Display *dpy = XtDisplay(dw);

    if (dw->dclock.reverse)
	XFillRectangle(dpy, pix, dw->dclock.foreGC, 0, 0, w, h);
    switch(n) {
	when 0: TOP  T_RIGHT  B_RIGHT  BOTTOM  B_LEFT  T_LEFT
	when 1: T_RIGHT  B_RIGHT
	when 2: TOP  T_RIGHT  MIDDLE  B_LEFT  BOTTOM
	when 3: TOP  T_RIGHT  MIDDLE  B_RIGHT  BOTTOM
	when 4: T_LEFT  MIDDLE  T_RIGHT  B_RIGHT
	when 5: TOP  T_LEFT  MIDDLE  B_RIGHT  BOTTOM
	when 6: T_LEFT  B_LEFT  MIDDLE BOTTOM B_RIGHT
	when 7: TOP  T_RIGHT  B_RIGHT
	when 8: T_LEFT B_LEFT MIDDLE TOP BOTTOM T_RIGHT B_RIGHT
	when 9: TOP T_LEFT T_RIGHT MIDDLE B_RIGHT
	otherwise: MIDDLE
    }
}

/* ARGSUSED */
static void
Redisplay  (w)
DclockWidget    w;
{
    Boolean save_scroll = w->dclock.scroll;
    long t;

    if (w->dclock.interval_id)
	XtRemoveTimeOut(w->dclock.interval_id);
    if (w->dclock.reverse)
	XFillRectangle(XtDisplay(w), XtWindow(w), w->dclock.foreGC,
	    0, 0, w->core.width, w->core.height);
    else
	XClearWindow(XtDisplay(w), XtWindow(w));
    before.tm_min = before.tm_hour = before.tm_wday = -1;
    old_pix[0] = w->dclock.digits[0];
    old_pix[1] = old_pix[2] = old_pix[3] = 0;
    w->dclock.scroll = FALSE;
    show_time(w);
    w->dclock.scroll = save_scroll;
    if (w->dclock.seconds)
	w->dclock.interval_id = XtAddTimeOut(1000, timeout, w);
    else {
	t = time(0);
	w->dclock.interval_id =
	    XtAddTimeOut((unsigned long)(60 - (t % 60)) * 1000, timeout, w);
    }
}

static void
show_time(w)
DclockWidget w;
{
    char buf[11];
    long t = time(0);
    register struct tm *l_time = localtime(&t);
    int digit_w = w->dclock.digit_w;
    int digit_h = w->dclock.digit_h;
    Display *dpy = XtDisplay(w);
    Window win = XtWindow(w);
    GC gc = w->dclock.reverse ? w->dclock.backGC : w->dclock.foreGC;

    (void) sprintf(buf, "%02d%02d",
	     (l_time->tm_hour) ?
	    ((l_time->tm_hour <= 12)? l_time->tm_hour: l_time->tm_hour-12): 12,
	      l_time->tm_min);

    if (l_time->tm_min != before.tm_min || l_time->tm_hour != before.tm_hour)
	scroll_time(w, buf);

    if (w->dclock.seconds) {
	XCopyArea(dpy, w->dclock.tiny_digits[l_time->tm_sec/10], win, gc, 0, 0,
		  digit_w/2, digit_h/2,
		  winwidth - 2*(digit_w/2 + (int)(BORDER*x_ratio)),
		  (int)(BORDER*y_ratio));
	XCopyArea(dpy, w->dclock.tiny_digits[l_time->tm_sec%10], win, gc, 0, 0,
		  digit_w/2, digit_h/2,
		  winwidth - digit_w/2 - (int)(BORDER*x_ratio),
		  (int)(BORDER*y_ratio));
    }

    XCopyArea(dpy, w->dclock.colon[(!w->dclock.seconds || l_time->tm_sec & 1)],
	    win, gc, 0, 0,
	    (int)(30*x_ratio), digit_h,
	    (int)(-digit_w*.75 + 2*BORDER*x_ratio) + 2*digit_w,
	    (int)(BORDER*y_ratio));

    if (w->dclock.date_fmt && before.tm_wday != l_time->tm_wday)
	show_date(w, l_time);

    if (w->dclock.bell && l_time->tm_sec == 0 &&
	(l_time->tm_min == 0 || l_time->tm_min == 30)) {
	XBell(dpy, 50);
	if (l_time->tm_min == 0)
	    XBell(dpy, 50);
    }
    before = *l_time;
}

static void
scroll_time(w, p)
DclockWidget w;
register char *p;
{
    int scroll_me[4], J = winheight - BORDER*2*y_ratio + 1;
    register int i, j, incr;
    int digit_w = w->dclock.digit_w;
    int digit_h = w->dclock.digit_h;
    Display *dpy = XtDisplay(w);
    Window win = XtWindow(w);
    GC gc = w->dclock.reverse? w->dclock.backGC : w->dclock.foreGC;
    Pixmap tmp[4];

    if (w->dclock.date_fmt)
	J -= w->dclock.font->ascent + w->dclock.font->descent;

    if ((incr = J / 30) < 1)
	incr = 1;

#define x ((int)(-digit_w*.75 + ((i+1)*BORDER + (i>1)*30)*x_ratio) + i*digit_w)
#define y (int)(BORDER * y_ratio)

    for (i = 0; i < 4; i++)    /* if pixrects don't match, scroll it */
	scroll_me[i] = ((tmp[i] = w->dclock.digits[*p++ - '0']) != old_pix[i]);
    if (!w->dclock.scroll)
	for (i = 0; i < 4; i++)
	    if (i || tmp[0] == w->dclock.digits[1])
		XCopyArea(dpy, tmp[i], win, gc, 0,0, digit_w, digit_h, x,y);
	    else
		XCopyArea(dpy, w->dclock.colon[0], win, gc,
		    0,0, digit_w,digit_h, x+5,y);
    else if (scroll_me[0] || scroll_me[1] || scroll_me[2] || scroll_me[3]) {
	for (j = 0; j <= J; j += incr)
	    for (i = 0; i < 4; i++)
		if (scroll_me[i]) {
		    if (old_pix[i])
			XCopyArea(dpy, old_pix[i], win, gc,
			    0, j, digit_w, digit_h - j, x, y);
		    if (i || tmp[i] == w->dclock.digits[1])
			XCopyArea(dpy, tmp[i], win, gc,
			    0, 0, digit_w, j, x, y + digit_h - j);
		    else
			XCopyArea(dpy, w->dclock.colon[0], win, gc,
			    0, 0, x+5, y + digit_h - j, digit_w, j);
		}
	/* the increments of the scroll may not be positioned right. */
	for (i = 0; i < 4; i++)
	    if (scroll_me[i])
		if (i || tmp[i] == w->dclock.digits[1])
		    XCopyArea(dpy, tmp[i], win, gc,
			    0, 0, digit_w, J, x, y + digit_h - J);
		else
		    XCopyArea(dpy, w->dclock.colon[0], win, gc,
			    0, 0, digit_w, j, x+5, y + digit_h - j);
    }
#undef x
#undef y
    for (i = 0; i < 4; i++)
	old_pix[i] = tmp[i];
}

static char *months[] = {
    "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul",
    "Aug", "Sep", "Oct", "Nov", "Dec"
};
static char *Months[] = {
    "January", "February", "March", "April", "May", "June", "July",
    "August", "September", "October", "November", "December"
};
static char *days[] = {
    "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat", "Sun"
};
static char *Days[] = {
    "Sunday", "Monday", "Tuesday", "Wednesday",
    "Thursday", "Friday", "Saturday", "Sunday"
};

static void
show_date(w, now)
DclockWidget w;
struct tm *now;
{
    char datestr[128];
    register char *datep = datestr, *p;
    int x;

    if (!w->dclock.date_fmt)
	return;

#if 0
    if (set_alarm)
	sprintf(datestr, "Alarm is %s.", alarm_time.tm_sec < 0? "off" : "on");
    else
#endif /* 0 */
    {
	for (p = w->dclock.date_fmt; *p; p++) {
	    if (*p != '%')
		*datep++ = *p;
	    else switch (*++p) {
		when 'M':
		    datep += strlen(strcpy(datep, Months[now->tm_mon]));
		when 'm':
		    datep += strlen(strcpy(datep, months[now->tm_mon]));
		when 'W':
		    datep += strlen(strcpy(datep, Days[now->tm_wday]));
		when 'w':
		    datep += strlen(strcpy(datep, days[now->tm_wday]));
		when 'd':
		    if (now->tm_mday >= 10)
			*datep++ = (now->tm_mday / 10 + '0');
		    *datep++ = now->tm_mday % 10 + '0';
		when 'Y':
		    *datep++ = '1', *datep++ = '9';
		    /* fall thru */
		case 'y':
		    *datep++ = now->tm_year / 10 + '0';
		    *datep++ = now->tm_year % 10 + '0';
		when '%':
		    *datep++ = *p;
		otherwise: ; /* nothing */
	    }
	}
	*datep = 0;
    }
    x = (w->core.width - XTextWidth(w->dclock.font, datestr, datep-datestr))/2;
    if (x < 2)
	x = 2;

    /* remove what was there in case the whole thing isn't overwritten */
    if (w->dclock.reverse)
	XFillRectangle(XtDisplay(w), XtWindow(w), w->dclock.foreGC,
	   0, winheight - (w->dclock.font->ascent + w->dclock.font->descent),
	   winwidth, w->dclock.font->ascent + w->dclock.font->descent);
    else
	XClearArea(XtDisplay(w), XtWindow(w),
	   0, winheight - (w->dclock.font->ascent + w->dclock.font->descent),
	   winwidth, w->dclock.font->ascent + w->dclock.font->descent, False);

    XDrawString(XtDisplay(w), XtWindow(w),
	w->dclock.reverse? w->dclock.backGC : w->dclock.foreGC,
	x, winheight - BORDER, datestr, (int)(datep - datestr));
}


static void
timeout(w, id)
DclockWidget w;
XtIntervalId *id;
{
    show_time(w);
    w->dclock.interval_id =
	XtAddTimeOut(w->dclock.seconds? 1000 : 60000, timeout, w);
}

/* ARGSUSED */
static Boolean
SetValues (current, request, new)
DclockWidget current, request, new;
{
    Boolean do_redraw = False;

    if (new->dclock.foreground != current->dclock.foreground
    ||  new->core.background_pixel != current->core.background_pixel
    ||  new->dclock.reverse != current->dclock.reverse) {
	XtDestroyGC (current->dclock.foreGC);
	XtDestroyGC (current->dclock.backGC);
	GetGC(new);
	Resize(new); /* pixmaps need to be redrawn */
	do_redraw = True;
    }
    if (new->dclock.seconds != current->dclock.seconds) {
	XtRemoveTimeOut(current->dclock.interval_id);
	Resize(new);
	do_redraw = True;
    }
    if (new->dclock.date_fmt != current->dclock.date_fmt) {
	do_redraw = True;
	before.tm_wday = -1;
    }

    return do_redraw;
}

static void
toggle_bell(w)
DclockWidget w;
{
    if (w->dclock.bell = !w->dclock.bell)
	XBell(XtDisplay(w), 50);
}

static void
toggle_scroll(w)
DclockWidget w;
{
    w->dclock.scroll = !w->dclock.scroll;
}

static void
toggle_reverse_video(w)
DclockWidget w;
{
    Arg arg;

    XtSetArg(arg, XtNreverseVideo, !w->dclock.reverse);
    XtSetValues(w, &arg, 1);
}

static void
toggle_seconds(w)
DclockWidget w;
{
    Arg arg;

    XtSetArg(arg, XtNseconds, !w->dclock.seconds);
    XtSetValues(w, &arg, 1);
}

static void
show_dialog_box(w)
Widget w;
{
    XBell(XtDisplay(w), 50);
}
