/*
.nf
 *			  COPYRIGHT 1988
 *	    MASSACHUSETTS COMPUTER CORPORATION (MASSCOMP)
 *		       WESTFORD, MASSACHUSETTS
 *			ALL RIGHTS RESERVED.
 *
 *		       Author: Richard Carling
 *
 * THE INFORMATION IN THIS SOFTWARE IS SUBJECT TO CHANGE WITHOUT NOTICE AND
 * SHOULD NOT BE CONSTRUED AS A COMMITMENT BY MASSCOMP CORPORATION.
 * MASSCOMP MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THIS SOFTWARE FOR
 * ANY PURPOSE.  IT IS SUPPLIED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *
 * IF THE SOFTWARE IS MODIFIED IN A MANNER CREATING DERIVATIVE COPYRIGHT 
 * RIGHTS, APPROPRIATE LEGENDS MAY BE PLACED ON THE DERIVATIVE WORK IN 
 * ADDITION TO THAT SET FORTH ABOVE.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that the
 * copyright notice, and this permission notice appear in 
 * supporting documentation.
 */
 
/*
 *  Parser.c -- an extensible parser for ascii display list widgets.
 *
 *  This code provides an easily extensible parser for
 *  loading ascii display lists into widgets.
 *  These concepts were presented at the Second Annual X Conference
 *  January 14, 1988  by Richard Carling.
 *  The talk was titled Pickling and Embellishing Widget.
 */

#include <stdio.h>
#include <X11/Xatom.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

#define MAX_INSTRUCTION_SIZE	200
#define FAIL	0

/*
 * This table is filled from the information contained in the ascii
 * registration file. It drives the display list interpreter.
 * It provides the mapping from index (opcode) to specific routines to handle
 * the various display list events, here only ascii load, resize and redisplay.
 */

struct opcode_table {
	char name[32];		/* opcode ascii name */
	int len;		/* length of opcode string */
	
	char asciiload_ftn[32];	/* ascii load funtion name */
	int (*asciiload)();	/* ascii load ftn pointer */
	
	char asciisave_ftn[32];	/* ascii save funtion name */
	int (*asciisave)();	/* ascii save ftn pointer */
	
	char resize_ftn[32];	/* resize function name */
	int (*resize)();	/* resize function pointer */
	
	char display_ftn[32];	/* display ftn name */
	int (*display)();	/* display ftn pointer */
	
	char size_ftn[32];	/* sizeof ftn name */
	int (*size)();		/* sizeof ftn pointer */
};

/*
 * The instructions the parser recognizes are read
 * in from an instruction registration file. The routine
 * XtReadDLRegistration() reads in this file. The routine
 * is called just once during application initialization
 * before any widgets are loaded from disk.
 */ 

#define MAX_INSTRUCTION_TABLE 100

int max_instructions = 0;
struct opcode_table instructions[MAX_INSTRUCTION_TABLE];


XtReadDLRegistration( name ) char *name;
{
	FILE *fp; int i, opcode, full_scan;
	char table_type[64], junk[64], instruction_name[128];
	int (*(XtRetrieveFtn()))(); /* pretty gross, wouldn't you say */

	/*
	 *  open file pointer
	 */
	 
	fp = fopen( name, "r" );
	if (!fp) return NULL;
	
	/* determine which type of table this is */
	
	if ( fscanf( fp, "%s %s", table_type, junk) == EOF ) goto eof;
	full_scan = 0;
	if ( !strcmp(table_type, "Editor") ) full_scan = 1;
	
	/* load registration file */
	
	for (i=0; i< MAX_INSTRUCTION_TABLE; i++) {
		
		if ( fscanf( fp, "%s", instruction_name) == EOF ) goto eof;
		if ( fscanf( fp, "%d", &opcode) == EOF ) goto eof;
		
		if (opcode > (MAX_INSTRUCTION_TABLE - 1)) {
			fprintf( stderr, "Registration table opcode size exceeded\n");
			fprintf( stderr, "    opcode: %d, maximum: %d\n", 
						opcode, MAX_INSTRUCTION_TABLE - 1 );
			exit();
		}
		if (opcode > (max_instructions-1)) max_instructions = opcode + 1;
		
		strcpy( instructions[opcode].name, instruction_name );
		
		if ( fscanf( fp,
			    "%s", instructions[opcode].asciiload_ftn) == EOF )
				goto eof;
		if (full_scan)
			if ( fscanf( fp,
			    "%s", instructions[opcode].asciisave_ftn) == EOF )
				goto eof;
		if ( fscanf( fp,
			    "%s", instructions[opcode].resize_ftn) == EOF ) 
				goto eof;
		if ( fscanf( fp,
			    "%s", instructions[opcode].display_ftn) == EOF)
				goto eof;
		if (full_scan)
			if ( fscanf( fp,
			    "%s", instructions[opcode].size_ftn) == EOF)
				goto eof;
				
		/* map function names to registered function addresses */
				
		instructions[opcode].asciiload = 
			XtRetrieveFtn( instructions[opcode].asciiload_ftn );
			
		if (full_scan) 
		    instructions[opcode].asciisave = 
			XtRetrieveFtn( instructions[opcode].asciisave_ftn );
		else instructions[opcode].asciisave = NULL;
		
		instructions[opcode].resize = 
			XtRetrieveFtn( instructions[opcode].resize_ftn );
			
		instructions[opcode].display = 
			XtRetrieveFtn( instructions[opcode].display_ftn );
			
		if (full_scan)
		    instructions[opcode].size = 
			XtRetrieveFtn( instructions[opcode].size_ftn );
		else instructions[opcode].size = NULL;
	}
    eof:
	return i;
}

/* 
 *
 *  ReadDisplaylist( fp, display_list_ptr, width, height )
 *
 *    this routine reads in a display list from the stream fp.
 *    all registered displaylist instructions are handled.
 *    The indirect pointer 'display_list_ptr' is set to the
 *    location of the display list and the dislay list size is returned.
 */ 
 
ReadDisplaylist( fp, display_list_ptr, width, height )
FILE *fp; 
unsigned short **display_list_ptr; 
int width, height;
{
	unsigned short length, size, opcode;
	unsigned short *dl_memory, *final_dl, *dlp, *dltmp;
	unsigned int dl_size;
	double fwd, fht;
	
	fwd = width; fht = height;/* these values are used to calc resize info */
	
	dl_size = 1000;  /* start with a hefty size buffer for display lists */
	dl_memory = (unsigned short *) malloc( dl_size );
	
	if (!dl_memory) {
		fprintf( stderr, "Malloc Failure in ReadDisplaylist()\n");
		exit();
	}
	
	dlp = dl_memory;
	size = 0;	/* initial display list size */
	
	while ( opcode = decode_instruction( fp, instructions ) )
	{
		/* check for unusual errors */
		if (opcode > (max_instructions - 1)) {
			fprintf( stderr, "Illegal Opcode <%d> detected\n");
			exit();
		}
		/* we first insert the opcode into the display list */
		
		*dlp = opcode;
		
		/*
		 * the registered ascii load instruction will
		 * insert the length of the instruction and
		 * fill in the rest of the instructions private data.
		 */
		
		if (instructions[opcode].asciiload)
		    (*(instructions[opcode].asciiload))(fp, dlp, fwd, fht );
		else {
			fprintf( stderr, "Unregistered opcode: %d\n", opcode );
			exit();
		}
		
		/*
		 * For now, we will ignore the use of a
		 * load_resolve routine, when REAL pickling
		 * is provided, this will be very important though.
		 */
		 
		/* we get the length and check the size */
		 
		length = *(dlp + 1);
		size += length;
		if (size >= dl_size) { /* we blew it, if we get here */
			fprintf( stderr,
				"Display list length has been exceeded\n");
			/* memory may now be corrupted, so... */
			exit();
		}
		/* increase buffer size if necessary */
		
		if (size > (dl_size - MAX_INSTRUCTION_SIZE)) { 
			dl_size *= 2;  /* double our local buffer size */
			dltmp = (unsigned short *) malloc( dl_size );
			if (!dltmp) {
				fprintf(stderr, "Malloc Failure in ReadDisplaylist()\n");
				exit();
			}
			bcopy( dl_memory, dltmp, size );
			free( dl_memory );
			dl_memory = dltmp;
			dlp = (dl_memory + (size >> 1)); /* relocate our ptr */
		}
		/* now we bump the display list ptr */
		
		/* cut length in half so increment is correct */
		
		length >>= 1;/* this MUST ALWAYS be an EVEN number (before the divide) */
			     /* 680x0 machines can't handle unaligned data formats! */
		
		dlp = dlp + length;  /* increment short ptr by (what was) byte count */
	}
	/* add the HALT */

	*dlp = 0;
	*(dlp + 1) = 4; /* not really used */
	size += 4;
	
	/* allocate final display list, free up temporary buffer */
	
	final_dl = (unsigned short *) malloc( size );
	if (!final_dl) {
		fprintf( stderr, "Malloc Failure in ReadDisplaylist()\n");
		exit();
	}
	bcopy( dl_memory, final_dl, size );
	*display_list_ptr = final_dl; /* pass back display list address */
	free( dl_memory );
	
	/* return size of our display list */
	
	return size;
}


/*
 * draw_display_list()
 *
 *  This routine generates the X calls that 
 *  represent this particular display list.
 */

draw_display_list( dpy, w, gc, display_list_ptr )
Display *dpy; Window w; GC gc;
unsigned short *display_list_ptr;
{
	int n;
	unsigned short opcode, length;
	
	/* traverse the display list until we hit a HALT instruction */

	n = 0;
	while ( opcode = *display_list_ptr )
	{
		if (instructions[opcode].display)
		    (*(instructions[opcode].display))(dpy, w, gc, display_list_ptr, &n);
		else {
			fprintf( stderr, "Unregistered opcode: %d\n", opcode );
			exit();
		}
		length = *(display_list_ptr + 1);
		display_list_ptr += (length >> 1);
	}
}

/*
 * resize_display_list()
 *
 *  This routine calls the individual handles for 
 * resizeing the display list.
 */

resize_display_list( display_list_ptr, x, y, wd, ht )
unsigned short *display_list_ptr; int x, y, wd, ht;
{
	int n;
	unsigned short opcode, length;
	double fx, fy, fwd, fht;
	
	fx = x; fy = y; fwd = wd; fht = ht;

	/* traverse the display list until we hit a HALT instruction */

	n = 0;
	while ( opcode = *display_list_ptr )
	{
		if (instructions[opcode].resize)
		   (*(instructions[opcode].resize))(display_list_ptr, fx, fy, fwd, fht );
		length = *(display_list_ptr + 1);
		display_list_ptr += (length >> 1);
	}
}

/*
 *  some simple support routines for displaylist parser
 */

lookup_instruction( str )
register char *str;
{
	register len, i;

	len = strlen( str );
	if (len <= 0) return FAIL;
	
	for ( i=0; i<max_instructions; i++ ){
		if ( strcmp( str, instructions[i].name ) == 0 )
			return i;
	}
	return FAIL;
}


#define IBUFSIZE	256

decode_instruction( fp )	
register FILE *fp;
{
	register int opcode, c;
	char buf[IBUFSIZE];
	
	while ( (c = getc( fp )) != EOF && (c==' ' || c=='\t' || c=='\n') );
	if ( c == EOF ) return EOF;
	/* absorb comment lines */
	if ( c == '#' ) {
		while ( (c = getc( fp )) != EOF && c !='\t' && c!='\n')  ;
	} else ungetc( c, fp );
	if ( strchr( "0123456789-+.", c ) ) return 0;
	if ( !get_token( fp, buf, IBUFSIZE )) return 0;
	opcode = lookup_instruction( buf );
	return opcode;
}
	

instruction_size( opcode ) unsigned short opcode;
{
	if (instructions[opcode].size) return (*(instructions[opcode].size))();
	else {
		fprintf( stderr, 
		    "instruction_size():Unregistered opcode: %d\n", opcode );
		exit();
	}
}



/*
 * save_display_list()
 *
 *  This routine writes out the ascii 
 *  representation of the display list
 */

save_ascii_display_list( fp, display_list_ptr )
FILE *fp; unsigned short *display_list_ptr;
{
	int n;
	unsigned short opcode, length;
	
	/* traverse the display list until we hit a HALT instruction */

	n = 0;
	while ( opcode = *display_list_ptr )
	{
	        printf("saving opcode: %d\n", opcode );
		if (instructions[opcode].asciisave)
		    (*(instructions[opcode].asciisave))( fp, display_list_ptr);
		else {
			fprintf( stderr, "Unregistered opcode: %d\n", opcode );
			exit();
		}
		length = *(display_list_ptr + 1);
		display_list_ptr += (length >> 1);
	}
	/* then put out a halt instruction */
	fprintf( fp, "halt\n");
}
