//---------------------------------------------------------------------------
//
// %FILE     dma.c
// %VSS-REV  $Revision: 15 $
// %CREATED  1995.12.15
// %REVISED  $Date: 4/18/97 4:02p $
// %AUTHOR   Noreen Bell
// %PROJECT  NS486SXF evaluation board software
// %PART     NS486SXF (B0+)
// %SUMMARY  Direct Memory Access module
//     
// %VSS      $Author: Miked $ $Date: 4/18/97 4:02p $ $Revision: 15 $
//
// DESCRIPTION
//
//   This file contains code for the NS486SXF DMA controller.
//
// INPUTS
//
//   Locations of registers
//
//     DMA_COMMAND - Command
//     DMA_CMRn - n=0,1 - Channel Mode Registers
//     DMA_DMRn - n=0,1 - Data Mode Registers
//     DMA_MASK - MASK
//     DMA_TCS - Terminal Count Status
//     DMA_RQS - ReQuest Status
//     DMA_CHAINn - n=0,1 - Chaining
//     DMA_CBESR - Chaining Base Empty Status Register
//     DMA_CMCSR - Chaining Mode Channel Status Register
//     DMA_CLEAR - Master Clear
//     DMA_CMASK - Clear Mask
//     DMA_IDRSR - Internal DMA Request Selection Register
//     DMA_CDRR - CPU DMA Request Register
//     DMA_BTAn - n=0-6 - Base and current Target Addr. (low 16 bits)
//     DMA_BTALPn - n=0-6 - Base and current Target Addr. Low Page
//     DMA_BTAHPn - n=0-6 - Base and current Target Addr. High Page
//     DMA_BRAn - n=0-6 - Base and current Requestor Addr. (low 16 bits)
//     DMA_BRAPn - n=0-6 - Base and current Requestor Addr. Page
//     DMA_BBCn - n=0-6 - Base and current Byte Count
//
// HISTORY
//
/*
 *
 * $History: dma.c $
 * 
 * *****************  Version 15  *****************
 * User: Miked        Date: 4/18/97    Time: 4:02p
 * Updated in $/nsdemo
 * Minor changes.   New header (comment) changes.
 * 
 * *****************  Version 14  *****************
 * User: Miked        Date: 12/04/96   Time: 4:51p
 * Updated in $/nsdemo
 * Fixed | operator where || intended.  Added paranthethis around
 * statement for clarity.
 * 
 * *****************  Version 13  *****************
 * User: Miked        Date: 12/04/96   Time: 3:56p
 * Updated in $/nsdemo
 * Eliminated an unused variable.  Added some paranthethis to clarify
 * code.
 * 
 * *****************  Version 12  *****************
 * User: Miked        Date: 8/06/96    Time: 11:59a
 * Updated in $/nsdemo
 * Version 1.4.  Maintainance release.  See README.TXT for info.
 * 
 * *****************  Version 11  *****************
 * User: Miked        Date: 7/23/96    Time: 2:25p
 * Updated in $/nsdemo
 * Maintainance release.  README.TXT describes changes.
 * 
 * *****************  Version 10  *****************
 * User: Miked        Date: 7/16/96    Time: 11:54a
 * Updated in $/nsdemo
 * Updated for rev C0 release.
 * 
 * *****************  Version 9  *****************
 * User: Miked        Date: 6/26/96    Time: 3:07p
 * Updated in $/nsdemo
 * Added DMA_ prefix to a bunch of #defines used in DMA structures.  The
 * one called "Set" in particular was interfering with other modules when
 * DMA.H was included before PIT.H.  The names were too generic to expect
 * them not to be steped on.
 * 
 * *****************  Version 8  *****************
 * User: Miked        Date: 6/26/96    Time: 2:39p
 * Updated in $/nsdemo
 * Added a Clocks per transfer field to the DMA Transfer structure to
 * support
 * different DMA speeds.  (LCD needs fast DMA, Memory to Memory
 * DMA needs slow.)
 * 
 * *****************  Version 7  *****************
 * User: Miked        Date: 6/26/96    Time: 11:10a
 * Updated in $/nsdemo
 * Changed name of IDRSR value to DMA_V_IDRSR
 * 
 * *****************  Version 6  *****************
 * User: Miked        Date: 5/03/96    Time: 2:50p
 * Updated in $/nsdemo
 * Maintainence release.
 * 
 * *****************  Version 5  *****************
 * User: Noreen       Date: 4/18/96    Time: 10:48a
 * Updated in $/nsdemo
 * Updated comments for release version
 * 
 * *****************  Version 4  *****************
 * User: Miked        Date: 4/17/96    Time: 1:28p
 * Updated in $/nsdemo
 * Removed DMA chaining and modified so that MemToMem writes to a buffer
 * in DRAM.
 * 
 * *****************  Version 3  *****************
 * User: Miked        Date: 4/12/96    Time: 3:07p
 * Branched in $/nsdemo
 * NS Demo
 * 
 * *****************  Version 2  *****************
 * User: Noreen       Date: 4/12/96    Time: 3:00p
 * Updated in $/nstest
 * Updated headers for VSS
 *
 */
//
// COPYRIGHT
//
//      (c) 1995, 1996, 1997 National Semiconductor Corporation
//
// NOTES
//
//   The NS486SXF DMA defines a write transfer as a transfer from the
//   target address in RAM to the DMA requesting device.  A read transfer is
//   from the DMA requesting device to the target address in RAM.
//   This is the opposite of the Intel 8237A, which defines a write
//   transfer as from I/O (requestor) to memory, and a read as the
//   opposite.  The NS486SXF naming refers to the point of view of the
//   DMA controller (it is writing to the DMA device or reading from the
//   DMA device).  The 8237 apparently names the transfers from the point
//   of view of the DMA device.
//
//---------------------------------------------------------------------------

#include "dma.h"

// Note: For the Address Register defintions, the Base registers are
// write only and the Current registers are read only, which allows them
// to occupy the same I/O locations without conflict. For this reason only
// one registers is defined for each set.

// Target Address (Bits 15-0) registers
PRIVATE WORD const TargetAddr[] = { DMA_BTA0, DMA_BTA1, DMA_BTA2,
		  	                        DMA_BTA3, DMA_BTA4, DMA_BTA5 };

// Target Address Low Page (Bits 23-16) registers
PRIVATE BYTE const LPTargetAddr[] = { DMA_BTALP0, DMA_BTALP1, DMA_BTALP2,
			                          DMA_BTALP3, DMA_BTALP4, DMA_BTALP5 };

// Target Address High Page (Bits 31-24) registers
PRIVATE BYTE const HPTargetAddr[] = { DMA_BTAHP0, DMA_BTAHP1, DMA_BTAHP2,
			                          DMA_BTAHP3, DMA_BTAHP4, DMA_BTAHP5 };

// Requestor Address (Bits 15-0) registers
PRIVATE WORD const ReqAddr[] = { DMA_BRA0, DMA_BRA0 ,DMA_BRA2, 
                                 DMA_BRA3, DMA_BRA4 };

// Requestor Address (Bits 26-16) registers
PRIVATE WORD const UpperReqAddr[] = { DMA_BRAP0, DMA_BRAP0, DMA_BRAP2, 
                                      DMA_BRAP3, DMA_BRAP4 };

// Base Byte Count registers
PRIVATE WORD const ByteCount[] = { DMA_BBC0, DMA_BBC1, DMA_BBC2,
			                       DMA_BBC3, DMA_BBC4, DMA_BBC5 };

PRIVATE BYTE const Channel_DMA[] = { DMA_Channel0,
			                 DMA_Channel1,
			                 DMA_Channel2,
			                 DMA_Channel3,
			                 DMA_Channel4,
			                 DMA_Channel5,
			               };

PRIVATE BYTE const Channel_DMR[] = { DMR_Channel0,
			                 DMR_Channel1,
			                 DMR_Channel2,
			                 DMR_Channel3,
			                 DMR_Channel4,
			                 DMR_Channel5,
			               };

PRIVATE BYTE const CPUDMAReq[] =   { DMA_Channel0,
                             DMA_Channel1,
                             DMA_Channel2,
			                 DMA_Channel3,
			                 DMA_Channel4,
			                 DMA_Channel5,
			               };

//---------------------------------------------------------------------------
//
// FUNCTION    DMA_Initialize
//
// INPUT       none
// OUTPUT      none
// RETURN      USHORT
//               SUCCESS - if call successful
//               FAIL - if BIU has not been configured to
//                      enable accesses to DMA
//
// DESCRIPTION  This function should be called before using the other
//              DMA functions to initialize the DMA for use.
//
//---------------------------------------------------------------------------

USHORT DMA_Initialize(void)
{

	//If DMA is not enabled in the BIU, return FAIL
	if ((IOR_BYTE(BIU_CONTROL1) & 0x20) != 0x20)
	   return FAIL;

	// Issue Master Clear Instruction, which aborts any
	// DMA in progress and puts everything in a known state.
	IOW_BYTE(DMA_CLEAR, DMA_Clear);

	// Enable the Controller
	IOW_BYTE(DMA_COMMAND, DMA_Command_Init);

	// Map Internal DMA Requests
	IOW_BYTE(DMA_IDRSR, DMA_V_IDRSR);

	return SUCCESS;
}

//---------------------------------------------------------------------------
//
// FUNCTION    DMA_Status
//
// INPUT       USHORT DMAChannel - DMA Channel to get status on
// OUTPUT      DMA_STATUS *pStatus - returns a pointer to the structure
//                                   which holds the status.
// RETURN      USHORT
//               SUCCESS - if status returned
//               INVALID_PARAMETER - if non existant DMAChannel specified
//               INVALID_PARAMETER - if DMA_STATUS * is null ("invalid")
//
// DESCRIPTION
//      This function returns a pointer to a structure which holds status
//      based on the channel information was requested on. The structure
//      includes the current addresses, the current byte count, boolean values
//      as to whether or not a terminal count has occurred, and whether a DMA
//      request is pending.
//
// NOTES
//
//      typedef struct {
//              BOOL PriorityType             Fixed=0, Rotating=1
//
//              BOOL ControllerEnable         Disabled=0, Enabled=1
//
//              USHORT LowestPriority         0,1,2,3,4,5
//
//              BOOL TCStatus                 TC_NotReached=0, TC_Reached=1
//
//              BOOL RequestStatus;           RequestNotPending=0, RequestPending=1
//
//              BOOL CPUDMAReg                Clear=0,Set=1
//
//              ULONG CurrentTargetAddress
//
//              ULONG CurrentRequestorAddress
//
//              USHORT CurrentByteCount
//
//              BOOL Mask;                    ChannelEnable=0,ChannelDisable=1
//
//              BOOL ChainingMode             NotEmpty=0, Empty=1
//
//       } DMA_STATUS;
//
//---------------------------------------------------------------------------

USHORT DMA_Status(USHORT DMAChannel, DMA_STATUS * pStatus)
{
	ULONG Address;

	if(pStatus == NULL)
	   return INVALID_PARAMETER;

	//Check for Valid Channel
	if(DMAChannel > MAX_DRQ)
	   return INVALID_PARAMETER;

	if( ( IOR_BYTE(DMA_COMMAND) & (1 << 0) ) == 0 )
	   pStatus->ControllerEnable = DMA_Disable;
	else
	   pStatus->ControllerEnable = DMA_Enable;

	if( ( IOR_BYTE(DMA_COMMAND) & (1 << 1) ) == 0)
	   pStatus->PriorityType = DMA_Fixed;
	else
	   pStatus->PriorityType = DMA_Rotating;


	switch (IOR_BYTE(DMA_COMMAND) >> 2)
	{
	    case 0:
	      pStatus->LowestBits = 0;
	      break;

	    case 1:
	      pStatus->LowestBits = 1;
	      break;

	    case 2:
	      pStatus->LowestBits = 2;
	      break;

	    case 3:
	      pStatus->LowestBits = 3;
	      break;

	    case 4:
	      pStatus->LowestBits = 4;
	      break;

	    case 5:
	    default:
	      pStatus->LowestBits = 5;
	      break;
	}

	// Get Terminal Count Status
	if ((IOR_BYTE(DMA_TCS) & (1 << DMAChannel)) == 0)
	    pStatus->TCStatus = DMA_TC_NotReached;
	else
	    pStatus->TCStatus = DMA_TC_Reached;

	// Get Request Status
	if ((IOR_BYTE(DMA_RQS) & (1 << DMAChannel)) == 0)
	    pStatus->RequestStatus = DMA_RequestNotPending;
	else
	    pStatus->RequestStatus = DMA_RequestPending;

	//Get CPU DMA Request Status
	if ((IOR_BYTE(DMA_CDRR) & (1 << DMAChannel)) == 0)
	    pStatus->CPUDMAReg = DMA_Clear;
	else
	    pStatus->CPUDMAReg = DMA_Set;

	// Get Current Target Address
	Address = IOR_WORD(TargetAddr[DMAChannel]);
	Address |= (IOR_BYTE(LPTargetAddr[DMAChannel]) << 16);
	Address |= (IOR_BYTE(HPTargetAddr[DMAChannel]) << 24);
	pStatus->CurrentTargetAddress = Address;

	//Memory to Memory Transfers can only be performed
	//on channels 0,2,3 and 4. Only get requestor
	//address for these channels
	if( (DMAChannel != 1) || (DMAChannel != 5) )
	{
	   Address = IOR_WORD(ReqAddr[DMAChannel]);
	   Address |= (IOR_WORD(UpperReqAddr[DMAChannel]) << 16);
	   pStatus->CurrentRequestorAddress = Address;
	}

	// Get Current Byte Count
	pStatus->CurrentByteCount = IOR_WORD(ByteCount[DMAChannel]);

	// Get Value of Mask Register
	if ((IOR_BYTE(DMA_MASK) & (1 << DMAChannel )) == 0)
	    pStatus->Mask = DMA_ChannelEnable;
	else
	    pStatus->Mask = DMA_ChannelDisable;

	//Get Chaining Mode Status
	if((IOR_BYTE(DMA_CMCSR) & (1 << DMAChannel)) != 0)
	{
	    //Channels Base Empty Status Register
	    if((IOR_BYTE(DMA_CBESR) & (1 << DMAChannel)) == 0)
		pStatus->ChainingMode = DMA_NotEmpty;
	    else
		pStatus->ChainingMode = DMA_Empty;
	}

	return SUCCESS;
}

//---------------------------------------------------------------------------
//
// FUNCTION     DMA_Cancel()
//
// INPUT        USHORT DMAChannel - channel to mask out.
//
// OUTPUT       none
//
// RETURN       USHORT
//              SUCCESS if call is successful
//              INVALID_PARAMETER if channel is not a valid value.
//
// DESCRIPTION
//      This function cancels a DMA operation that has already started, by
//      masking out the DRQ for the given channel.  This will prevent the 
//      servicing of further DMA requests on this channel.
//
//-------------------------------------------------------------------------

USHORT DMA_Cancel(USHORT DMAChannel)
{

	// Check for valid channel
	if(DMAChannel > MAX_DRQ)
	   return INVALID_PARAMETER;

	IOW_BYTE(DMA_MASK, (IOR_BYTE(DMA_MASK) | (1 << DMAChannel)) );

	return SUCCESS;
}

//---------------------------------------------------------------------------
//
// FUNCTION     CMR_Set()
//
// INPUT        USHORT DMAChannel - channel
//              DMA_TRANSFER * pTransfer - structure for transfer
//
// OUTPUT       none
//
// RETURN       BYTE
//              ChannelModeRegister value that corresponds to input information
//
// SIDE EFFECT
//
// DESCRIPTION
//      This function is called by DMA_Transfer to determine the value
//      to be written to the Channel Mode Register.  This value is based upon
//      the information passed to the DMA_Transfer function.  This function
//      should be considered as part of that function and is only seperated to 
//      improve readibility and maintainability.
//
// NOTES
//    See DMA_Transfer NOTES section for description of INPUT structure.
//
//----------------------------------------------------------------------------

PRIVATE BYTE CMR_Set(USHORT DMAChannel, DMA_TRANSFER * pTransfer)
{

	// Set Channel Select
	BYTE CMRValue = Channel_DMA[DMAChannel];

	// Set Data Transfer Mode
	if(pTransfer->Mode == TransferMode_Demand)
	   CMRValue |= CMR_DemandMode;
	else if(pTransfer->Mode == TransferMode_Single)
	   CMRValue |= CMR_SingleMode;
	else if(pTransfer->Mode == TransferMode_Block)
	   CMRValue |= CMR_BlockMode;
	else
	   CMRValue |= CMR_CascadeMode;

	// Set mode of TC/~EOP pin
	if(pTransfer->TC_EOP)
	   CMRValue |= CMR_EOP;
	else
	   CMRValue |= CMR_TC;

	// Set Requestor Device Type
	if(pTransfer->IO_MEM)
	   CMRValue |= CMR_RequestorMem;
	else
	   CMRValue |= CMR_RequestorIO;

	// Set Transfer Type
	if(pTransfer->Direction)
	   CMRValue |= CMR_TransferWrite;
	else
	   CMRValue |= CMR_TransferRead;

	// Set Autoinitialization
	if(pTransfer->AutoInit)
	   CMRValue |= CMR_AutoInitEnable;
	else
	   CMRValue |= CMR_AutoInitDisable;

	return CMRValue;
}

//---------------------------------------------------------------------------
//
// FUNCTION     DMR_Set()
//
// INPUT        USHORT channel - channel
//              DMA_TRANSFER * pTransfer - structure for transfer
//
// OUTPUT       none
//
// RETURN       BYTE
//              Data Mode Register value that corresponds to input information
//
// SIDE EFFECT
//
// DESCRIPTION
//      This function is called by DMA_Transfer to determine the value
//      to be written to the Data Mode Register.  This value is based upon
//      the information passed to the DMA_Transfer function.  This function
//      should be considered as part of that function and is only seperated to 
//      improve readibility and maintainability.
//
// NOTES
//    See DMA_Transfer NOTES section for description of INPUT structure.
//
//---------------------------------------------------------------------------

PRIVATE BYTE DMR_Set(USHORT DMAChannel, DMA_TRANSFER * pTransfer)
{
	// Set Channel Select
	BYTE DMRValue = Channel_DMR[DMAChannel];

	//Set Requestor Direction
	if(pTransfer->RequestorDirection)
	   DMRValue |= DMR_RequestorIncrement;
	else
	   DMRValue |= DMR_RequestorDecrement;

	//Set Target Direction
	if(pTransfer->TargetDirection)
	   DMRValue |= DMR_TargetIncrement;
	else
	   DMRValue |= DMR_TargetDecrement;

	//Set Clock per transfer for maximum
	//Note:This bit is not part of the transfer structure
	switch ( pTransfer->Clocks )
	{
	  case Clocks2 :
	    DMRValue |= DMR_Clocks2;
	    break;
	  case Clocks4 :
	    DMRValue |= DMR_Clocks4;
	    break;
	  case Clocks6 :
	    DMRValue |= DMR_Clocks6;
	    break;
	  case Clocks8 :
	    DMRValue |= DMR_Clocks8;
	    break;
	  case Clocks10 :
	    DMRValue |= DMR_Clocks10;
	    break;
	  case Clocks12 :
	    DMRValue |= DMR_Clocks12;
	    break;
	  case Clocks14 :
	    DMRValue |= DMR_Clocks14;
	    break;
	  case Clocks16 :
	    DMRValue |= DMR_Clocks16;
	    break;
	  default:
	    DMRValue |= DMR_Clocks16;
	}

	//Set Bus Size
	if(pTransfer->BusSize)
	   DMRValue |= DMR_Width16;
	else
	   DMRValue |= DMR_Width8;

	return DMRValue;
}

//---------------------------------------------------------------------------
//
// FUNCTION    DMA_Transfer
//
// INPUT       DMA_TRANSFER Transfer - structure for transfer
// OUTPUT      none
// RETURN      USHORT
//               SUCCESS - if status returned
//               INVALID_PARAMETER - if invalid DMA channel is given
//
// SIDE EFFECT none
//
// DESCRIPTION
//      This function is passed a structure which it uses to perform a DMA operation.
//      The structure includes elements such as the channel to use, the mode of the
//      channel, boolean values for programming the channel and data mode registers.
//      and information pertaining to the Chaining Mode registers.
//
// NOTES
//      typedef struct {
//              enum DATA_TRANSFER_MODE Mode
//              BOOL TC_EOP                   -TC=0,EOP=1
//
//              BOOL IO_MEM                   -IO=0,MEM=1
//
//              BOOL Direction                -Read=0,Write=1
//
//              BOOL AutoInit                 -Disabled=0,Enabled=1
//
//              USHORT Channel                -0,1,2,3,4,5
//
//              BOOL RequestorDirection       -Decrement=0,Increment=1
//
//              BOOL TargetDirection          -Decrement=0,Increment=1
//
//              enum DMA_CLOCKS_PER_TRANSFER Clocks
//
//              BOOL BusSize                  -8bit=0,16bit=1
//
//              USHORT CPUDMARequest          -NoAction=0,Set=1,Clear=2
//
//              ULONG TargetAddress
//
//              ULONG RequestorAddress
//
//              USHORT TransferByteCount
//     } DMA_TRANSFER;
//
//      enum DATA_TRANSFER_MODE {
//              TransferMode_Demand,
//              TransferMode_Single,
//              TransferMode_Block,
//              TransferMode_Cascade
//      };
//
//---------------------------------------------------------------------------

USHORT DMA_Transfer(DMA_TRANSFER Transfer)
{

	BYTE  DMRValue, CMRValue;
	USHORT DMAChannel;
	DMA_TRANSFER * pTransfer = &Transfer;

	DMAChannel = Transfer.Channel;

	if(DMAChannel > MAX_DRQ)
	    return INVALID_PARAMETER;

	//Disable DMA channel
	DMA_Cancel(DMAChannel);

	// Program channel mode register
	CMRValue = CMR_Set(DMAChannel, pTransfer);

	if (DMAChannel > SPLIT_DRQ)
	    IOW_BYTE(DMA_CMR1, CMRValue);
	else
	    IOW_BYTE(DMA_CMR0, CMRValue);

	// Program Data Mode register
	DMRValue = DMR_Set(DMAChannel, pTransfer);

	if(DMAChannel > SPLIT_DRQ)
	   IOW_BYTE(DMA_DMR1, DMRValue);
	else 
	   IOW_BYTE(DMA_DMR0, DMRValue);

	// Program Base Target Address
	IOW_WORD(TargetAddr[DMAChannel], (WORD) Transfer.TargetAddress);
	IOW_BYTE(LPTargetAddr[DMAChannel], (BYTE) ( Transfer.TargetAddress >> 16 ) );
	IOW_BYTE(HPTargetAddr[DMAChannel], (BYTE) ( Transfer.TargetAddress >> 24 ) );

	// Program Base Byte count
	IOW_WORD(ByteCount[DMAChannel], Transfer.TransferByteCount);

	//Memory to Memory Transfer Only
	if((CMRValue & CMR_RequestorMem) == CMR_RequestorMem)
	{   // Check for valid Channel for memory to memory transfer
	    if( (DMAChannel != 1) || (DMAChannel != 5) )
	    {   // Program Base Requestor Address (26 bits)
	       IOW_WORD(ReqAddr[DMAChannel], (WORD) Transfer.RequestorAddress);
	       IOW_WORD(UpperReqAddr[DMAChannel],  (WORD) ( Transfer.RequestorAddress >> 16 ) );
	    }
	    else
	       return INVALID_PARAMETER;
	}

	// Disable Chaining Mode

	if(DMAChannel > SPLIT_DRQ)
	   IOW_BYTE(DMA_CHAIN1, (Channel_DMA[DMAChannel] | CHAIN_Disable));
	else
	  IOW_BYTE(DMA_CHAIN0, (Channel_DMA[DMAChannel] | CHAIN_Disable));

	//Set or Clear CPU DMA Request Register only if function is
	//new DMA transfer and not in demand mode
	if(Transfer.CPUDMARequest && (Transfer.Mode != TransferMode_Demand))
	{
	   if(Transfer.CPUDMARequest == DMA_Set)
	   {
	      //Clear Register before setting
	      IOW_BYTE(DMA_CDRR, (IOR_BYTE(DMA_CDRR) & ~(1 << DMAChannel)) );
	      IOW_BYTE(DMA_CDRR, (IOR_BYTE(DMA_CDRR) | (1 << DMAChannel)) );
	   }
	   else if (Transfer.CPUDMARequest == DMA_Clear)
	      IOW_BYTE(DMA_CDRR, (IOR_BYTE(DMA_CDRR) & ~(1 << DMAChannel)) );
	   else
	      IOW_BYTE(DMA_CDRR, (IOR_BYTE(DMA_CDRR) & ~(1 << DMAChannel)) );
	}

	//Enable DMA Channel
	IOW_BYTE(DMA_MASK, ( (~(1 << DMAChannel)) & IOR_BYTE(DMA_MASK)) );

	return SUCCESS;
}

//---------------------------------------------------------------------------
//
// FUNCTION    DMA_ClearTC
//
// INPUT       none
// OUTPUT      none
// RETURN      USHORT
//               SUCCESS - if register cleared
//               INVALID_PARAMETER - if invalid DMA channel is given
//
// DESCRIPTION
//      This function clears the terminal count status register and
//      should be called on completion of a DMA transfer.
//
//---------------------------------------------------------------------------

USHORT DMA_ClearTC(USHORT DMAChannel)
{
	// Check for valid channel
	if(DMAChannel > MAX_DRQ)
	   return INVALID_PARAMETER;

	//Clear TC Status Register bit based on Channel input
	IOW_BYTE(DMA_TCS, (IOR_BYTE(DMA_TCS) | (1 << DMAChannel)));
	return SUCCESS;
}

//---------------------------------------------------------------------------
// END       dma.c
//---------------------------------------------------------------------------
