//---------------------------------------------------------------------------
//
// %FILE     pcmcia.c
// %VSS-REV  $Revision: 11 $
// %CREATED  1996.04.08
// %REVISED  $Date: 4/18/97 4:14p $
// %AUTHOR   Noreen Bell
// %PROJECT  NS486SXF evaluation board software
// %PART     NS486SXF (B0+)
// %SUMMARY  PCMCIA module
//     
// %VSS      $Author: Miked $ $Date: 4/18/97 4:14p $ $Revision: 11 $
//
// INPUTS
//
//      Register locations.
//
//   PCM_BASE - can be defined as PCM_BASE1 or PCM_BASE2 to indicate
//              where the PCMCIA controller is mapped in the part.
//              The possible values are:
//
//              PCM_BASE1       0x003E0
//              PCM_BASE2       0x003E2
//
//   PCM_INDEX  PCM_BASE + 0x00  Index Register
//   PCM_DATA   PCM_BASE + 0x01  Data Register
//
// HISTORY
//
/*
 *
 * $History: pcmcia.c $
 * 
 * *****************  Version 11  *****************
 * User: Miked        Date: 4/18/97    Time: 4:14p
 * Updated in $/nsdemo
 * Minor functional changes.   New header (comment) changes.
 * 
 * *****************  Version 9  *****************
 * User: Miked        Date: 12/04/96   Time: 4:00p
 * Updated in $/nsdemo
 * Added a typecast and a set of paranthethis to clarify code.
 * 
 * *****************  Version 8  *****************
 * User: Miked        Date: 8/06/96    Time: 11:59a
 * Updated in $/nsdemo
 * Version 1.4.  Maintainance release.  See README.TXT for info.
 * 
 * *****************  Version 7  *****************
 * User: Miked        Date: 7/23/96    Time: 2:25p
 * Updated in $/nsdemo
 * Maintainance release.  README.TXT describes changes.
 * 
 * *****************  Version 6  *****************
 * User: Miked        Date: 7/16/96    Time: 11:54a
 * Updated in $/nsdemo
 * Updated for rev C0 release.
 * 
 * *****************  Version 5  *****************
 * User: Noreen       Date: 5/07/96    Time: 3:30p
 * Updated in $/nstest
 * Implemented code to support interrupts based on card removal.
 * Added PCMCIA_Status function.
 */
//
// COPYRIGHT
//
//      (c) 1996, 1997 National Semiconductor Corporation
//
//---------------------------------------------------------------------------

#include "pcmcia.h"

//---------------------------------------------------------------------------
//
// FUNCTION    PCMCIA_Enable
//
// INPUT       none
// OUTPUT      none
// RETURN      USHORT
//               SUCCESS if call successful
//               FAIL if PCMCIA card is not found
//
// DESCRIPTION
//       This function enables the NS486 PCIC Controller, maps a PCMCIA
//       SRAM card into the System Memory Address space and checks
//       for a certain signature at a known location. It is meant only as
//       an example of configuring PCMCIA on the NS486SXF Evaluation Board
//       and does not provide a solution for PCMCIA applications.
//
//
// NOTES
//   It is capable of detecting any PCMCIA Type II card.
//
//---------------------------------------------------------------------------

USHORT PCMCIA_Enable(void)
{
	BYTE value;
	ULONG StartAddress, StopAddress, Offset;

  // If PCMCIA is not enabled in the BIU, return FAIL

  if ( ( IOR_BYTE( BIU_CONTROL2 ) & 0x02 ) != 0x02 )
    return FAIL;

    // Address values
	StartAddress = WINDOW_START;
	StopAddress = WINDOW_STOP;
	Offset = (ULONG) WINDOW_OFFSET;

	// Program Memory Base Address Register
	IOW_BYTE(BIU_PBMAR, StartAddress >> 24);

	// Program Clock Selection Register
	IOW_BYTE(BIU_PCLOCK, CPUCLK4);

	// Enable Card Power
	IOW_BYTE(PCM_INDEX, PCM_POWER_CTRL);
	IOW_BYTE(PCM_DATA, PCM_POWER_V_CTRL);

	// Reset Card / No Interrupts Selected
	IOW_BYTE(PCM_INDEX, PCM_INT3);
	IOW_BYTE(PCM_DATA, PCM_NOIRQ);

	// Reset Inactive
	IOW_BYTE(PCM_INDEX, PCM_INT3);
	IOW_BYTE(PCM_DATA, PCM_RESET_INACTIVE);

	//Enable Card Detect
	IOW_BYTE(PCM_INDEX, PCM_GENERAL_CTRL);
	IOW_BYTE(PCM_DATA, PCM_CDE);

	// Setup IRQ12 to generate an interrupt
	// on detection of a card change (based on CD1,CD2).
	IOW_BYTE(PCM_INDEX, PCM_INT5);
	IOW_BYTE(PCM_DATA, PCM_CDIRQ);

    // Brief Delay
    PIT_Delay(1,1);     // 1ms delay

	// Card Detection
	IOW_BYTE(PCM_INDEX, PCM_INTERFACE_STATUS);
	value = IOR_BYTE(PCM_DATA);
    if((value & 0x0c) != 0x0c) return FAIL;

	// Address Mapping
	IOW_BYTE(PCM_INDEX, PCM_MEM0_STARTLB);  // Start Address 0 Low Byte
	IOW_BYTE(PCM_DATA, (StartAddress >> 12) & 0xff);

	IOW_BYTE(PCM_INDEX, PCM_MEM0_STARTHB);  // Start Address 0 High Byte
	IOW_BYTE(PCM_DATA, (StartAddress >> 20) & 0x0f);

	IOW_BYTE(PCM_INDEX, PCM_MEM0_STOPLB);   // Stop Address 0 Low Byte
	IOW_BYTE(PCM_DATA, (StopAddress >> 12) & 0xff);

	IOW_BYTE(PCM_INDEX, PCM_MEM0_STOPHB);   // Stop Address 0 High Byte
	IOW_BYTE(PCM_DATA, (StopAddress >> 20) & 0x0f);

	IOW_BYTE(PCM_INDEX, PCM_MEM0_OFFSETLB);    // Card Offset Address 0 Low Byte
	IOW_BYTE(PCM_DATA, (Offset >> 12) & 0xff);

	IOW_BYTE(PCM_INDEX, PCM_MEM0_OFFSETHB);    // Card Offset Address 0 High Byte
	IOW_BYTE(PCM_DATA, (Offset >> 20) & 0x3f);

	IOW_BYTE(PCM_INDEX, PCM_WINDOW_ENABLE);    // Address 0 Window Enable
	IOW_BYTE(PCM_DATA, PCM_MEM0_ENABLE);

	return SUCCESS;
}

//---------------------------------------------------------------------------
//
// FUNCTION    PCMCIA_Status
//
// INPUT/OUTPUT PCM_STATUS * pStatus - a pointer to the structure
//                                     which holds the status.
// RETURN      USHORT
//               SUCCESS - if status returned
//               INVALID_PARAMETER - if PCM_STATUS * is null
//
// DESCRIPTION
//      This function returns a pointer to a structure which contains
//      information from the Card Status Change Register. Contents
//      of the structure are outlined below in the NOTES section.
//
// NOTES
//
//      typedef struct {
//          BOOL BatteryDead;    NotDead=0, Dead=1
//
//          BOOL BatteryWarn;    NoWarn=0, Warn=1
//
//          BOOL CardChange;     NoChangeDetected=0, ChangeDetected=1
//
//          BOOL ReadyChange;    NoChangeDetected=0, ChangeDetected=1
//
//          BOOL GPIChange;      NoChangeDetected=0, ChangeDetected=1
//
//      } PCM_STATUS;
//
//---------------------------------------------------------------------------

USHORT PCMCIA_Status(PCM_STATUS *pStatus)
{
	BYTE cardval;

	if(pStatus == NULL)
	   return INVALID_PARAMETER;

	// Read the Card Status Register
	IOW_BYTE(PCM_INDEX, PCM_CARD_STATUS);
	cardval = IOR_BYTE(PCM_DATA);

	if((cardval & (1 << 0)) == 0)
	   pStatus->BatteryDead = NotDead;
	else
	   pStatus->BatteryDead = Dead;

	if((cardval & (1 << 1)) == 0)
	   pStatus->BatteryWarn = NoWarn;
	else
	   pStatus->BatteryDead = Warn;

	if((cardval & (1 << 2)) == 0)
	   pStatus->ReadyChange = ChangeNotDetected;
	else
	   pStatus->ReadyChange = ChangeDetected;

	if((cardval & (1 << 3)) == 0)
	   pStatus->CardChange = ChangeNotDetected;
	else
	   pStatus->CardChange = ChangeDetected;

	if((cardval & (1 << 4)) == 0)
	   pStatus->GPIChange = ChangeNotDetected;
	else
	   pStatus->GPIChange = ChangeDetected;

	return SUCCESS;

}

//---------------------------------------------------------------------------
// END       pcmcia.c
//---------------------------------------------------------------------------
