/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* CMOS data area control and access routines */

#include "lib.h"		/* Digital Debug Monitor */
#include "uilib.h"
#include "northbridge.h"
#include "platform.h"

#include "cmos_rtc.h"




/*----------------------------------------------------------------------*/
/* CMOS real-time clock access routines */

/* Read the CMOS clock and produce a date string in unix 'date' style */
static void CMOSGetDate( char *dbuf )
{
    RTC_t r;
    unsigned int ipl;

    ipl = swpipl(7);	 /* disable interrupts */
    cmosready();	 /* Wait until an update is just finished */
    RTC_Get( &r );
    swpipl(ipl);

    /* perform some data validation on what was read back */
    if ( r.month > 12 )	r.month = 0;		/* validation */
    if ( r.daywk > 7 )	r.daywk = 0;		/* validation */

    /* even when on a 24 Hour clock this doesn't affect things - BCD encoded */
    r.hours &= 0x7F;			/* strip off AM/PM bit (if present) */

    /* form date string */
    sprintf_dbm(dbuf, "%s %s %d %d:%02d:%02d %4d",
		DaysOfWeek + r.daywk*4, MonthsOfYear + r.month*4,
		r.daymth, r.hours, r.mins, r.secs, r.year + rtc_epoch );

    mobo_logf( LOG_INFO "DATE/TIME: reading taken: %s\n", dbuf );
}

static DBM_STATUS get_date_wrap( int a, char **b )
{
    char date[64];

    CMOSGetDate( date );
    mobo_alertf( "Date/Time settings",
		 "Date & time are set to: %s", date );

    return STATUS_SUCCESS;
}


/* parse a date string in unix 'date' style and write it to CMOS */
static DBM_STATUS CMOSSetDate( int a, char **b )
{
    char daywkstr[30], monthstr[30];
    char date[64];
    RTC_t r;
    unsigned daywk, daymth, month, hours, mins, secs, year;
    unsigned rval;
    unsigned ipl;
    static const String parse_error = "Arfle Barfle Gloop?";


    mobo_cls();

    CMOSGetDate( date );
    printf_dbm( "Date/Time is currently set to: %s\n"
		"Time is in 24-hour format.", date );


    /* get input from the user */
    mobo_goto( p_help );
    printf_dbm( "%-64s", "Please enter a new date and time [return to quit]:");
    mobo_goto( p_prompt );
    printf_dbm( Prompt );
    mobo_input( cbuf, sizeof( cbuf ) );


    /* parse the user input */
    if ( strlen( cbuf ) == 0 )			return STATUS_FAILURE;
    rval = sscanf( cbuf, "%s %s %d %d:%d:%d %d", daywkstr, monthstr, &daymth,
                &hours, &mins, &secs, &year );
    if ( rval != 7 )
    {
	mobo_alertf( parse_error, "I could only find %d parameters in '%s'",
			rval, cbuf );
	return STATUS_FAILURE;
    }

    /* lookup day of week */
    for( daywk=1; daywk<=7; daywk++ )
	if ( strcmp( daywkstr, DaysOfWeek + daywk*4 ) == 0 )
		break;
    if ( daywk > 7 ) {
	mobo_alertf( parse_error, "I don't recognise weekday '%s'", daywkstr );
	return STATUS_FAILURE;
    }
    
    /* lookup month of year */
    for( month=1; month<=12; month++ )
	if ( strcmp( monthstr, MonthsOfYear + month*4 ) == 0 )
		break;
    if ( month > 12 ) {
	mobo_alertf( parse_error, "I don't recognise month '%s'", monthstr );
	return STATUS_FAILURE;
    }
    
    /* fixup year adjustment as per current console date format (ARC or SRM) */
    year -= rtc_epoch;


    /* Read the current date settings, and overwrite with our values */
    ipl = swpipl(7);            /* disable interrupts */
    cmosready();         	/* Wait until an update is just finished */
    RTC_Get( &r );

    r.month = month;		/* overwrite with our values */
    r.daywk = daywk;
    r.daymth = daymth;
    r.hours = hours;
    r.mins = mins;
    r.secs = secs;
    r.year = year;

    RTC_Set( &r );
    swpipl(ipl);


    /* Read it back and verify */
    CMOSGetDate( date );
    mobo_alertf( "Date/Time update complete",
		 "Clock now set to: %s", date );
    return STATUS_SUCCESS;
} 



/* Change the Epoch year used to offset CMOS year byte */

static const struct { unsigned short year; String desc; } epochs[] = {
	{ EPOCH_SRM, "SRM console firmware epoch" },
	{ 2000, "SRM console firmware epoch (after 2000)" },
	{ EPOCH_ARC, "ARC/AlphaBIOS firmware epoch" },
	{ EPOCH_PC, "MS-DOS epoch [not normally used on Alpha]" }
};

#define NEPOCHS ( sizeof(epochs) / sizeof( epochs[0] ) )

static DBM_STATUS CMOSEpoch( int a, char **b )
{
    int i, newyear;
    char yearbuf[8];
    RTC_t t;

    do { 
	mobo_box( r_lrgapp, "Change Epoch year" );
	printf_dbm( "Available Epoch year options are (current %d):\r", 
		rtc_epoch );

	for( i=0; i<NEPOCHS; i++ )
	    printf_dbm( "  %d: %s (%d)\r", i+1,
		epochs[i].desc, epochs[i].year );

	printf_dbm( "  %d: Quit without changing epoch year\r"
		    "Please choose an option: ", NEPOCHS+1 );

	i = mobo_key(0) - '1';
    } while ( i<0 || i>NEPOCHS );

    if ( i == NEPOCHS )			/* request to quit? */
	return STATUS_SUCCESS;

    rtc_epoch = epochs[ i ].year;
    mobo_box( r_lrgapp, "Set current year" );
    printf_dbm( "Please enter the current year (as a 4-digit number): " );
    mobo_input( yearbuf, sizeof(yearbuf) );
    newyear = atoi( yearbuf );
    
    cmosready();
    RTC_Get( &t );
    t.year = newyear - rtc_epoch;
    RTC_Set( &t );

    mobo_logf( LOG_INFO "RTC: updated epoch year to %s (%d), current year %d\n",
	epochs[ i ].desc, rtc_epoch, newyear );
    
    return STATUS_SUCCESS;
} 

/*----------------------------------------------------------------------*/
/* Raw Dump of CMOS  */

#define MAXCMOS 128

static DBM_STATUS CMOSDump( int a, char **b )
{
    unsigned i, j;
    unsigned char cmos[MAXCMOS];
    unsigned long start, end, sum, nsec;
    unsigned int ipl;

    mobo_cls();
    ipl = swpipl(7);

    /* perform the read here, in one single burst */
    if ( cmosready() != STATUS_SUCCESS )
		printf_dbm("WARNING - clock not ticking\n");

    for ( i=0, sum=0; i < MAXCMOS; i++ ) {
	start = rpcc();				/* PCC timing */
	cmos[i] = cmosrb( i );
	end = rpcc();					/* PCC timing */
	if (end <= start)	end += 1UL << 32;	/* counter wrap */
	sum += end - start;
    }
    swpipl(ipl);

    /* calculate timing info: cycle_cnt = picosecs per clock tick */
    nsec = (primary_impure->CYCLE_CNT * sum) / (MAXCMOS * 1000); 


    printf_dbm("CMOS data dump:\n\n   ");	/* 3 sp for alignment */
    for ( i = 0; i < 16; i++ )	printf_dbm("%02X ", i);	/* column heads */

    for ( i = 0; i < MAXCMOS; i+=16 ) {
	printf_dbm("\n%02X ", i );
	for ( j=0 ; j < 16; j++ )
		printf_dbm("%02X ", cmos[i+j]);
    }

    printf_dbm("\n\n%d bytes read.  Mean access time = %d ns\n\n", 
		MAXCMOS, nsec );
    mobo_key( 0 );
    return STATUS_SUCCESS;
}

/*----------------------------------------------------------------------*/

static DBM_STATUS CMOSCount( int a, char **b )
{
    mobo_alertf("Timer ticks count", "%d timer interrupts since power-up",
		jiffies );
    return STATUS_SUCCESS;
}


/*----------------------------------------------------------------------*/
/* Test of the programmable periodic timer */

static DBM_STATUS CMOSTimer( int argc, char **b )
{
    unsigned i;
    BOOLEAN skipflag, mode_bad;
    unsigned long tmp, start, end, ticks, etime;
    unsigned char v, w, a = cmosrb( CM_STATUSA ) & 0xF0;
    int ipl;
    int accuracy;
    DBM_STATUS final_rval = STATUS_SUCCESS;

    mobo_cls();
    printf_dbm( "CMOS Timer Mode Test\n" );

    /* Enable RTC interrupts */
    mobo_logf( LOG_INFO "CMOS: Enabling timer interrupts...\n" );
    ipl = swpipl( 4 );		/* Enables timer but no devices */

#ifdef CONFIG_SHARK
    /* Shark (CS20) doesn't use the real-time clock for its timer interrupts,
     * instead relying on some other piece of logic supporting only the 
     * standard period. 
     */
#define SHARK_TIMER 976			/* Shark single timer mode in usec */

    start = end = rpcc();

    tmp = jiffies;
    while( jiffies == tmp )
	start = rpcc();

    tmp = jiffies;
    while( jiffies == tmp )
	end = rpcc();

    if ( end <= start ) 	end += 1UL << 32;	/* wraparound */
    ticks = end - start;

    etime = ( primary_impure->CYCLE_CNT * ticks) / 1000000UL;
    printf_dbm( "Shark Timer: %ld us, expected %d us\n", etime, SHARK_TIMER );

    accuracy = (etime * 100) / SHARK_TIMER;
    mode_bad = (accuracy < 98) || (accuracy > 102);
    if ( mode_bad )
    {
	mobo_alertf( "System timer inaccuracy",
		"The timer was measured at %d%% of its correct value\r"
		"(Measured %dusec, wanted %dusec)",
		accuracy, etime, SHARK_TIMER );
	final_rval = STATUS_FAILURE;
    }

#else		/* CONFIG_SHARK */

    /* Configure CMOS RTC */
    cmoswb( CM_STATUSA, CM_STATUSA__INIT );
    cmoswb(CM_STATUSB, CM_STATUSB__INIT | STATUSB_PER);

    /* For each mode the timer can be set in (1->15) */
    for ( i=1; i<16; i++ )
    {
	w = a | i;
	cmoswb( CM_STATUSA, w );
	v = cmosrb( CM_STATUSA );
	if ( v != w )				/* validation */
		printf_dbm("odd... wrote 0x%02X got back 0x%02X...\n", w, v );

	printf_dbm("Mode %d... ", i );
	tmp = jiffies;
	start = rpcc();
	skipflag = 0;
	while ( tmp == jiffies )			/* for fresh tick */
	{
	    end = rpcc();	
	    if ( end <= start )	end += 1UL << 32;
	    ticks = end - start;
	    if ( ticks  >= 1UL << 31 )			/* a sec or two */
	    {
		skipflag = 1;
		printf_dbm("No tick, skipping\n");
		break;
	    }
	}
	if ( skipflag )		continue;

	start = rpcc();
	tmp = jiffies;
	while ( tmp == jiffies );
	end = rpcc();

	if ( end <= start ) 	end += 1UL << 32;	/* wraparound */
	ticks = end - start;

	etime = ( primary_impure->CYCLE_CNT * ticks) / 1000000UL;

	accuracy = (etime * 100) / TimerModes[i-1];
	mode_bad = (accuracy < 98) || (accuracy > 102);
	if ( mode_bad )
	    final_rval = STATUS_FAILURE;

	printf_dbm( "%dusec, expected %dusec [%s]\n",
		etime, TimerModes[i-1], mode_bad ? "FAIL" : "OK" );

    }

    /* reset to default value */
    cmoswb( CM_STATUSA, CM_STATUSA__INIT );
    cmoswb( CM_STATUSB, CM_STATUSB__INIT );
#endif

    printf_dbm("\nTests completed.  Press any key to return\n" );
    mobo_key( 0 );
    swpipl( ipl );
    return final_rval;
}



/*----------------------------------------------------------------------*/
/* Main access routine */

static Cmd_t cmos_cmds[] =
{
    { NULL,	"get",		get_date_wrap,	SYS_NONE },
    { NULL,	"set",		CMOSSetDate,	SYS_NONE },
    { NULL,	"epoch",	CMOSEpoch,	SYS_NONE },
    { NULL,	"dump",		CMOSDump,	SYS_NONE },
    { NULL,	"count",	CMOSCount,	SYS_NONE },
    { NULL,	"timer", 	CMOSTimer,	SYS_NONE },
};



#define MAX_CMD_LEN	16		/* longest command plus some */

DBM_STATUS cmos( int argc, char *argv[] )
{
    const Cmd_t *C;

    /* look for command line arguments */
    if ( argc == 1 )			/* no sub-command specified */
    {
	InitRTC();
        if (cmosready() != STATUS_SUCCESS )
	{		/* wait until just after refresh */
	    mobo_logf( LOG_CRIT "DATE/TIME: CMOS clock is not ticking\n");
	    mobo_alertf("Date/Time failure",
			"The CMOS clock is not ticking" );
	}
	get_date_wrap( 0, NULL );
	return STATUS_SUCCESS;
    }

    /* Lookup a command sub-function */
    C = mobo_cmdlookup( argv[1], cmos_cmds, ARRAYLEN(cmos_cmds) );
    if ( C != NULL )	
    {
	return mobo_runcmd( C, NULL );				/* exec */
    }
    else return STATUS_FAILURE;
}

