/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/


#include "lib.h"
#include "uilib.h"			/* DEBUG */

#include "platform.h"
#include "northbridge.h"
#include "southbridge.h"

#include "ledcodes.h"
#include "beepcodes.h"
#include "rom.h"

#include "cserve.h"

#undef DEBUG
#ifdef DEBUG

void init_ioport(void)
{
    primary_impure->AUTOBAUD = cServe(0, 0, CSERVE_K_SROM_INIT);
}

void init_putstr(const char *msg)
{
    if( primary_impure->AUTOBAUD == 0 )      return;         /* not putting out msgs */

    while (*msg) {
        if ( *msg == '\n' ) {
            cServe(0, '\n', CSERVE_K_SROM_PUTC, primary_impure->AUTOBAUD );
            cServe(0, '\r', CSERVE_K_SROM_PUTC, primary_impure->AUTOBAUD );
        } else {
            cServe(0, (unsigned long)*msg, CSERVE_K_SROM_PUTC, primary_impure->AUTOBAUD );
        }
        msg++;
    }
}

#endif


/*----------------------------------------------------------------------*/
/* Private data structures and functions */

extern int storage_initialized;		/* stuff from malloc() */
extern ul storage_pool_start;
extern ul storage_pool_end;


enum fsb_errcode { FSB_ROMSCAN=2, FSB_FLOPPY=3, FSB_NOUPDATES=4,
		   FSB_FILECORRUPT=5, FSB_ROMFULL=6, FSB_FLASHFAIL=7,
		   FSB_SETUP=8, FSB_FINISHED=0 };

#define EXTNAME	".rom"			/* this is the extension we want */
#define EXTLEN	4			/* length of the above extension */

#define MAXNAME 32
typedef struct dirlist {
    char name[ MAXNAME ];
    struct dirlist *next;
} DirList;

DirList *files = NULL;

static size_t BytesRead;
static romheader_t *R;			/* start of loaded image */
static rom_t *H;			/* target reflash region */


ImpureRgn_t *primary_impure;

volatile unsigned char Secondary = 0;

static void fsb_failure( enum fsb_errcode );
static DBM_STATUS fsb_setup( void );
static DBM_STATUS fsb_dir( void );
static DBM_STATUS fsb_load( String SourceFile );
static DBM_STATUS fsb_imgverify( void );
static DBM_STATUS fsb_romalloc( void );
static DBM_STATUS fsb_reflash( void );



void fsb_main(
	 unsigned long init_halt_code,
         ImpureRgn_t *sysdata,
         unsigned is_secondary )

{
    DBM_STATUS sval;
    unsigned icount;			/* count of image updates made */
    DirList *D;
    unsigned flen;

    primary_impure = sysdata;			/* plug in the impure region */

#ifdef DEBUG
    init_ioport();
    init_putstr("\n\n\n\n\n\n\n\nHello!!!\n\n\n\n\n" );
#endif

#ifdef DEBUG
    printf_dbm( LOG_WARN	"Hello from FSB!!!\n" );	/* DEBUG */
#endif

    /* Signal that we're running */
    Beep( 200, 2048 );
    Beep( 200, 1024 );
    Beep( 200, 2048 );
    Beep( 200, 1024 );

    sval = fsb_setup();
    if ( sval != STATUS_SUCCESS )		fsb_failure( FSB_SETUP );


    /* Scan the ROM for images, build image tree */
    /* note, we're often called when the ROM is damaged, so failure here
     * should not stop the update process */
    (void) rom_index();

    /* Scan the floppy for image files */
    sval = fsb_dir();
    if ( sval != STATUS_SUCCESS )		fsb_failure( FSB_FLOPPY );

    for( D=files, icount=0; D != NULL; D=D->next )
    {
	/* for each .rom filename */
#ifdef DEBUG
	printf_dbm( LOG_WARN "analysing file %s...\n", D->name );/* DEBUG */
#endif
	flen = strlen( D->name );
	if ( flen < EXTLEN )			continue;
	if ( strncmp( D->name + (flen - EXTLEN), EXTNAME, EXTLEN ) != 0 )
						continue;

#ifdef DEBUG
	printf_dbm( LOG_WARN "processing file %s...\n", D->name );/* DEBUG */
#endif
	/* load the file from floppy */
	sval = fsb_load( D->name );
	if ( sval != STATUS_SUCCESS )		fsb_failure( FSB_FLOPPY );

	/* verify the file is a valid .rom image, if not then freak out */
#ifdef DEBUG
	printf_dbm( LOG_WARN "verifying file %s...\n", D->name );/* DEBUG */
#endif
	sval = fsb_imgverify( );
	if ( sval == STATUS_WARNING )	continue;	/* don't flash!*/
	if ( sval == STATUS_FAILURE )	fsb_failure( FSB_FILECORRUPT );

	/* is there space in the ROM? */
#ifdef DEBUG 
	printf_dbm( LOG_WARN "allocating file %s...\n", D->name );/* DEBUG */
#endif
	sval = fsb_romalloc( );
	if ( sval != STATUS_SUCCESS )		fsb_failure( FSB_ROMFULL );

	/* perform the update - might fail if write-protected */
#ifdef DEBUG
	printf_dbm( LOG_WARN "reflashing file %s...\n", D->name );/* DEBUG */
#endif
 	sval = fsb_reflash( );
	if ( sval != STATUS_SUCCESS )		fsb_failure( FSB_FLASHFAIL );
        Beep( 200, 2048 );
	icount++;
    }

    if( icount == 0 )				fsb_failure( FSB_NOUPDATES );


    /* otherwise, we're finished, what do we do now? */
    /* return icount; */
    fsb_failure( FSB_FINISHED );
}


static void fsb_failure( enum fsb_errcode E )
{
    unsigned i;

#ifdef DEBUG
    printf_dbm( LOG_WARN __FUNCTION__ " code %d\n", E );
#endif
    while( 1 ) {
 	if ( E == FSB_FINISHED )
	{
	    Beep( 200, 512 );
	    Beep( 200, 1024 );
	    Beep( 200, 2048 );
	    Beep( 200, 4096 );
	    Beep( 200, 8192 );

	} else {
	    
	    for( i=0; i<E; i++ ) {
		Beep( 200, 2048 );
		msleep( 200 );
	    }

	}

	msleep( 2000 );
    }
}



/* preparation of the system for the fail-safe booter */

static DBM_STATUS fsb_setup( void )
{
#ifdef DEBUG
    printf_dbm( LOG_WARN __FUNCTION__ "\n" );		/* DEBUG */
#endif

    /* Initialize calls to malloc. */
    storage_initialized = FALSE;
    init_storage_pool();

    /* configure the principal system components */
    nb_setup();
    sb_setup();

    nb_fixup();
    sb_fixup();

    return STATUS_SUCCESS;
}


static DBM_STATUS fsb_dir( void )
{
    FILE * FilePointer;
    DBM_STATUS Status = STATUS_SUCCESS;
    String name;
    DirList *D=NULL;
    char *src, *dst;
    static const String dir_fmt = "%f";		/* %f = filename only */

#ifdef DEBUG
    printf_dbm( LOG_WARN __FUNCTION__ "\n" );		/* DEBUG */
#endif

    /*  First try to open the file for reading.  If not found
     *  or some type of error is encountered about the operation.
     */
    if ((FilePointer = fopen( "\\", "d")) == NULL)	return STATUS_FAILURE;

    name = fdir(FilePointer, dir_fmt );
    while (name != NULL)
    {
	/* we have another file entry to process, put it into the list */
	D = malloc( sizeof( DirList ) );
	if ( D == NULL ) {
	    fclose( FilePointer );
	    return STATUS_FAILURE;
	}

#ifdef DEBUG
	printf_dbm( LOG_WARN "found file name %s\n", name );	/* DEBUG */
#endif
	for( src=name, dst=D->name; (*src != 0) && (*src !=' '); src++, dst++ )
	    *dst = *src;

#if 0
	sscanf( name, "%s", D->name );		/* copy name, less spacing */
#endif
	D->next = files;			/* link onto head of list  */
	files = D;

	/* onto next entry in the directory */
        name = fdir(NULL, dir_fmt );
    }

    /* Were there are any errors? */
    if (errno != NO_ERRORS)			Status = STATUS_FAILURE;
    if (fclose(FilePointer) == EOF)		Status = STATUS_FAILURE;

    return Status;
}



static DBM_STATUS fsb_load( String SourceFile )
{
    FILE * FilePtr;

#ifdef DEBUG
    printf_dbm( LOG_WARN __FUNCTION__ "\n" );		/* DEBUG */
#endif
    BytesRead=0;		/* start with a clean slate */

    /*  First try to open the file for reading.  If not found
     *  or some type of error is encountered about the operation.
     */
    if ((FilePtr = fopen(SourceFile, "r")) == NULL)	return STATUS_FAILURE;


    /*  Read the file and check for errors.
     */
    if ((BytesRead = fload(FilePtr, (char *)DBM_DLD)) == 0)
							return STATUS_FAILURE;
    if (fclose(FilePtr) == EOF)				return STATUS_FAILURE;

    return STATUS_SUCCESS;
}


/* Validate the image checksums of a ROM image candidate loaded into memory */

static DBM_STATUS fsb_imgverify( void )
{
    unsigned csum;
    unsigned *romval;
    unsigned i;
    FWID I;

#ifdef DEBUG
    printf_dbm( LOG_WARN __FUNCTION__ "\n" );		/* DEBUG */
#endif
    /* validate the header */
    R = (romheader_t *)DBM_DLD;			/* start of loaded image */
    if ( !rom_hdrvalid( R ) )			return STATUS_FAILURE;

    I = R->romh.V1.fw_id;

    /* filter out any images we don't dare erase */
    switch ( I ) {
	case FW_FSB:
	case FW_CBOX:
	case FW_SR:
		return STATUS_WARNING;

	default:				/* we can do business */
    }

    /* validate the rest of the image, */
    /* nb image must be an integral number of 32-bit words in size */
    romval = (unsigned *)(DBM_DLD + R->romh.V0.hsize);

#ifdef DEBUG
    printf_dbm( LOG_WARN "validating ROM at 0x%X (%d/%d bytes)\n",
		romval, R->romh.V0.size, BytesRead );	/* DEBUG */
#endif

    for( i=0, csum=0; i<R->romh.V0.size; i+=sizeof(unsigned) )
    {
        csum = compute_checksuml( *romval++, csum );
    }
#ifdef DEBUG
    printf_dbm( LOG_WARN "Checksum was 0x%X expected 0x%x\n", 
			csum, R->romh.V0.checksum );	/* DEBUG */
#endif

    if ( csum != R->romh.V0.checksum )	return STATUS_FAILURE;

    return STATUS_SUCCESS;
}



/* find space in the ROM for this proposed image */

static DBM_STATUS fsb_romalloc( void )
{
#ifdef DEBUG
    printf_dbm( LOG_WARN __FUNCTION__ "\n" );		/* DEBUG */
#endif
    /* First thing to check is an existing image of this type, with space
     * to be upgraded.  For the trickier case of an existing image that is 
     * not big enough to handle being overwritten, we currently fail... */

    H = rom_search( R->romh.V1.fw_id, rom_map );
    if ( H != NULL ) {			/* match: but is it big enough? */
	if( H->asize < BytesRead )	return STATUS_FAILURE;

	return STATUS_SUCCESS;
    }


    /* If there is no current image of this type, find the first empty region
     * big enough to contain this new image */

    for( H=rom_map; H!=NULL; H=H->next )
    {
	/* filter out regions that don't match */
	if( (H->flags & ROM_EMPTYRGN) == 0 )	continue;
	if( H->asize < BytesRead )		continue;

	return STATUS_SUCCESS;		/* found an empty region big enough */
    }

    /* There is neither a current image with room to upgrade, nor an empty 
     * space big enough */

    return STATUS_FAILURE;
}


/* we have a region to erase defined by the rom_t handle */

static DBM_STATUS fsb_reflash( void )
{
    DBM_STATUS sval;

#ifdef DEBUG
    printf_dbm( LOG_WARN __FUNCTION__ "\n" );		/* DEBUG */
#endif
    /* first erase the region */
    sval = plat_romerase( H->astart, H->astart + BytesRead - 1, NULL, 0 );
    if ( sval != STATUS_SUCCESS ) 		return STATUS_FAILURE;

    /* then write the region */
    sval = plat_romwrite( H->astart, (const unsigned char *)DBM_DLD, BytesRead,
                        NULL, 0);
    if ( sval != STATUS_SUCCESS ) 		return STATUS_FAILURE;

    rom_free( rom_phys_map );
    rom_phys_map = NULL;
    rom_index();			/* check it scanned in OK */
    return STATUS_SUCCESS;
}

