/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* System-related configuration details for the API Nautilus platform */
/* Begun by Stig Telfer, Alpha Processor Inc, 28 Mar 1999 */


#include "lib.h"
#include "uilib.h"
#include "info.h"
#include "pci.h"

#include "osf.h"
#include "mcheck.h"
#include "cpu/ev6.h"
#include "hwrpb.h"			/* for systype settings */

#include "specifics.h"			/* motherboard specific definitions */
#include "platform.h"			/* interface for this module */
#include "northbridge.h"		/* Generic NB access */
#include "southbridge.h"		/* Generic SB access */
#include "tabledriver.h"
#include "beepcodes.h"
#include "nvram.h"

#include "i2c.h"
#include "i2c/lm75.h"
#include "i2c/adm9240.h"

#include "southbridge/pic.h"

const char *Prompt = "Nautilus> ";


const unsigned plat_sys_type = ST_API_NAUTILUS;
const unsigned plat_sys_variation = 1 << 10;

static const PCISlot_t pcislots[] = { 
	{ PCI_AMD_IG,	"AMD onboard" },
	{ PCI_AMD_IG2,	"AMD onboard" },
        { PCI_ALI_ISA,	"ALI onboard" },
        { PCI_ALI_IDE,	"ALI onboard" },
        { PCI_ALI_PMU,	"ALI onboard" },
        { PCI_ALI_USB,	"ALI onboard" },
        { PCI_SLOT1,	"Slot 1" },
        { PCI_SLOT2,	"Slot 2" },
        { PCI_SLOT3,	"Slot 3" },
        { PCI_SLOT4,	"Slot 4" },
        { 0, 0 } };

static const PCISlot_t agpslots[] = { 
        { AGP_SLOT1, "AGP Slot" }, 
        { 0, 0 } };


/* note - ordering here is critical, it must match the bus numbering as 
 * decided by the PCI device tree building algorithm */

const PCIStructure_t PCIStructure[] = {
    { VARIANT_PCI, pcislots },			/* PCI hose 0 layout */
    { VARIANT_AGP, agpslots },			/* AGP hose 1 layout */
    };

const short plat_pci_hose_count = ARRAYLEN( PCIStructure );


/* Locations of popular PCI southbridge devices */
const unsigned short plat_isaid = PCI_ALI_ISA;	/* Southbridge ISA */
const unsigned short plat_usbid = PCI_ALI_USB;	/* Southbridge USB */
const unsigned short plat_pmuid = PCI_ALI_PMU;	/* Southbridge PMU */
const unsigned short plat_ideid = PCI_ALI_IDE;	/* Southbridge IDE */
             

static uint8 i2c_bus_setup = FALSE;


DBM_STATUS plat_setup( void ) 
{
    /* Register early info */
    info_submit( SRC_DIAGS, SUBJ_MOTHERBOARD, "Motherboard", "API UP1000" );
    return STATUS_SUCCESS;
}

DBM_STATUS plat_fixup( void ) 
{
    plat_nmiclr();			/* startup seems to incurr these */
    return STATUS_SUCCESS;
}



DBM_STATUS plat_post( void )
{
    DBM_STATUS sval;

    nb_setup();				/* need working PCI for this */
    sval = i2c_post();
    if ( sval != STATUS_SUCCESS )
    {
	/* Bring up a console */
	BeepCode( beep_k_system_health );
	bring_up_console( );
	
	/* Get the details from the I2C module */
	i2c_post_report();
    }
    return sval;
}


DBM_STATUS plat_reset( int argc, char *argv[] )
{
    outportb( 0x92, 0x01 );		/* toggle port 92 to reset */

    sleep( 1 );
    mobo_logf( LOG_WARN "RESET: nothing happened!\n" );
    return STATUS_FAILURE;			/* reset not successful */
}



/*----------------------------------------------------------------------*/
/* System status string, for printing out during memory stress test */
/* we have four lines of room here */

void plat_sysstat( String S )
{
    ECC_t ECC;                          /* Irongate CSRs */
    Status_t Status;
    Command_t Command;
    reg_handle e, s, c;
    char eccbuf[80];

    unsigned sys61;
    unsigned sbnmi;
    unsigned ipl = swpipl( 7 );
    swpipl( ipl );			/* find our IPL */

    e.p16 = &ECC.i;			/* read the ECC register */
    ig_rdcsr( IG_ECC, e );
    s.p16 = &Status.i;			/* read the IG Status register */
    ig_rdcsr( IG_STATUS, s );
    c.p16 = &Command.i;			/* read the IG Command register */
    ig_rdcsr( IG_COMMAND, c );

    ig_pcsr( IG_ECC, e, eccbuf );
    
    /* bits 6,7 report NMI has occurred, bits 3,2 are pulsed high to reset */
    sys61 = inportb( 0x61 );                        /* NMI ctrl/status port */

    /* bit 0 in this register is 1 if SERR->NMI is enabled */
    sbnmi = pcicfgrb( 0, PCI_ALI_ISA, 0, ISA_PIPM );

    sprintf_dbm( S, "Irongate: system NMI is %s and %s\r"
		    "Irongate: ECC is %s\r"
		    "Southbridge: system NMI reporting is %s, NMI is %s\r"
		    "System interrupts are at level %d (0=all enabled, 7=all disabled)",
		    Command.i & 0x0100 ? "enabled" : "disabled",
		    Status.i & 0x4000 ? "asserted" : "clear",
		    eccbuf,
		    (sbnmi & 1) ? "enabled" : "disabled",
		    (sys61 & 0xC0) ? "asserted" : "clear",
		    ipl	);
}



/*----------------------------------------------------------------------*/
/* setup of all system interrupts - o = 0 (off), 1 (on) */

static const TblArray configure_pci_int_routings[] = {
 
#if 1
    CFGWB( 0, PCI_ALI_ISA, 0, ISA_PIRT1, 0x15 ),  /* INT2=IRQ9, INT1=IRQ5 */
    CFGWB( 0, PCI_ALI_ISA, 0, ISA_PIRT2, 0x93 ),  /* INT4=IRQ11, INT3=IRQ10 */
#else
    CFGWB( 0, PCI_ALI_ISA, 0, ISA_PIRT1, 0x99 ),  /* INT2=IRQ9, INT1=IRQ5 */
    CFGWB( 0, PCI_ALI_ISA, 0, ISA_PIRT2, 0x99 ),  /* INT4=IRQ11, INT3=IRQ10 */
#endif
    CFGWB( 0, PCI_ALI_ISA, 0, ISA_PIRT3, 0x00 ),  /* unused in nonpoll mode */
    CFGWB( 0, PCI_ALI_ISA, 0, ISA_PIRT4, 0x00 ),

#if 0
    CFGWB( 0, PCI_ALI_ISA, 0, ISA_PILET, 0x0F ),  /* make edge-triggered?? */
#endif

    CFGWB( 0, PCI_SLOT1,   0, PCIHDR_IRQ,0x05 ),  /* Store this on the device */
    CFGWB( 0, PCI_SLOT2,   0, PCIHDR_IRQ,0x09 ),  /* Store this on the device */
    CFGWB( 0, PCI_SLOT3,   0, PCIHDR_IRQ,0x0A ),  /* Store this on the device */
    CFGWB( 0, PCI_SLOT4,   0, PCIHDR_IRQ,0x0B ),  /* Store this on the device */

    OP_EOT
};

/* NB: need to move to a symbolic representation without magic numbers */
void plat_intsetup( int o ) 
{
    int rval;

    plat_intclr();			/* wipe the slate clean beforehand */

    if ( o )				/* enabling case */
    {
	/* Note: IRQ 8 is disabled because the RTC is directly connected */
	/* DEBUG - here I disable IRQ 15 because it is annoying me... */
	mobo_logf( LOG_INFO "Enabling ISA interrupts...\n");
	outportb( 0x21, 0x00 );                     /* IRQs 0-7, active low */
	outportb( 0xA1, 0x81 );                     /* IRQs 8-15, active low */

	mobo_logf( LOG_INFO "Enabling PCI interrutps...\n");
	/* disable polling mode */
	rval = pcicfgrb( 0, PCI_ALI_ISA, 0, ISA_PIPM );	
	rval &= 0x7FU;				    /* clr bit 7 (poll mode) */
	pcicfgwb( 0, PCI_ALI_ISA, 0, ISA_PIPM, rval );

	TblWrite( configure_pci_int_routings );

	pic_irq_trig( IRQ11, IRQ_LEVEL );
	pic_irq_trig( IRQ9, IRQ_LEVEL );
	pic_irq_trig( IRQ5, IRQ_LEVEL );
	pic_irq_trig( IRQ10, IRQ_LEVEL );

	diags_subsys_stat( SYS_IRQ, DEV_SETUP );
    }
    else
    {
	/* disabling case */
	mobo_logf( LOG_INFO "Disabling ISA interrupts...\n");
	outportb( 0x21, 0xFF );                     /* IRQs 0-7, active low */
	outportb( 0xA1, 0xFF );                     /* IRQs 8-15, active low */
    }
}



/* Clear any pending interrupts (except machine checks) on this machine */


void plat_intclr( void )
{
    unsigned char iis0, iis1;
    int iters;
    int irq;
    int ipl;

    ipl = swpipl( 7 );			/* noodling this stuff is dangerous */

#define MAX_ACKS	16

    /* until there are no more interrupts in service, send EOI signals */
    for ( iters=0; iters < MAX_ACKS; iters++)          /* 8 IRQ lines per reg */
    {
	irq = inIack();				/* ack the int, returns IRQ */

        /* read Int in-service regs */
        outportb( 0x20, 0x0B );
        iis0 = inportb( 0x20 );
        outportb( 0xA0, 0x0B );
        iis1 = inportb( 0xA0 );

        if ( iis0 == 0 && iis0 == 0 )   break;

        if ( iis0 )      outportb( 0x20, 0x20 );                /* EOI */
        if ( iis1 )      outportb( 0xA0, 0x20 );                /* EOI */
    }

    swpipl( ipl );			/* restore previous IPL */

    if ( iters >= MAX_ACKS )
    	mobo_logf( LOG_CRIT "ERROR - recurring interrupts - not clearing\n");
}



/* Clear any pending system machine checks (NMIs) */

void plat_nmiclr( void )
{
    ECC_t ECC;
    Status_t Status;
    reg_handle r;
    unsigned val;


    /* first have a look at the Irongate and try to get it to de-assert SERR */
    Status.i = 0;
    Status.r.serr = 1;
    r.p16 = &Status.i;
    ig_wrcsr( IG_STATUS, r );		/* write a 1 to de-assert */

    ECC.i = 0xFFFF;			/* DEBUG - try writing all ones?? */
    r.p16 = &ECC.i;
    ig_wrcsr( IG_ECC, r );		/* write zeroes to clear */

    /* now try to de-assert in the ALI southbridge */
    /* technique - read port 61. bit 7=1 on NMI.  bit 2 ->1->0 to reset NMI */
    val = inportb( 0x61 );                        /* NMI ctrl/status port */
    val |= 0x0C;
    outportb( 0x61, val );                        /* write 1's to clear NMIs */
    val &= ~0x0C;
    outportb( 0x61, val );                        /* write 0's to re-enable */


    /* finally, verify that the SERR has been deasserted */
    r.p16 = &Status.i;
    ig_rdcsr( IG_STATUS, r );
    if ( Status.r.serr == 1 )	
	mobo_logf( LOG_WARN "NMI: SERR is still asserted by Irongate!\n");
}


/*----------------------------------------------------------------------*/
/* I2C and System Management Bus */

/* Private definitions */
/* Select a particular sub-bus by setting the correct multiplexer */

#define MUX_BUS_NONE	0x0	/* no mux dependency */
#define MUX_BUS_SLOTB	0x2	/* Disconnect sub-bus 1 */
#define MUX_BUS_DIMMS	0x1	/* Disconnect sub-bus 0 */
#define MUX_UNKNOWN	0xFF	/* unknown state before first init */


/* Published data */
/* A data structure enumerating and describing the I2C devices */

const I2Cbus_t plat_i2c_bus[] = {

    /* Main motherboard I2C sensors */
    { I2C_SMD0,		I2CSENS_TEMP0,	I2CCHIP_ADM9240,
	NAUT_I2C_SMD, MUX_BUS_NONE, "Motherboard temperature" },
    { I2C_SMD0,		I2CSENS_FAN0,	I2CCHIP_ADM9240,
	NAUT_I2C_SMD, MUX_BUS_NONE, "Slot-B Fan 0 (top)" },
    { I2C_SMD0,		I2CSENS_FAN1,	I2CCHIP_ADM9240,
	NAUT_I2C_SMD, MUX_BUS_NONE, "Slot-B Fan 1 (bottom)" },

    { I2C_SMD0, I2CSENS_3_3V,   I2CCHIP_ADM9240,
	NAUT_I2C_SMD, MUX_BUS_NONE, "System 3.3V supply level" },
    { I2C_SMD0, I2CSENS_12V,    I2CCHIP_ADM9240,
	NAUT_I2C_SMD, MUX_BUS_NONE, "System 12V supply level" },
    { I2C_SMD3, I2CSENS_5V,     I2CCHIP_ADM9240,
	NAUT_I2C_SMD, MUX_BUS_NONE, "System 5V supply level" },
#if 0
    /* Processor Core voltage may vary over processor revisions... ? */
    { I2C_SMD1, I2CSENS_VCCP1,  I2CCHIP_ADM9240,	/* 2.5V nominal? */
	NAUT_I2C_SMD, MUX_BUS_NONE, "Vccp1" },
#endif


    /* Slot-B I2C devices */
    { I2C_SLOTB0_THERM,	I2CSENS_TEMP0,	I2CCHIP_LM75,
	NAUT_I2C_LM75, MUX_BUS_SLOTB, "Slot-B temperature" },

    /* Other misc devices */
    { I2C_MUX0,		I2CSENS_REGMUX,	I2CCHIP_PCF8574,
	NAUT_I2C_MUX, MUX_BUS_NONE, "Sub-bus multiplexer" },
    { I2C_LED,		I2CSENS_REGMUX,	I2CCHIP_PCF8574,
	NAUT_I2C_LED, MUX_BUS_NONE, "Motherboard LEDs" },

    { I2C_NODEV, 0, 0, 0, 0, NULL }

#if 0			/* unused or unsupported devices */
    { I2C_DEV_ICS9179,   I2C_CLK_BUFFER,   NAUT_I2C_SDRAMCLK, MUX_BUS_NONE,
        "Zero delay clock buffer" },

    /* DIMMs */
    { I2C_DEV_JEDEC,	 I2C_DIMM0,	   NAUT_I2C_DIMM0, MUX_BUS_DIMMS,
	"DIMM in slot 0" },
    { I2C_DEV_JEDEC,	 I2C_DIMM1,	   NAUT_I2C_DIMM1, MUX_BUS_DIMMS,
	"DIMM in slot 1" },
    { I2C_DEV_JEDEC,	 I2C_DIMM2,	   NAUT_I2C_DIMM2, MUX_BUS_DIMMS,
	"DIMM in slot 2" },

    { I2C_DEV_PCF8582,   I2C_SLOTB0_REV,   NAUT_I2C_SLOTB_EPROM, MUX_BUS_SLOTB,
	"Slot-B revision" },
#endif
};


/* I2C interface routines */

/* We may need to do platform specific things whilst reading an I2C device,
 * such as configuring multiplexers correctly, etc.
 */

DBM_STATUS plat_i2c_select( const I2Cbus_t *I )
{
    static uint8 sub_bus_mux = MUX_UNKNOWN;

    /* Any mux action required to access this device? */
    if ( I->config == MUX_BUS_NONE )
	return STATUS_SUCCESS;

    /* Already in the right configuration? */
    if ( sub_bus_mux == I->config )
	return STATUS_SUCCESS;

    /* Set up the new value and put it out on the bus */
    sub_bus_mux = I->config;
    I = i2c_lookup( I2C_MUX0 );

    /* returning STATUS_WARNING would indicate the device was absent in 
     * a non-faulty way (eg, its on an absent CPU module) */
    return (i2cdev_write( I->addr, -1, 1, &sub_bus_mux ) == STATUS_FAILURE) ?
	STATUS_FAILURE : STATUS_SUCCESS;
}


DBM_STATUS plat_i2cinit(void )
{
    DBM_STATUS sval;
    const I2Cbus_t *I;

    diags_subsys_stat( SYS_I2C, DEV_PROBING );
    sval = i2cdev_init( );
    if ( sval != STATUS_SUCCESS )
    {
	diags_subsys_stat( SYS_I2C, DEV_FAILED );
	return sval;
    }


    /* Configure the slot-B LM75 for UP1000 parameters. */

    I = i2c_lookup( I2C_SLOTB0_THERM );
    plat_i2c_select( I );
    sval = lm75_init( I->addr );
    if ( sval == STATUS_FAILURE )
    {
	mobo_logf( LOG_CRIT "Write failure to LM75 sensor during I2C init\n" );
	mobo_alertf( "I2C Write Failure",
		"Failure occurred whilst writing to LM75 thermal sensor" );
    }

    I = i2c_lookup( I2C_SMD0 );
    plat_i2c_select( I );
    sval = adm_init( I->addr );
    if ( sval == STATUS_FAILURE )
    {
	mobo_logf( LOG_CRIT "Write error to ADM9240 sensor during I2C init\n" );
	mobo_alertf( "I2C Write Failure",
		"Failure occurred whilst writing to ADM9240 system monitor" );

    } else {

	/* Enable all our configured I2C ADM9240 sensors */
	adm_add_dev( NAUT_I2C_SMD, I2CSENS_TEMP0,
		    TEMP_MAX_MOBO * I2C_TEMP_SCALE,
		    TEMP_HYST_MOBO * I2C_TEMP_SCALE );
	adm_add_dev( NAUT_I2C_SMD, I2CSENS_FAN0, 7800, 7000 );
	adm_add_dev( NAUT_I2C_SMD, I2CSENS_FAN1, 7800, 7000 );

	/* Now the sensors are configured, set the thing sampling */
	adm_start( NAUT_I2C_SMD );
    }


    i2c_bus_setup = TRUE;
    diags_subsys_stat( SYS_I2C, DEV_SETUP );
    return STATUS_SUCCESS;
}


/*----------------------------------------------------------------------*/
/* Non-maskable interrupt (NMI) analysis */

void plat_mchkinterp( String interp, int scb, LogoutFrame_t *L)
{
    ECC_t     ECC;
    Status_t  Status;
    reg_handle  r;
    char IG_ecc[80];
    char IG_stat[80];

    switch( scb ) {
    case SCB_Q_SYSERR:		/* not currently used by Nautilus PALcode */
    case SCB_Q_SYSMCHK:		/* NMI */

	r.p16 = &ECC.i;
	ig_rdcsr( IG_ECC, r );
	ig_pcsr( IG_ECC, r, IG_ecc );
	r.p16 = &Status.i;
	ig_rdcsr( IG_STATUS, r );
	ig_pcsr( IG_STATUS, r, IG_stat );
	sprintf_dbm( interp, "IG Status: %s\rECC: %s", IG_stat, IG_ecc );
    }
}

/*----------------------------------------------------------------------*/
/* SMP support - Nautilus is uniprocessor so this does nothing */

void plat_smpstart( unsigned long palbase ) { }


/*----------------------------------------------------------------------*/
/* Debugging LED writes */

#define LEDPORT 0x80

void outled( unsigned d)
{
    uint8 ledval;

    outportb(LEDPORT, d);

    /* Put the data out onto the motherboard LEDs ... */
    if ( i2c_bus_setup )
    {
	ledval = ~d & 0xFFU;
	i2cdev_write( NAUT_I2C_LED, -1, 1, &ledval );
    }
}


