// Whirlwind simulator, based on Guy Fedorkow's simulator

/**
 * Constants
 * @constructor
 */
ConstWWbitClass = function () {
  this.NBANKS = 6;

  // Caution -- Whirlwind puts bit 0 to the left (Big Endian, no?)
  this.WWBIT0 = 0100000;
  this.WWBIT1 = 0040000;
  this.WWBIT5 = 002000;   // Bit 5 is the most-significant bit of the address field
  this.WWBIT6 = 001000;   // this is Bit 6 starting from Zero on the left in a 16-bit word
  this.WWBIT7 = 000400;
  this.WWBIT8 = 000200;
  this.WWBIT9 = 000100;
  this.WWBIT10_12 = 000070;  // used in CF instruction
  this.WWBIT13_15 = 000007;
  this.WWBIT6_15 = 001777;  // half the address space, used in mem bank decode
  this.WW_ADDR_MASK = 003777;

  this.WWBIT1_15 = 0077777;
  this.WWBIT0_15 = 0177777;

  this.pyBIT30 = (1 << 30);  // little-endian bits for concatenated AC and BR

  this.pyBIT30 = (1 << 30);  // little-endian bits for concatenated AC and BR
  this.pyBIT15 = (1 << 15);  // little-endian bits for concatenated AC and BR
  this.pyBIT29_0 = 0x3FFFFFFF;  // I can't count that many octal digits
  this.pyBIT31_0 = 0xFFFFFFFF;  // I can't count that many octal digits

  this.CORE_SIZE = 2048;

  this.SHIFT_RIGHT = 1;
  this.SHIFT_LEFT = 2;

  // Instruction execution would screech to a halt if various alarms went off
  // The first few are defined by the instruction set

  this.NO_ALARM = 0;
  this.OVERFLOW_ALARM = 1;
  // Then there are some synthetic alarms I've added as part of the sim
  this.UNIMPLEMENTED_ALARM = 2;     // hit an unimplemented instruction
  this.READ_BEFORE_WRITE_ALARM = 3;   // tried to operate on a "none" operand, i.e., uninitualized memory
  this.UNKNOWN_IO_DEVICE_ALARM = 4;   // hit an unimplemented I/O Device
  this.HALT_ALARM = 5;        // hit a "halt" instruction, aka "si 0" or "si 1"
  this.CHECK_ALARM = 6;         // alarm if the machine fails a Check instruction
  this.QUIT_ALARM = 7;        // synthetic alarm to stop the sim
};


/**
 * Handle switches
 */
WWSwitchClass = function() {
  this.switchNameDict = {
    // name: [default_val, mask]
    "CheckAlarmSpecial": [0, 001],  // Controls the behavior of the CK instruction; see 2M-0277
                    // "normal" is 'off'
  }

  this.parse_switch_directive = function(args) { // return one for error, zero for ok.
    if (args.length != 2) {
      alert("Switch Setting: expected <name> <val>, got: " + args);
      return 1;
    };

    var name = args[0];
    if (!(name in this.switchNameDict)) {
      alert("No machine switch named " + name);
      return 1;
    };

    var valstr = args[1];
    var val = 0;
    try {
      if (valstr.startsWith("0o")) {
        valstr = valstr.substr(2);
      }
      val = parseInt(valstr, 8)
    } catch(err) {
      alert(".SWITCH " + name + " setting " + valstr + " must be an octal number: " + err);
      return 1;
    }
    var mask = this.switchNameDict[name][1];
    if ((~mask & val) != 0) {
      alert("max value for switch " + name + " is " + toOctal(mask) + "o, got " + toOctal(val) + "o");
      return 1;
    }
    this.switchNameDict[name][0] = val;
    console.log(".SWITCH " + name + " set to " + toOctal(val) + "o");
    return 0
  };

  this.read_switch = function(name) {
    return this.switchNameDict[name][0];
  };

};

/**
 * See manual 2M-0277 pg 46 for flexowriter codes and addresses
 * @constructor
 */
FlexoClass = function(sim) {
    this.sim = sim;
    this._uppercase = false;  // Flexo used a code to switch to upper case, another code to return to lower
    this._color = false;  // the Flexo had a two-color ribbon, I assume it defaulted to Black
    this.stop_on_zero = null;
    this.packed = null;
    this.null_count = 0;
    this.FlexoOutput = [];
    this.name = "Flexowriter";

    this.FLEXO_BASE_ADDRESS = 0224;  // Flexowriter printers
    this.FLEXO_ADDR_MASK = ~(0013);  // mask out these bits to identify any Flexo address

    this.FLEXO3 = 0010;  // code to select which flexo printer.  #3 is said to be 'unused'
    this.FLEXO_STOP_ON_ZERO = 001;  // code to select whether the printer "hangs" if asked to print a zero word
    this.FLEXO_PACKED = 002;    // code to interpret three (six-bit) characters per word (??)

    this.FLEXO_UPPER = 0071;   // character to switch to upper case
    this.FLEXO_LOWER = 0075;   // character to switch to lower case
    this.FLEXO_COLOR = 0020;   // character to switch ribbon color
    this.FLEXO_NULLIFY = 0077; // the character that remains on a tape after the typiest presses Delete


    //  From "Making Electrons Count:
    // "Even the best of typists make mistakes. The error is nullified by pressing the "delete" button. This
    // punches all the holes resulting in a special character ignored by the computer"

    this.flexocode_lcase = ["//", "#", "e", "8", "#", "|", "a", "3",
                " ", "=", "s", "4", "i", "+", "u", "2",
                "<color>", ".", "d", "5", "r", "l", "j", "7",
                "n", ",", "f", "6", "c", "-", "k", "//",

                "t", "//", "z", "<bs>", "l", "\t", "w", "#",
                "h", "\n", "y", "//", "p", "#", "q", "#",
                "o", "<stop>", "b", "//", "g", "#", "9", "#",
                "m", "<upper>", "x", "//", "v", "<lower>", "0", "<null>"];

    this.flexocode_ucase = ["//", "#", "E", "8", "#", "_", "A", "3",
                " ", ":", "S", "4", "I", "/", "U", "2",
                "<color>", ")", "D", "5", "R", "l", "J", "7",
                "N", "(", "F", "6", "C", "-", "K", "//",

                "T", "//", "Z", "<bs>", "L", "\t", "W", "#",
                "H", "\n", "Y", "//", "P", "#", "Q", "#",
                "O", "<stop>", "B", "//", "G", "#", "9", "#",
                "M", "<upper>", "X", "//", "V", "<lower>", "0", "<null>"];

  this.code_to_letter = function(code) {
    if (code == this.FLEXO_NULLIFY) {
      this.null_count += 1
      return ''
    }
    if (code == this.FLEXO_UPPER) {
      this._uppercase = true;
    } else if (code == this.FLEXO_LOWER) {
      this._uppercase = false;
    } else if (code == this.FLEXO_COLOR) {
      this._color = !this._color;
      if (this._color) {
        return  "\033[1;31m";
      } else {
        return "\033[0m";
      }
    } else {
      if (this._uppercase) {
        return this.flexocode_ucase[code];
      }
      return this.flexocode_lcase[code];
    }
    return '';
  };

  this.is_this_for_me = function(io_address) {
    if ((io_address & this.FLEXO_ADDR_MASK) == this.FLEXO_BASE_ADDRESS) {
      return this;
    } else {
      return null;
    }
  };

  this.si = function(device, accumulator) {
    // 0224  # select printer #2 test control by console.
    // 0234  # select printer #3
    if (device & this.FLEXO3) {
      alert("Printer //3 not implemented");
      return cb.UNIMPLEMENTED_ALARM;
    }

    this.stop_on_zero = (device & this.FLEXO_STOP_ON_ZERO);
    this.packed = (device & this.FLEXO_PACKED);

    if (this.packed) {
      alert("Flexowriter packed mode not implemented");
      return this.sim.cb.UNIMPLEMENTED_ALARM;
    }

    alert("configure flexowriter //2, stop_on_zero=" + this.stop_on_zero + 
        ", packed=" + this.packed);
    return this.sim.cb.NO_ALARM;
  };

  this.rc = function(acc) {  // "record", i.e. output instruction to tty
    code = acc >> 10;  // the code is in the upper six bits of the accumulator
    symbol = this.code_to_letter(code);  // look up the code, mess with Upper Case and Color
    this.FlexoOutput.append(symbol);
    if (symbol == '\n') {
      symbol = '\\n';
    }
    return [this.sim.cb.NO_ALARM, symbol];
  };

  this.get_saved_output = function() {
    return this.FlexoOutput
  };

};

/**
 * @constructor
 */
XwinCrt = function(sim) {
  // coordinate definitions for Whirlwind CRT display
  this.sim = sim;

  this.WW_MAX_COORD = 1024.0;
  this.WW_MIN_COORD = -this.WW_MAX_COORD;
  this.WW_CHAR_HSTROKE = 20;
  this.WW_CHAR_VSTROKE = 15;

  this.WIN_MAX_COORD = 1024.0;  // size of window to request from the laptop window  manager
  this.WIN_MOUSE_BOX = this.WIN_MAX_COORD / 50.0

  // coordinate definitions for Whirlwind CRT display
  this.WW_MAX_COORD = 1024.0;
  this.WW_MIN_COORD = -this.WW_MAX_COORD;
  this.WW_CHAR_HSTROKE = 20 * (this.WIN_MAX_COORD / (this.WW_MAX_COORD * 2));
  this.WW_CHAR_VSTROKE = 15 * (this.WIN_MAX_COORD / (this.WW_MAX_COORD * 2));

  this.canvas = $("#crt")[0];
  this.canvas.style.background = "black";
  this.ctx = this.canvas.getContext('2d');
  this.ctx.lineWidth = 4;
  this.canvas.width = this.WIN_MAX_COORD;
  this.canvas.height = this.WIN_MAX_COORD * .75;

  this.mouseX = null;
  this.mouseY = null;

  var crt = this;

  $("#crt").click(function(e) {
    var posX = $(this).offset().left;
    var posY = $(this).offset().top;
    crt.mouseX = e.pageX - posX;
    crt.mouseY = e.pageY - posY;
  });

  /**
   * Updates the screen to make the display fade.
   */
  this.dimScreen = function() {
    this.ctx.globalAlpha = .07;
    this.ctx.fillStyle = "black";
    this.ctx.fillRect(0, 0, this.canvas.width, this.canvas.height);
    this.ctx.globalAlpha = 1;
    this.ctx.strokeStyle = "#00ff00";
  };

  /**
   * The Whirlwind CRT character generator uses a seven-segment format with a bit in a seven-bit
   * word to indicate each segment.  This list defines the sequence in which the bits are
   * converted into line segments
   */
  this.WW_CHAR_SEQ = ["down", "right", "up", "left", "up", "right", "down"];

  // recall the most recent light gun display point so it can be erased when the next one comes up
  this.last_pen_point = null;
  // remember the location of the last unprocessed mouse click
  this.last_mouse = null;
  // remember the last character drawn
  this.last_crt_char = null;

  this.ww_scope_update = function() {
    // this.gfx.update();
  };

  this.ww_to_xwin_coords = function(ww_x, ww_y) {
    /**
     * change coordinates from WW to Windows
     * WW coords have zero at the center, and go to +/- 1023.  Plus is up and to the right
     * X-Windows coords have 0,0 in the top left corner.  More Positive is down and to the right
     */
    xwin_y = -ww_y/this.WW_MAX_COORD * (this.WIN_MAX_COORD/2.0) + this.WIN_MAX_COORD/2.0;
    xwin_x = ww_x/this.WW_MAX_COORD * (this.WIN_MAX_COORD/2.0) + this.WIN_MAX_COORD/2.0;

    return [Math.floor(xwin_x), Math.floor(xwin_y)];
  };

  /**
   * Seven-segment display
   */
  this.ww_draw_char = function(ww_x, ww_y, mask, bright=false) {
    var pt = this.ww_to_xwin_coords(ww_x, ww_y);
    var x = pt[0];
    var y = pt[1];
    var last_x = x;
    var last_y = y;
    for (var i = 0; i < 7; i++) {
      if (this.WW_CHAR_SEQ[i] == "down") {
        y = last_y + this.WW_CHAR_VSTROKE;
      } else if (this.WW_CHAR_SEQ[i] == "up") {
        y = last_y - this.WW_CHAR_VSTROKE;
      } else if (this.WW_CHAR_SEQ[i] == "left") {
        x = last_x - this.WW_CHAR_HSTROKE;
      } else if (this.WW_CHAR_SEQ[i] == "right") {
        x = last_x + this.WW_CHAR_HSTROKE;
      } else {
        console.log("OMG its a bug! WW_CHAR_SEQ[i]=" + this.WW_CHAR_SEQ[i]);
      }

      if (mask & 1 << (6 - i)) {
        this.ctx.strokeStyle = "#00ff00";
      } else {
        last_x = x;
        last_y = y;
        continue; // No drawing
      }

      this.ctx.beginPath();
      this.ctx.moveTo(last_x, last_y);
      this.ctx.lineTo(x, y);
      this.ctx.closePath();
      this.ctx.stroke();
      last_x = x;
      last_y = y;
    }
  };

  this.ww_draw_point = function(ww_x, ww_y, light_gun=false) {
    var pt = this.ww_to_xwin_coords(ww_x, ww_y);
    var x = pt[0];
    var y = pt[1];
    this.ctx.fillStyle = "#00ff00";
    this.ctx.beginPath();
    this.ctx.arc(x, y, 5, 0, 2 * Math.PI);
    this.ctx.closePath();
    this.ctx.fill();
    if (light_gun) {
      this.last_pen_point = [x, y]; // remember the point so it can be undrawn later
    }
  };

  this.ww_dim_previous_point = function() {
    if (self.last_pen_point) {
      this.ctx.fillStyle = "#00ff00";
      this.ctx.beginPath();
      this.ctx.arc(self.last_pen_point[0], self.last_pen_point[1], 5, 0, 2 * Math.PI);
      this.ctx.closePath();
      this.ctx.fill();
    }
  };

  this.ww_highlight_point = function() {
    if (self.last_pen_point) {
      this.ctx.fillStyle = "#ff0000";
      this.ctx.beginPath();
      this.ctx.arc(self.last_pen_point[0], self.last_pen_point[1], 5, 0, 2 * Math.PI);
      this.ctx.closePath();
      this.ctx.fill();
    }
  };

  this.checkMouse = function() {
    if (this.mouseX != null) {
      var pt = [this.mouseX, this.mouseY];
      this.mouseX = null;
      this.mouseY = null;
      return pt;
    } else {
      return null;
    }
  };

  this.ww_check_light_gun = function() {
    var pt = this.checkMouse()
    if (pt == null) {
      return [this.sim.cb.NO_ALARM, null]
    }
    if (this.last_pen_point == null) {
      alert("Light Gun checked, but no dot displayed");
      return [this.sim.cb.QUIT_ALARM, null];
    }

    console.log("dot (" + this.last_pen_point[0] + ", " + this.last_pen_point[1]
        + ");  mouse (" + pt[0] + ", " + pt[1] + ")");
    return [this.sim.cb.NO_ALARM, pt];
  };
};

// Global function

/**
 * Converts number to octal.
 * @param num {integer} - The argument.
 * @param length {integer} - Field length. Number is zero-padded to fit.
 */
function toOctal(num, length = 0) {
  if (typeof num == "undefined") {
    return "***UNDEFINED***";
  } else if (num == null) {
    return "***null***";
  }
  if (num < 0) {
    num = 0xffff^(-num);
  }
  var s = num.toString(8);
  while (length && s.length < length) {
    s = "0" + s;
  }
  return s;
}
