			;
			;--------------------------------------------------------------
			;	File:	Micral 8022G.ROM
			;		EPROM 2716 Enabled in [F000..FFEF]
			;
			;	CPU:	Zilog Z80
			;		F = 11,9808 /4 = 2,9952 MHz
			;
			;	Date:	2023-08-17
			;
			;--------------------------------------------------------------
			;
			VideoExt1	EQU	0FFF6H		; Video external board
			VideoExt2	EQU	0FFF7H		; Video external board
			;
			CrtDataWr	EQU	0FFF8H		; Data for video controler
			CrtCharRd	EQU	0FFF8H		; Read Crt character
			CrtCmdWr	EQU	0FFF9H		; Command to video controler
			CrtStatRd	EQU	0FFF9H		; Crt Status bits
				; bit 0 : Crt synchro started
				; bit 1 : Video mode ST26
				; bit 2 : Video mode ST27
			;
			PrtDataWr	EQU	0FFFAH		; Printer data out
			;
			KeyDataRd	EQU	0FFFAH		; Keyboard data
			;
			UartData	EQU	0FFFBH		; RD/WR Uart Data
			UartHdshkWr	EQU	0FFFCH		; UART Handshake
				; bit 6 : DTR, Data Terminal Ready
				; bit 7 : RTR, Ready To Receive
			;
			DevStatRd	EQU	0FFFCH		; Status bits for Keyboard, UART, Printer 
				; bit 0 : Key pressed
				; bit 1 : Printer busy
				; bit 2 : UART data Rx valid
				; bit 3 : UART data Tx empty
				; bit 4 : UART CTS
				; bit 5 : UART DSR
				; bit 6 : UART DCD
				; bit 7 : Pinter port DB25 pin 2 
			;
			FdSelWr		EQU	0FFFDH		; Floppy drive and sector selector
				; bit 0-3 : Sector number
				; bit 4 : Enable drive 0
				; bit 5 : Enable drive 1
				; bit 6-7 : Enable drive 2-3
			;
			FdStatRd	EQU	0FFFDH		; Floppy status
				; bit 0 : Sector done
				; bit 1 : Track 0
				; bit 2 : Write protected
				; bit 3 : Double side switch
				; bit 4 : First cmd flag from reset (?)
			;
			FdCmdWr		EQU	0FFFEH		; Floppy drive and sector selector
				; bit 0 : Step direction
				; bit 1 : Step command
				; bit 2 : Floppy extension pinout
				; bit 3 : Motor
				; bit 4 : Data mode / clear bit memory count
				; bit 5 : Sector seek
				; bit 6 : Write start
				; bit 7 : Side select
			;
			FdData		EQU	0FFFFH		; Floppy data (bit 0)
			;
			;--------------------------------------------------------------
			TopStack	EQU	0110h

				ORG	0
				PHASE	0F800h
			;
			;--------------------------------------------------------------
			EpromBase:
F800 : C3 08 F8			jp	Start
			;
			Restart:
F803 : 0E 23			ld	c,'#'
F805 : CD 6D FB			call	EchoChar

			Start:
F808 : 31 10 01			ld	sp,TopStack	; 0110H
F80B : ED 56			im	1

F80D : 21 24 FA			ld	hl,TxtStart
F810 : 06 0D			ld	b,TxtStartLen
F812			TxtStartLoop:
F812 : 4E			ld	c,(hl)
F813 : CD 6D FB			call	EchoChar
F816 : 23			inc	hl
F817 : 10 F9			djnz	TxtStartLoop

F819 : CD 69 FB			call	EchoKeyWait
F81C : 47			ld	b,a		; B = key down
F81D : FE 0D			cp	0DH
F81F : CA 31 FA			jp	z,BootDefault	; jump if key = OD

F822 : 0E 3A			ld	c,':'
F824 : CD 6D FB			call	EchoChar	; echo :

F827 : 78			ld	a,b
F828 : FE 54			cp	'T'
F82A : 28 40			jr	z,Terminal	; jump if key = T

F82C : FE 42			cp	'B'
F82E : CA F5 F9			jp	z,KeyBoot	; jump if key = B

F831 : FE 47			cp	'G'
F833 : 28 09			jr	z,JumpCmd	; jump if key = G

F835 : FE 2A			cp	'*'
F837 : 20 CA			jr	nz,Restart	; jump if key <> *

				; Echo key forever
			FatalLoop:
F839 : CD 69 FB			call	EchoKeyWait
F83C : 18 FB			jr	FatalLoop
			;
			;--------------------------------------------------------------
			; Key = G
			;--------------------------------------------------------------
			JumpCmd:
F83E : 11 00 00			ld	de,0000H	; DE = address to jump
F841			JumpCmdLoop:
F841 : CD 69 FB			call	EchoKeyWait
F844 : FE 0D			cp	00DH
F846 : 28 18			jr	z,JumpCmdExec

F848 : D6 30			sub	'0'
F84A : FE 0A			cp	10
F84C : 38 02			jr	c,JumpCmd10	; jump if char <= 9
F84E : D6 07			sub	7		; A = 10, etc
F850			JumpCmd10:
F850 : 06 10			ld	b,16
F852 : B8			cp	b
F853 : 30 AE			jr	nc,Restart	; Restart if value >=16
F855 : 6F			ld	l,a
F856 : 26 00			ld	h,0		; HL = value typed
F858			JumpCmdX16:
F858 : 19			add	hl,de		; HL = HL + DE*16
F859 : 38 A8			jr	c,Restart	; Restart if OV
F85B : 10 FB			djnz	JumpCmdX16

F85D : EB			ex	de,hl		; DE = HL
F85E : 18 E1			jr	JumpCmdLoop
			;
F860			JumpCmdExec:
F860 : EB			ex	de,hl		; HL = DE
F861			JumpToHLinRAM:
F861 : 3E C3			ld	a,0C3H		; C3 = code jmp
				; Write to RAM
F863 : 32 ED FF			ld	(RamCode1),a
F866 : 22 EE FF			ld	(RamCode2),hl

F869 : C3 EB FF			jp	JumpCode
			;
			;--------------------------------------------------------------
			; Key = T
			;--------------------------------------------------------------
			Terminal:
F86C : 21 F4 EF			ld	hl,0EFF4H
F86F : E5			push	hl
F870 : 06 0C			ld	b,12
F872			Term2:
F872 : CD 69 FB			call	EchoKeyWait
F875 : 77			ld	(hl),a		; Write key to EFF4..EFFF
F876 : FE 0D			cp	0DH
F878 : 28 06			jr	z,TermStart		; jump if OD

F87A : 23			inc	hl
F87B : 10 F5			djnz	Term2

F87D			ColdRestart:
F87D : C3 03 F8			jp	Restart		; Restart if more than 11 char typed
			;
F880			TermStart:
F880 : 06 11			ld	b,11H
F882 : CD 62 F9			call	SerialSendRegB
F885 : CD 3C F9			call	SerialWait96B4
F888 : 3E 01			ld	a,01H
F88A : CD 2E F9			call	SerialOutIn
F88D : 3E 20			ld	a,20H
F88F : CD 2E F9			call	SerialOutIn
F892 : AF			xor	a
F893 : CD 2E F9			call	SerialOutIn

F896 : 06 14			ld	b,14H
F898 : AF			xor	a
F899			LF899:
F899 : CD 2E F9			call	SerialOutIn
F89C : 10 FB			djnz	LF899

F89E : E1			pop	hl
				; HL = EFF4 start of typed chars
F89F : 06 0C			ld	b,12
F8A1			LF8A1:
F8A1 : 7E			ld	a,(hl)
F8A2 : CD 2E F9			call	SerialOutIn
F8A5 : 23			inc	hl
F8A6 : 10 F9			djnz	LF8A1

F8A8 : CD 8E F9			call	SerialSendC6
F8AB : CD 51 F9			call	SerialWaitA5
F8AE : A7			and	a
F8AF : 20 CC			jr	nz,ColdRestart

F8B1 : CD 2E F9			call	SerialOutIn
F8B4 : CD 2E F9			call	SerialOutIn
F8B7 : 4F			ld	c,a
F8B8 : CD 2E F9			call	SerialOutIn
F8BB : CD 2E F9			call	SerialOutIn

F8BE : D9			exx
F8BF : 21 00 00			ld	hl,0000H
F8C2 : D9			exx

F8C3 : 16 01			ld	d,1
F8C5 : CD 94 F9			call	LF994
F8C8 : 5F			ld	e,a
F8C9 : 1D			dec	e
F8CA : 1D			dec	e
F8CB : CD 94 F9			call	LF994
F8CE : FE D0			cp	0D0H
F8D0 : 20 AB			jr	nz,ColdRestart

F8D2 : 06 02			ld	b,02H
F8D4			LF8D4:
F8D4 : CD 94 F9			call	LF994
F8D7 : A7			and	a
F8D8 : 20 A3			jr	nz,ColdRestart

F8DA : 10 F8			djnz	LF8D4

F8DC : 1D			dec	e
F8DD : 20 9E			jr	nz,ColdRestart
F8DF			LF8DF:
F8DF : CD 94 F9			call	LF994
F8E2 : 5F			ld	e,a
F8E3 : 1D			dec	e
F8E4 : CD 94 F9			call	LF994
F8E7 : FE C6			cp	0C6H
F8E9 : 28 2F			jr	z,LF91A

F8EB : FE C2			cp	0C2H
F8ED : C2 03 F8			jp	nz,Restart

F8F0 : 1D			dec	e
F8F1 : 1D			dec	e
F8F2 : 1D			dec	e
F8F3 : CD 94 F9			call	LF994
F8F6 : 67			ld	h,a
F8F7 : CD 94 F9			call	LF994
F8FA : 6F			ld	l,a
F8FB : CD 94 F9			call	LF994
F8FE			LF8FE:
F8FE : 47			ld	b,a
F8FF : 7C			ld	a,h
F900 : FE 01			cp	01H
F902 : 20 0B			jr	nz,LF90F

F904 : 7D			ld	a,l
F905 : FE 10			cp	10H
F907 : 30 06			jr	nc,LF90F

F909 : 78			ld	a,b
F90A : CD 94 F9			call	LF994
F90D : 18 05			jr	LF914
			;
F90F			LF90F:
F90F : 78			ld	a,b
F910 : CD 94 F9			call	LF994
F913 : 77			ld	(hl),a
F914			LF914:
F914 : 23			inc	hl
F915 : 1D			dec	e
F916 : 20 E6			jr	nz,LF8FE

F918 : 18 C5			jr	LF8DF
			;
			;--------------------------------------------------------------
F91A			LF91A:
F91A : CD 94 F9			call	LF994
F91D : 67			ld	h,a
F91E : CD 94 F9			call	LF994
F921 : 6F			ld	l,a
F922			LF922:
F922 : CD 2E F9			call	SerialOutIn
F925 : 15			dec	d
F926 : 20 FA			jr	nz,LF922

F928 : 32 FB FF			ld	(UartData),a
F92B : C3 61 F8			jp	JumpToHLinRAM
			;
			;--------------------------------------------------------------
			SerialOutIn:
F92E : 32 FB FF			ld	(UartData),a	; UART out
F931			SerialWaitIn:
F931 : 3A FC FF			ld	a,(DevStatRd)	; Read UART receive flag
F934 : E6 04			and	04H
F936 : 28 F9			jr	z,SerialWaitIn		; loop if bit 2 = 0

F938 : 3A FB FF			ld	a,(UartData)	; Read UART
F93B : C9			ret
			;
			;--------------------------------------------------------------
			; Wait protocol : wait 96, out A5, wait B4, out 00
			;--------------------------------------------------------------
F93C			SerialWait96B4:
F93C : CD 31 F9			call	SerialWaitIn
F93F : FE 96			cp	096H
F941 : 20 F9			jr	nz,SerialWait96B4

F943 : 3E A5			ld	a,0A5H
F945 : 32 FB FF			ld	(UartData),a
F948			SerialWaitB4:
F948 : CD 31 F9			call	SerialWaitIn
F94B : D6 B4			sub	0B4H
F94D : 20 F9			jr	nz,SerialWaitB4

F94F : 18 DD			jr	SerialOutIn

			;--------------------------------------------------------------
			; Wait protocol : wait A5, out B4 x5
			;--------------------------------------------------------------
F951			SerialWaitA5:
F951 : CD 31 F9			call	SerialWaitIn
F954 : FE A5			cp	0A5H
F956 : 20 F9			jr	nz,SerialWaitA5

F958 : 3E B4			ld	a,0B4H
F95A : 06 05			ld	b,5

F95C			SerialSendB4:
F95C : CD 2E F9			call	SerialOutIn
F95F : 10 FB			djnz	SerialSendB4

F961 : C9			ret
			;
			;--------------------------------------------------------------
F962			SerialSendRegB:
F962 : CD 3C F9			call	SerialWait96B4

F965 : 3E 81			ld	a,081H
F967 : CD 2E F9			call	SerialOutIn

F96A : 3E 03			ld	a,003H
F96C : CD 2E F9			call	SerialOutIn

F96F : AF			xor	a
F970 : CD 2E F9			call	SerialOutIn

F973 : 78			ld	a,b
F974 : CD 2E F9			call	SerialOutIn

F977 : 3E 0D			ld	a,0DH
F979 : CD 2E F9			call	SerialOutIn

F97C : 3E 01			ld	a,01H
F97E : CD 2E F9			call	SerialOutIn

F981 : CD 8E F9			call	SerialSendC6

F984 : CD 51 F9			call	SerialWaitA5
F987 : A7			and	a
F988 : C2 03 F8			jp	nz,Restart

F98B : CD 2E F9			call	SerialOutIn

F98E			SerialSendC6:
F98E : 3E C6			ld	a,0C6H
F990 : 32 FB FF			ld	(UartData),a
F993 : C9			ret
			;
			;--------------------------------------------------------------
			; 
			;--------------------------------------------------------------
F994			LF994:
F994 : CD 2E F9			call	SerialOutIn
F997 : 15			dec	d
F998 : C0			ret	nz

F999 : 50			ld	d,b
F99A : 32 FB FF			ld	(UartData),a

F99D : 06 10			ld	b,010H
F99F : CD 62 F9			call	SerialSendRegB

F9A2 : CD 3C F9			call	SerialWait96B4

F9A5 : 3E 01			ld	a,001H
F9A7 : CD 2E F9			call	SerialOutIn

F9AA : 3E 0D			ld	a,00DH
F9AC : CD 2E F9			call	SerialOutIn

F9AF : AF			xor	a		; A = 0
F9B0 : CD 2E F9			call	SerialOutIn

F9B3 : 3E 60			ld	a,60H
F9B5 : CD 2E F9			call	SerialOutIn

F9B8 : AF			xor	a
F9B9 : 06 04			ld	b,4
F9BB			LF9BB:
F9BB : CD 2E F9			call	SerialOutIn
F9BE : 10 FB			djnz	LF9BB

F9C0 : 3E 07			ld	a,07H
F9C2 : CD 2E F9			call	SerialOutIn

F9C5 : 79			ld	a,c
F9C6 : CD 2E F9			call	SerialOutIn

F9C9 : AF			xor	a
F9CA : 06 03			ld	b,003H
F9CC			LF9CC:
F9CC : CD 2E F9			call	SerialOutIn
F9CF : 10 FB			djnz	LF9CC

F9D1 : 3C			inc	a
F9D2 : CD 2E F9			call	SerialOutIn
F9D5 : D9			exx
F9D6 : 7D			ld	a,l
F9D7 : CD 2E F9			call	SerialOutIn

F9DA : 7C			ld	a,h
F9DB : CD 2E F9			call	SerialOutIn

F9DE : 23			inc	hl
F9DF : D9			exx
F9E0 : CD 8E F9			call	SerialSendC6

F9E3 : CD 51 F9			call	SerialWaitA5
F9E6 : A7			and	a
F9E7 : C2 03 F8			jp	nz,Restart

F9EA : 06 05			ld	b,005H
F9EC			LF9EC:
F9EC : CD 2E F9			call	SerialOutIn
F9EF : 10 FB			djnz	LF9EC
F9F1 : 42			ld	b,d
F9F2 : 16 00			ld	d,0
F9F4 : C9			ret
			;
			;--------------------------------------------------------------
			; Key = B : boot [drive,] track ENTER
			;--------------------------------------------------------------
			KeyBoot:
F9F5 : 1E 00			ld	e,0		; drive 0 by default
F9F7 : CD 69 FB			call	EchoKeyWait	; Wait from drive number

F9FA : D6 30			sub	'0'
F9FC : FE 0A			cp	10
F9FE : D2 03 F8			jp	nc,Restart	; jump if char >=10

FA01 : 57			ld	d,a		; d = [0..9]

FA02 : CD 69 FB			call	EchoKeyWait
FA05 : FE 2C			cp	','
FA07 : 20 06			jr	nz,KeyBootDrv0	; jump if no comma

FA09 : 5A			ld	e,d		; E = drive unit
FA0A : 16 00			ld	d,0		; D = track

FA0C			KeyBootTrack:
FA0C : CD 69 FB			call	EchoKeyWait
FA0F			KeyBootDrv0:
FA0F : D6 0D			sub	0DH
FA11 : 28 21			jr	z,BootDE	; jump if ENTER, boot track D on drive E

FA13 : D6 23			sub	'0'-0Dh
FA15 : 06 0A			ld	b,10
FA17 : B8			cp	b
FA18 : D2 03 F8			jp	nc,Restart	; jump if char >=10

				; Calc D = A + 10*D
FA1B			KeyBootMult:
FA1B : 82			add	a,d
FA1C : DA 03 F8			jp	c,Restart	; jump if overflow
FA1F : 10 FA			djnz	KeyBootMult

FA21 : 57			ld	d,a
FA22 : 18 E8			jr	KeyBootTrack

			;
			;--------------------------------------------------------------
			TxtStart:
FA24 : 1F 0A 0D			DB	1Fh, 0Ah, 0Dh
FA27 : 4D 49 43			DB	'MICRAL... '
FA2A : 52 41 4C
FA2D : 2E 2E 2E
FA30 : 20
			TxtStartLen	EQU	$-TxtStart


			;--------------------------------------------------------------
			; Boot disk, E = drive, D = track
			;--------------------------------------------------------------
			BootDefault:
FA31 : 11 00 08			ld	de,0800H	; E = drive 0, D = track 8

FA34			BootDE:
FA34 : 7B			ld	a,e
FA35 : FE 02			cp	2
FA37 : D2 03 F8			jp	nc,Restart	; jump if drive >=2

FA3A : A7			and	a
FA3B : 5A			ld	e,d		; E = track
FA3C : 16 10			ld	d,10H		; D = 10h if drive = 0
FA3E : 28 02			jr	z,BootDrvsel

FA40 : 16 20			ld	d,20H		; D = 20h if drive = 1
FA42			BootDrvsel:
FA42 : D5			push	de		; Save in pseudo global var
				; D = (TopStack -1 = 010Fh) = track
				; E = (TopStack -2 = 010Eh) = drive
FA43 : 7A			ld	a,d
FA44 : 32 FD FF			ld	(FdSelWr),a	; Write drive select

FA47 : DD 21 5D FB	  	ld	ix,IXcmdSide1	; (IXcmdSide1) = 0C

FA4B : 3E 0C			ld	a,0CH
FA4D : 32 FE FF			ld	(FdCmdWr),a	; bit 3 on : Start Motor

FA50 : 0E 85			ld	c,133
FA52 : CD 51 FB			call	DelayC		; Wait 1000 mS

FA55 : 0E 00			ld	c,0

FA57			SeekTrack0:
				; Search track 0
FA57 : 3A FD FF			ld	a,(FdStatRd)	; Read Drive status
FA5A : E6 02			and	02H		; Test track 0
FA5C : 20 07			jr	nz,SeekTrack0Done	; jump if bit 1 = 0

FA5E : 1E 00			ld	e,0		; Set head dir to -1
FA60 : CD 1D FB			call	TrackSkip	; Move to track -1
FA63 : 18 F2			jr	SeekTrack0

FA65			SeekTrack0Done:
				; Head in track 0
FA65 : 3A FD FF			ld	a,(FdStatRd)
FA68 : E6 08			and	08H		; Test Switch ST6

FA6A : 3A 0E 01			ld	a,(TopStack -2)	; A = (010Eh) = track
FA6D : 28 06			jr	z,FdSectorFirst	; jump if bit 3 = 0 (single side drive)

				; double side drive : div track by 2, C=16 if odd track
FA6F : A7			and	a		; Carry = 0
FA70 : 1F			rra			; >>, bit 7 = 0, C = bit 0
FA71 : 30 02			jr	nc,FdSectorFirst	; jump if bit 0 = 0
				; A = A/2
FA73 : 0E 10			ld	c,16

FA75			FdSectorFirst:
FA75 : 47			ld	b,a
FA76 : 50			ld	d,b		; D = B = track to find
				; Single side : C = 0
				; Double side : C = 16

			;--------------------------------------------------------------
			; B = current track
			; D = track uncount
			; C = sector to read
			BootLoop:
FA77 : 15			dec	d
FA78 : F2 15 FB			jp	p,SkipNextTrack	; jump if D = [0..7F]

				; D = 0FFh
				; Track found
FA7B			FdReadSect:
FA7B : C5			push	bc		; Save track, sector

FA7C : 50			ld	d,b
FA7D : 59			ld	e,c		; E = sector
FA7E : 7B			ld	a,e		; A = sector
FA7F : E6 10			and	10H		; Test if sector >=16 : read face 2

FA81 : DD 21 5D FB	  	ld	ix,IXcmdSide1	; (IXcmdSide1) = 0C
FA85 : 28 08			jr	z,FdReadSectSeek	; jump if bit 4 = 0

				; Double sided : A, C = 10H
FA87 : 7B			ld	a,e
FA88 : D6 10			sub	16
FA8A : 5F			ld	e,a		; E = sector - 16 on face 2
FA8B : DD 21 5E FB	  	ld	ix,IXcmdSide2	; (IXcmdSide2) = 8C

FA8F			FdReadSectSeek:
FA8F : DD 7E 00			ld	a,(ix+0)
FA92 : 21 FE FF			ld	hl,FdCmdWr
FA95 : 77			ld	(hl),a		; Write cmd IXcmdSide1  or IXcmdSide2

FA96 : 3A 0F 01			ld	a,(TopStack -1)	; (010Fh) = drive select
FA99 : B3			or	e		; OR Sector num.
FA9A : 2B			dec	hl
				; HL = FFFD, FdSelWr
FA9B : 77			ld	(hl),a		; Write drive/sector select

FA9C : 23			inc	hl
				; HL = FFFE, FdCmdWr
FA9D : DD 7E 00			ld	a,(ix+0)
FAA0 : F6 20			or	20H		; bit 5 : start sector seek
FAA2 : 77			ld	(hl),a
FAA3 : 2B			dec	hl
				; HL = FFFD, FdStatRd

FAA4			FdWaitSector:
FAA4 : 7E			ld	a,(hl)		; Read FD status
FAA5 : 1F			rra
FAA6 : 30 FC			jr	nc,FdWaitSector	; loop if sector not found

FAA8 : 23			inc	hl
				; HL = FFFE, FdCmdWr
FAA9 : 36 1C			ld	(hl),1CH	; Clear data cnt

FAAB : 23			inc	hl
				; HL = FFFF, FdData
FAAC : 7E			ld	a,(hl)		; pop first bit

FAAD : CD 37 FB			call	FdGetByte	; dummy read

FAB0 : CD 37 FB			call	FdGetByte	; Read track tag
FAB3 : 92			sub	d		; Sub desired track number
FAB4 : 57			ld	d,a		; D = A, should be 0, if no error

FAB5 : CD 37 FB			call	FdGetByte	; Read sector tag

FAB8 : C1			pop	bc
FAB9 : 91			sub	c		; C = Sector
FABA : B2			or	d
FABB : C5			push	bc
FABC : 20 2B			jr	nz,FdCheckErr2	; jump if track or sector not good

FABE : 42			ld	b,d		; Check byte B = 0

FABF : CD 37 FB			call	FdGetByte	; Get block size

FAC2 : F5			push	af
FAC3 : F5			push	af

FAC4 : CD 37 FB			call	FdGetByte	; Get RAM adr MSB
FAC7 : 57			ld	d,a

FAC8 : CD 37 FB			call	FdGetByte	; Get RAM adr LSB
FACB : 5F			ld	e,a

FACC			FdReadBlock:
FACC : F1			pop	af		; POP block size count
FACD : A7			and	a
FACE : 28 1D			jr	z,FdReadBlockEnd

FAD0 : 3D			dec	a		; byte count -1
FAD1 : F5			push	af

				; Test adr DE limits
FAD2 : 7A			ld	a,d
FAD3 : FE 01			cp	01H
FAD5 : 20 0A			jr	nz,FdReadNextByte

FAD7 : 7B			ld	a,e
FAD8 : FE 10			cp	10H
FADA : 30 05			jr	nc,FdReadNextByte

				; Read byte without memory write if DE in [0100h..0110h]
FADC : CD 37 FB			call	FdGetByte
FADF : 18 04			jr	FdSkipNextByte
			;
FAE1			FdReadNextByte:
FAE1 : CD 37 FB			call	FdGetByte
FAE4 : 12			ld	(de),a		; Write read byte in memory

FAE5			FdSkipNextByte:
FAE5 : 13			inc	de
FAE6 : 18 E4			jr	FdReadBlock
			;
FAE8			FdCheckErr:
FAE8 : F1			pop	af
FAE9			FdCheckErr2:
FAE9 : C1			pop	bc
FAEA : C3 7B FA			jp	FdReadSect	; Try to read again
			;
FAED			FdReadBlockEnd:
FAED : C5			push	bc
FAEE : CD 37 FB			call	FdGetByte	; Get check byte
FAF1 : C1			pop	bc

FAF2 : B8			cp	b		; Check control
FAF3 : 20 F3			jr	nz,FdCheckErr	; Jump if bad check

FAF5 : F1			pop	af		; POP block size count
FAF6 : A7			and	a
FAF7 : 20 09			jr	nz,FdIncSector	; BUG ?, A can't be <>0 at this point...

				; DE = start address
FAF9 : EB			ex	de,hl

FAFA : 3E 08			ld	a,08H
FAFC : 32 FE FF			ld	(FdCmdWr),a	; Reset ext pinout, keep motor on

FAFF : C3 61 F8			jp	JumpToHLinRAM	; Execute code in RAM
			;
FB02			FdIncSector:
FB02 : C1			pop	bc
FB03 : 0C			inc	c		; C = sector number +1

FB04 : 3A FD FF			ld	a,(FdStatRd)
FB07 : E6 08			and	08H		; Test Switch ST6, single/double side floppy
FB09 : 3E 10			ld	a,16
FB0B : 28 01			jr	z,FdIncSectorF2

FB0D : 07			rlca			; A = 32

FB0E			FdIncSectorF2:
FB0E : 91			sub	c		; Test if last sector in the track
FB0F : C2 7B FA			jp	nz,FdReadSect	; Jum if not last

FB12 : 4F			ld	c,a		; C = sector 0
FB13 : 04			inc	b		; B = Track +1
FB14 : 51			ld	d,c		; D = 0

FB15			SkipNextTrack:
FB15 : 1E 01			ld	e,1
FB17 : CD 1D FB			call	TrackSkip	; Move head track +1
FB1A : C3 77 FA			jp	BootLoop
			;
			;--------------------------------------------------------------
			; Move drive head by write at FFFE : E = 0 or E = 1 (direction)
			;--------------------------------------------------------------
			TrackSkip:
FB1D : 21 FE FF			ld	hl,FdCmdWr
FB20 : DD 7E 00			ld	a,(ix+000H)	; A = 0Ch or 8Ch
FB23 : B3			or	e
FB24 : 77			ld	(hl),a
FB25 : F6 02			or	02H		; set bit 1 : Pulse 26 uS
FB27 : 77			ld	(hl),a
				; Delay
FB28 : E3			ex	(sp),hl
FB29 : E3			ex	(sp),hl
FB2A : E3			ex	(sp),hl
FB2B : E3			ex	(sp),hl

FB2C : EE 02			xor	02H		; reset bit 1
FB2E : 77			ld	(hl),a

FB2F : D9			exx
FB30 : 0E 07			ld	c,7
FB32 : CD 51 FB			call	DelayC		; Wait 53 mS
FB35 : D9			exx

FB36 : C9			ret
			;
			;--------------------------------------------------------------
			; In : HL = FFFF, FdData
			; In/out : B = check byte
			; Return disk byte in A
			;--------------------------------------------------------------
			FdGetByte:
FB37 : 7E			ld	a,(hl)
FB38 : 0E 07			ld	c,7
FB3A			FdGetByte2:
FB3A : 07			rlca
FB3B : A6			and	(hl)
FB3C : 0D			dec	c
FB3D : 20 FB			jr	nz,FdGetByte2

FB3F : 2F			cpl

FB40 : 4F			ld	c,a
FB41 : A8			xor	b
FB42 : 78			ld	a,b
FB43 : F2 4A FB			jp	p,FdGetByte3

FB46 : 81			add	a,c
FB47 : 47			ld	b,a
FB48 : 79			ld	a,c
FB49 : C9			ret
			;
			;--------------------------------------------------------------
FB4A			FdGetByte3:
FB4A : 81			add	a,c
FB4B : 47			ld	b,a
FB4C : A9			xor	c
FB4D : 79			ld	a,c
FB4E : F0			ret	p

FB4F : 04			inc	b
FB50 : C9			ret
			;
			;--------------------------------------------------------------
			; Delay C
			; T = (C-1) * (4+5+7 + 256*(4*19 + 13) +8+12) +4+11 cycles
			; At 4 Mhz : T = (C-1) * 5,7mS
			; At 2,9952 MHz : T = (C-1) * 7,6mS
			;--------------------------------------------------------------
			DelayC:
FB51 : 0D			dec	c
FB52 : C8			ret	z
FB53 : 06 00			ld	b,0
FB55			DelayC2:
FB55 : E3			ex	(sp),hl		; Take time
FB56 : E3			ex	(sp),hl
FB57 : E3			ex	(sp),hl
FB58 : E3			ex	(sp),hl
FB59 : 10 FA			djnz	DelayC2		; loop 256 times

FB5B : 18 F4			jr	DelayC
			;
			;--------------------------------------------------------------
			; Two data byte used by IX pointeur for drive command
			;--------------------------------------------------------------
			IXcmdSide1:
FB5D : 0C			DB	0Ch	; Cmd and select side 1
			IXcmdSide2:
FB5E : 8C			DB	8Ch	; Cmd and select side 2

			;--------------------------------------------------------------
			;
			; Wait key down and return ASCII in A
			;
			;--------------------------------------------------------------
			KeyWait:
FB5F : 3A FC FF			ld	a,(DevStatRd)
FB62 : 0F			rrca			; Bit 0 to Carry
FB63 : 30 FA			jr	nc,KeyWait	; loop if Bit 0 not set

FB65 : 3A FA FF			ld	a,(KeyDataRd)	; read char
FB68 : C9			ret

			;--------------------------------------------------------------
			;
			; Wait key down, write to CRT and return ASCII in C
			;
			;--------------------------------------------------------------
			EchoKeyWait:
FB69 : CD 5F FB			call	KeyWait
FB6C : 4F			ld	c,a

			;--------------------------------------------------------------
			;
			; Write char C to CRT
			;
			; Send data to TMS9927 :
			;  1) Write data to CrtDataWr (FFF8)
			;  2) Write register 0..0F to CrtCmdWr (FFF9)
			;
			; CrtSetCurAndWriteC and CrtReadChar use special DE addresses :
			; - 1E40H : Init at 55h and tested
			; - 1F40H : Cursor position X
			; - 1F41H : Cursor position Y
			; - 1F42H : CRT Reg 6, bottom line number (-1)
			; - 1F43H : Init at 15h and tested
			; - 1F44H : flag 00h (Init) or 80h (reverse video), ORed with char to write
			;--------------------------------------------------------------
			EchoChar:
FB6D : E5			push	hl
FB6E : D5			push	de
FB6F : C5			push	bc

FB70 : 21 F9 FF			ld	hl,CrtStatRd	; FFF9H
FB73 : 36 B0			ld	(hl),0B0H	; Start Wait HSync and start Video RAM Read
							; Disable Prom P6 by writing 10h to address

FB75 : 06 00			ld	b,0		; 256 loop timeout
FB77
			EchoCharWait:
FB77 : 7E			ld	a,(hl)
FB78 : 0F			rrca
FB79 : 38 6E			jr	c,EchoCharOk	; Wait CRT HSync, jump if bit 0 at FFF9 is set

FB7B : 10 FA			djnz	EchoCharWait

				; timeout, no CRT HSync, bit 0 not set
			CrtInit:
FB7D : 36 0E			ld	(hl),0EH	; START TIMING CHAIN
FB7F : 36 0A			ld	(hl),0AH	; RESET
FB81 : 2B			dec	hl
				; HL = FFF8

FB82 : CD 71 FD			call	CrtGetConfGrafCol

FB85 : 3E 63			ld	a,99		; Text/Graf. B&W Mode
FB87 : 30 02			jr	nc,CrtConfTxt

FB89 : 3E 6E			ld	a,110		; Graf. Color Mode
FB8B			CrtConfTxt:
FB8B : 77			ld	(hl),a		; Set H char count + H retrace
FB8C : 23			inc	hl
				; HL = FFF9
FB8D : 36 00			ld	(hl),00H	; Write data in reg 0

FB8F : 2B			dec	hl
				; HL = FFF8
FB90 : CD 71 FD			call	CrtGetConfGrafCol
							; Text/Graf. B&W Mode
FB93 : 3E 53			ld	a,53h		; bit 2-0 = 011 : 3 H sync delay
							; bit 6-3 = 1010 : 10 H sync width
FB95 : 30 02			jr	nc,CrtConfBW
							; Graf. Color Mode
FB97 : 3E 7F			ld	a,7Fh		; bit 2-0 = 111 : 7 H sync delay
							; bit 6-3 = 1111 : 15 H sync width
FB99			CrtConfBW:
FB99 : 77			ld	(hl),a
FB9A : 23			inc	hl
				; HL = FFF9
FB9B : 36 01			ld	(hl),1		; Write data in reg 1

FB9D : 2B			dec	hl
				; HL = FFF8
FB9E : 36 65			ld	(hl),65H	; bit 2-0 = 101 : 80 characters/data row
							; bit 6-3 = 1100 : 12 Scans/data row
FBA0 : 23			inc	hl
				; HL = FFF9
FBA1 : 36 02			ld	(hl),2		; Write data in reg 2

FBA3 : 2B			dec	hl
				; HL = FFF8
FBA4 : 06 17			ld	b,23		; bit 5-0 = 10111 : 23 data rows/frame
							; bit 7-6 = 00 : 0 HSkew bits
FBA6 : 70			ld	(hl),b
FBA7 : 23			inc	hl
				; HL = FFF9
FBA8 : 36 03			ld	(hl),3		; Write data in reg 3

FBAA : 2B			dec	hl
				; HL = FFF8
FBAB : CD 71 FD			call	CrtGetConfGrafCol
							; Text/Graf. B&W Mode
FBAE : 3E 28			ld	a,40		; 2*40+256 = 336 Scan lines/frame
FBB0 : 30 02			jr	nc,CrtConf40
							; Graf. Color Mode
FBB2 : 3E 60			ld	a,96		; 2*96 + 256 = 448 Scan lines/frame
FBB4			CrtConf40:
FBB4 : 77			ld	(hl),a
FBB5 : 23			inc	hl
				; HL = FFF9
FBB6 : 36 04			ld	(hl),4		; Write data in reg 4

FBB8 : 2B			dec	hl
				; HL = FFF8
FBB9 : CD 71 FD			call	CrtGetConfGrafCol
FBBC : 3E 13			ld	a,013H		; Text/Graf. B&W Mode
FBBE : 30 02			jr	nc,CrtConf13
FBC0 : 3E 50			ld	a,050H		; Graf. Color Mode
FBC2			CrtConf13:
FBC2 : 77			ld	(hl),a		; Vertical data start
FBC3 : 23			inc	hl
				; HL = FFF9
FBC4 : 36 05			ld	(hl),5		; Write data in reg 5

FBC6 : 2B			dec	hl
				; HL = FFF8
FBC7 : 70			ld	(hl),b		; 23 = Last displayed data row -1
FBC8 : 23			inc	hl
				; HL = FFF9
FBC9 : 36 06			ld	(hl),6		; Write data in reg 6

FBCB : 36 0E			ld	(hl),0EH	; START TIMING CHAIN

FBCD : CD 55 FC			call	CrtInit2

FBD0 : C5			push	bc
FBD1 : 11 42 1F			ld	de,1F42H	; Y = 31, X = 66
FBD4 : 48			ld	c,b		; C = B = 23 = Last displayed data row -1
FBD5 : CD 96 FD			call	CrtSetCurAndWriteC

				; Init VRAM Test 1
FBD8 : 11 40 1E			ld	de,1E40H	; Y = 30, X = 64
FBDB : 0E 55			ld	c,55H
FBDD : CD 96 FD			call	CrtSetCurAndWriteC

				; Init VRAM Test 2
FBE0 : 11 43 1F			ld	de,1F43H	; Y = 31, X = 67
FBE3 : 0E 15			ld	c,15H
FBE5 : CD 96 FD			call	CrtSetCurAndWriteC

FBE8 : C1			pop	bc
				; Crt Init Ended

			;--------------------------------------------------------------
			EchoCharOk:
				; VRAM Test 1
FBE9 : 11 40 1E			ld	de,1E40H	; X = 30, Y = 64
FBEC : CD 89 FD			call	CrtReadChar
FBEF : FE 55			cp	055H
FBF1 : 20 8A			jr	nz,CrtInit

				; VRAM Test 2
FBF3 : 11 43 1F			ld	de,1F43H	; X = 31, Y = 67
FBF6 : CD 89 FD			call	CrtReadChar
FBF9 : FE 15			cp	015H
FBFB : 20 80			jr	nz,CrtInit

				; Read cursor pos XY and bottom line number
FBFD : 1B			dec	de		; DE = 1F42h
FBFE : CD 89 FD			call	CrtReadChar
FC01 : 47			ld	b,a		; B = bottom line number -1

FC02 : 1B			dec	de		; DE = 1F41h
FC03 : CD 89 FD			call	CrtReadChar
FC06 : F5			push	af		; Read cursor pos Y

FC07 : 1B			dec	de		; DE = 1F40h
FC08 : CD 89 FD			call	CrtReadChar	; Read cursor pos X
FC0B : D1			pop	de		; D = Y
FC0C : 5F			ld	e,a		; E = X
				; DE = Cursor Position, B = bottom line number -1

				; Test control char
FC0D : 79			ld	a,c		; C = char to write
FC0E : FE 0D			cp	00DH		; CR
FC10 : CA D4 FC			jp	z,CrtCurCR

FC13 : FE 0A			cp	00AH		; LF, line feed
FC15 : CA 25 FD			jp	z,CrtCurIncY

FC18 : FE 0C			cp	00CH		; FF, form feed
FC1A : 28 3C			jr	z,CrtClear	; Clear screen

FC1C : FE 08			cp	008H		; BS, back space
FC1E : CA D8 FC			jp	z,CrtCurDecX

FC21 : FE 06			cp	006H		; ACK, cursor X+1
FC23 : CA 1E FD			jp	z,CrtCurIncX

FC26 : FE 0B			cp	00BH		; VT, vertical up
FC28 : CA DE FC			jp	z,CrtCurDecY

FC2B : FE 0A			cp	00AH		; LF
FC2D : CA 25 FD			jp	z,CrtCurIncY	; ALREADY TESTED... bug...

FC30 : FE 05			cp	005H		; ENQ, Cursor Home (0, 0)
FC32 : CA D2 FC			jp	z,CrtCurHome

FC35 : FE 07			cp	007H		; Bell
FC37 : 20 05			jr	nz,EchoCharTestRV

				; Send beep pulse to speaker
FC39 : 36 08			ld	(hl),08H	; HL = FFF9
FC3B : C3 51 FD			jp	EchoCharEnd
			;
FC3E			EchoCharTestRV:
				; Test if reverse video control
FC3E : D6 1F			sub	01FH		; No reverse video char
FC40 : 28 07			jr	z,EchoChar1FH	; jump if A was ascii 31

FC42 : D6 FF			sub	0FFH		; inc A
FC44 : C2 E6 FC			jp	nz,EchoCharPrt	; jump if A was <>1

				; A was ascii 30, enable reverse video char
FC47 : 3E 80			ld	a,80H

FC49			EchoChar1FH:
				; Save A in 1F44H = 00h or 80h
FC49 : 4F			ld	c,a
FC4A : D5			push	de
FC4B : 11 44 1F			ld	de,1F44H
FC4E : CD 96 FD			call	CrtSetCurAndWriteC
FC51 : D1			pop	de
FC52 : C3 51 FD			jp	EchoCharEnd
			;
			;--------------------------------------------------------------
			;
			;--------------------------------------------------------------
			CrtInit2:
FC55 : E5			push	hl
FC56 : D5			push	de
FC57 : C5			push	bc
FC58			CrtClear:
FC58 : 2B			dec	hl
				; HL = FFF8, CrtDataWr
FC59 : 36 01			ld	(hl),01H	; Set bit 0
FC5B : 23			inc	hl
				; HL = FFF9, CrtCmdWr
FC5C : 36 0F			ld	(hl),0FH	; Video signal OFF by setting RETBL = 1

				; Clear screen
FC5E : 0E 20			ld	c,' '
FC60 : 16 17			ld	d,23		; y = 24 -1
FC62			CrtClear2:
FC62 : 1E 4F			ld	e,79		; x = 80 -1
FC64			CrtClear3:
FC64 : CD 96 FD			call	CrtSetCurAndWriteC
FC67 : 1D			dec	e		; dec x
FC68 : F2 64 FC			jp	p,CrtClear3

FC6B : 15			dec	d		; dec line
FC6C : F2 62 FC			jp	p,CrtClear2

FC6F : 0E 00			ld	c,0
FC71 : 11 44 1F			ld	de,1F44H	; Y = 31, X =  68
FC74 : CD 96 FD			call	CrtSetCurAndWriteC	; Set Flag Reverse Video char to 0

FC77 : CD 7C FD			call	CrtGetConfGraf
FC7A : 30 4C			jr	nc,CrtInit3	; Jump if Text Mode

				; Init Graf. Mode Color and B&W
				; With this config, job is done with FFF6 and FFF7
FC7C : 16 17			ld	d,23
FC7E			CrtInitGraf1:
FC7E : 1E 4F			ld	e,79
FC80			CrtInitGraf2:
FC80 : 06 0C			ld	b,0CH
FC82 : CD AE FD			call	CrtSetCurYX	; Reg B is not used by CrtSetCurYX
FC85			CrtInitGraf3:
FC85 : 7E			ld	a,(hl)		; HL = FFF9, Wait Sync
FC86 : 0F			rrca
FC87 : 30 FC			jr	nc,CrtInitGraf3

FC89 : CD 71 FD			call	CrtGetConfGrafCol
FC8C : 30 22			jr	nc,CrtInitGraf4	; Jump if Graf. B&W

				; Graf. Color Mode
FC8E : 3E 30			ld	a,30H
FC90 : B0			or	b
FC91 : 32 F7 FF			ld	(VideoExt2),a
FC94 : D6 10			sub	010H
FC96 : 47			ld	b,a

FC97 : AF			xor	a
FC98 : 32 F6 FF			ld	(VideoExt1),a

FC9B : 78			ld	a,b
FC9C : 32 F7 FF			ld	(VideoExt2),a
FC9F : D6 10			sub	010H
FCA1 : 47			ld	b,a

FCA2 : AF			xor	a
FCA3 : 32 F6 FF			ld	(VideoExt1),a

FCA6 : 78			ld	a,b
FCA7 : 32 F7 FF			ld	(VideoExt2),a
FCAA : D6 10			sub	010H
FCAC : 47			ld	b,a
FCAD : AF			xor	a
FCAE : 18 07			jr	CrtInitGraf5
			;
FCB0			CrtInitGraf4:
FCB0 : 3E 10			ld	a,010H
FCB2 : B0			or	b
FCB3 : 32 F7 FF			ld	(VideoExt2),a
FCB6 : AF			xor	a

FCB7			CrtInitGraf5:
FCB7 : 32 F6 FF			ld	(VideoExt1),a
FCBA : 36 A0			ld	(hl),0A0H	; Start Wait HSync and start Video RAM Write

FCBC : 05			dec	b
FCBD : F2 85 FC			jp	p,CrtInitGraf3

FCC0 : 1D			dec	e
FCC1 : F2 80 FC			jp	p,CrtInitGraf2

FCC4 : 15			dec	d
FCC5 : F2 7E FC			jp	p,CrtInitGraf1

FCC8			CrtInit3:
FCC8 : 7E			ld	a,(hl)
FCC9 : 0F			rrca
FCCA : 30 FC			jr	nc,CrtInit3	; Wait HSync

				; Activate screen
FCCC : 2B			dec	hl
				; HL = FFF8
FCCD : 36 00			ld	(hl),0
FCCF : 23			inc	hl
				; HL = FFF9
FCD0 : 36 0F			ld	(hl),0FH	; Normal screen, RETBL = 0

FCD2			CrtCurHome:
FCD2 : 16 00			ld	d,0

FCD4			CrtCurCR:
FCD4 : 1E 00			ld	e,0
FCD6 : 18 79			jr	EchoCharEnd
			;
FCD8			CrtCurDecX:
FCD8 : 1D			dec	e
FCD9 : F2 51 FD			jp	p,EchoCharEnd
FCDC : 1E 4F			ld	e,79

FCDE			CrtCurDecY:
FCDE : 15			dec	d
FCDF : F2 51 FD			jp	p,EchoCharEnd

FCE2 : 16 17			ld	d,23
FCE4 : 18 6B			jr	EchoCharEnd
			;
FCE6			EchoCharPrt:
				; Ascii code conversion
FCE6 : E5			push	hl
FCE7 : C5			push	bc
FCE8 : 79			ld	a,c
FCE9 : 21 BD FD			ld	hl,TabCharAscii
FCEC : 01 10 00			ld	bc,TabCharAsciiLen	; = 16

FCEF : ED B1			cpir			; if A<>(HL) inc HL, dec BC until BC=0
FCF1 : 20 0A			jr	nz,EchoCharPrt2	; jump char not found

FCF3 : 21 CD FD			ld	hl,TabCharReplace
FCF6 : 09			add	hl,bc
FCF7 : 7E			ld	a,(hl)
FCF8 : C1			pop	bc
FCF9 : E1			pop	hl
FCFA : 4F			ld	c,a
FCFB : 18 06			jr	EchoCharPrt3
			;
FCFD			EchoCharPrt2:
FCFD : C1			pop	bc
FCFE : E1			pop	hl

FCFF : FE 20			cp	' '
FD01 : 38 4E			jr	c,EchoCharEnd	; jump if char <' '

FD03			EchoCharPrt3:
FD03 : D5			push	de
FD04 : 11 44 1F			ld	de,1F44H
FD07 : CD 89 FD			call	CrtReadChar	; Read reverse video flag bit 7

FD0A : B1			or	c		; Set reverse video char if set
FD0B : 4F			ld	c,a		; C = char to write

FD0C : D1			pop	de
FD0D : D5			push	de

FD0E : 78			ld	a,b
FD0F : 3C			inc	a
FD10 : 82			add	a,d
FD11 : 57			ld	d,a
FD12 : D6 18			sub	24
FD14 : 38 01			jr	c,EchoCharPrt4	; jump if D <24

FD16 : 57			ld	d,a		; D = D - 24
FD17			EchoCharPrt4:
FD17 : CD 9F FD			call	CrtSetCurXY
FD1A : CD 99 FD			call	CrtWriteC
FD1D : D1			pop	de

FD1E			CrtCurIncX:
FD1E : 1C			inc	e
FD1F : 7B			ld	a,e
FD20 : D6 50			sub	80
FD22 : 38 2D			jr	c,EchoCharEnd	; jump if E <80

FD24 : 5F			ld	e,a		; E = E - 80
FD25			CrtCurIncY:
FD25 : 14			inc	d
FD26 : 7A			ld	a,d
FD27 : D6 18			sub	24
FD29 : 38 26			jr	c,EchoCharEnd	; jump if D <24

FD2B : 15			dec	d		; D locked to last line 23

FD2C : 04			inc	b
FD2D : 78			ld	a,b
FD2E : D6 18			sub	24
FD30 : 38 01			jr	c,CrtCurWait	; jump if B <24

FD32 : 47			ld	b,a		; B = B-24

FD33			CrtCurWait:
FD33 : 7E			ld	a,(hl)		; HL = FFF9
FD34 : 0F			rrca
FD35 : 30 FC			jr	nc,CrtCurWait	; Wait Crt Sync

FD37 : 2B			dec	hl
FD38 : 70			ld	(hl),b		; HL = FFF8

FD39 : 23			inc	hl		; HL = FFF9
FD3A : 36 46			ld	(hl),46H	; Set Reg 6 = bottom line -1, Start V Sync

				; Clear bottom line 'B'
FD3C : D5			push	de
FD3D : 0E 20			ld	c,' '
FD3F : 50			ld	d,b
FD40 : 1E 4F			ld	e,79

FD42			CrtClBotLine:
FD42 : CD 96 FD			call	CrtSetCurAndWriteC
FD45 : 1D			dec	e
FD46 : F2 42 FD			jp	p,CrtClBotLine

				; Save bottom line number (-1)
FD49 : 11 42 1F			ld	de,1F42H
FD4C : 48			ld	c,b
FD4D : CD 96 FD			call	CrtSetCurAndWriteC
FD50 : D1			pop	de

FD51			EchoCharEnd:
FD51 : 78			ld	a,b		; B = bottom line -1
FD52 : 3C			inc	a
FD53 : 82			add	a,d		; + Cursor Y
FD54 : 47			ld	b,a
FD55 : D6 18			sub	24
FD57 : 38 01			jr	c,EchoCharEnd2	; jump if B <24

FD59 : 47			ld	b,a		; B = B - 24

FD5A			EchoCharEnd2:
				; Save cursor pos X (reg E) in 1F40H
FD5A : 4B			ld	c,e
FD5B : C5			push	bc
FD5C : 42			ld	b,d
FD5D : 11 40 1F			ld	de,1F40H	; X = 31, Y = 64
FD60 : CD 96 FD			call	CrtSetCurAndWriteC

				; Save cursor pos Y (reg E) in 1F41H
FD63 : 48			ld	c,b
FD64 : 13			inc	de		; DE = 1F41H
FD65 : CD 96 FD			call	CrtSetCurAndWriteC

FD68 : D1			pop	de
FD69 : CD 9F FD			call	CrtSetCurXY

FD6C : C1			pop	bc
FD6D : 79			ld	a,c
FD6E : D1			pop	de
FD6F : E1			pop	hl
FD70 : C9			ret
			;
			;--------------------------------------------------------------
			; Test Switch ST26-ST27
			;
			; ST26 ST27	D3-D0	C	Mode
			;  A	A	x11x	0	Num B&W
			;  A	B	x01x	1	Graf. Color
			;  B	A	x10x	0	---
			;  B	B	x00x	0	Graf. B&W
			;--------------------------------------------------------------
FD71			CrtGetConfGrafCol:
FD71 : 3A F9 FF			ld	a,(CrtStatRd)
FD74 : E6 06			and	06H
FD76 : FE 06			cp	06H
FD78 : C8			ret	z	; C = 0 : ST26A-27A
FD79 : 0F			rrca
FD7A : 0F			rrca
FD7B : C9			ret		; ST27B, C = ST26-A
			;
			;--------------------------------------------------------------
			; Test Switch ST26-ST27
			;
			; ST26 ST27	D3-D0	C	Mode
			;  A	A	x11x	0	Num B&W
			;  A	B	x01x	1	Graf. Color
			;  B	A	x10x	0	---
			;  B	B	x00x	1	Graf. B&W
			;--------------------------------------------------------------
FD7C			CrtGetConfGraf:
FD7C : 3A F9 FF			ld	a,(CrtStatRd)
FD7F : E6 06			and	006H
FD81 : FE 06			cp	006H
FD83 : C8			ret	z	; C = 0 : ST26A-27A
FD84 : 2F			cpl
FD85 : 0F			rrca
FD86 : 0F			rrca
FD87 : 0F			rrca		; ST26B, C = ST27B
FD88 : C9			ret

			;
			;--------------------------------------------------------------
			; Set Cursor and Read Char
			;--------------------------------------------------------------
FD89			CrtReadChar:
FD89 : CD AE FD			call	CrtSetCurYX	; HL = FFF9
FD8C : 36 B0			ld	(hl),0B0H	; Start Wait HSync and start Video RAM Read
FD8E			CrtReadChar2:
FD8E : 7E			ld	a,(hl)
FD8F : 0F			rrca
FD90 : 30 FC			jr	nc,CrtReadChar2	; Wait Sync, jump if bit 0 = 0

FD92 : 2B			dec	hl
FD93 : 7E			ld	a,(hl)		; HL = FFF8

FD94 : 23			inc	hl
				; HL = FFF9
FD95 : C9			ret
			;
			;--------------------------------------------------------------
			; DE = Cursor position, Write Char C
			;--------------------------------------------------------------
FD96			CrtSetCurAndWriteC:
FD96 : CD AE FD			call	CrtSetCurYX
FD99			CrtWriteC:
FD99 : 2B			dec	hl
FD9A : 71			ld	(hl),c		; HL = FFF8

FD9B : 23			inc	hl		; HL = FFF9
FD9C : 36 A0			ld	(hl),0A0H	; Start Wait HSync and start Video RAM Write
FD9E : C9			ret
			;
			;--------------------------------------------------------------
			; Set cursor at DE : D = Y, E = X
			;--------------------------------------------------------------
FD9F			CrtSetCurXY:
FD9F : 7E			ld	a,(hl)		; HL = FFF9
FDA0 : 0F			rrca
FDA1 : 30 FC			jr	nc,CrtSetCurXY	; Wait Sync, jump if bit 0 = 0

FDA3 : 2B			dec	hl
FDA4 : 73			ld	(hl),e		; HL = FFF8

FDA5 : 23			inc	hl
FDA6 : 36 0C			ld	(hl),0CH	; HL = FFF9, LOAD CURSOR CHARACTER ADDRES (X)

FDA8 : 2B			dec	hl
FDA9 : 72			ld	(hl),d		; HL = FFF8

FDAA : 23			inc	hl
FDAB : 36 0D			ld	(hl),0DH	; ; HL = FFF9, LOAD CURSOR ROW ADDRESS (Y)
FDAD : C9			ret
			;
			;--------------------------------------------------------------
			; Set cursor at DE : D = Y, E = X
			;--------------------------------------------------------------
FDAE			CrtSetCurYX:
FDAE : 7E			ld	a,(hl)		; HL = FFF9
FDAF : 0F			rrca
FDB0 : 30 FC			jr	nc,CrtSetCurYX	; Wait Sync, jump if bit 0 = 0

FDB2 : 2B			dec	hl
FDB3 : 72			ld	(hl),d		; HL = FFF8

FDB4 : 23			inc	hl
FDB5 : 36 0D			ld	(hl),0DH	; HL = FFF9, LOAD CURSOR ROW ADDRESS (Y)

FDB7 : 2B			dec	hl
FDB8 : 73			ld	(hl),e		; HL = FFF8

FDB9 : 23			inc	hl
FDBA : 36 0C			ld	(hl),0CH	; HL = FFF9, LOAD CURSOR CHARACTER ADDRES (X)
FDBC : C9			ret
			;
			;
			;--------------------------------------------------------------
			; Conversion table ascii to char code in video RAM
			;--------------------------------------------------------------
			TabCharAscii:
FDBD : 7C			DB	7Ch
FDBE : 7E			DB	7Eh
FDBF : 9D			DB	9DH
FDC0 : 9C			DB	9CH
FDC1 : 97			DB	97H
FDC2 : 96			DB	96H
FDC3 : 95			DB	95H
FDC4 : 94			DB	94H
FDC5 : 92			DB	92H
FDC6 : 91			DB	91H
FDC7 : 90			DB	90H
FDC8 : 8E			DB	8EH
FDC9 : 84			DB	84H
FDCA : 99			DB	99H
FDCB : 9A			DB	9AH
FDCC : FF			DB	0FFH

			TabCharAsciiLen	EQU	$-TabCharAscii

			;--------------------------------------------------------------
			; Accents : code 10h - 1Eh
			;--------------------------------------------------------------
			TabCharReplace:
FDCD : 00			DB	0	; for 0FFH
FDCE : 1A			DB	1AH	; for 9AH
FDCF : 19			DB	19H	; for 99H
FDD0 : 10			DB	10H	; for 84H
FDD1 : 11			DB	11H	; for 8EH
FDD2 : 12			DB	12H	; for 90H
FDD3 : 13			DB	13H	; for 91H
FDD4 : 14			DB	14H	; for 92H
FDD5 : 15			DB	15H	; for 94H
FDD6 : 16			DB	16H	; for 95H
FDD7 : 17			DB	17H	; for 96H
FDD8 : 18			DB	18H	; for 97H
FDD9 : 1B			DB	1BH	; for 9CH
FDDA : 1C			DB	1CH	; for 9DH
FDDB : 1D			DB	1DH	; for 7Eh
FDDC : 1E			DB	1EH	; for 7Ch

			;--------------------------------------------------------------
			; End of code (526 bytes nop)
			;--------------------------------------------------------------
FDDD :
				mlist	-1
			;
				rept	0FFEBH - $	; 526 byte to 0 until 0FFEBH
				db	0
				endm


			;--------------------------------------------------------------
FFEB			JumpCode:
FFEB : DB FF			in	a,(0FFH)	; Disable EPROM space
FFED			RamCode1:
FFED : 00			nop			; EPROM disabled, switch to code in RAM
FFEE			RamCode2:
FFEE : 00			nop
FFEF : 00			nop

			;--------------------------------------------------------------
			;
			; FFF0 - FFFF : Space reserved for IO devices
			;
			;--------------------------------------------------------------
			;
				rept	16		; 16 byte to 0 until end of EPROM
				db	0
				endm
			;

			;--------------------------------------------------------------
				END	EpromBase
