;	SWITCH FILE FOR TERMINAL LOGIC FIRMWARE
;						3/21/81
;
RAMSIZE		EQU	0100H	;CPU RAM SIZE
LINELENGTH	EQU	0050H	;CHAR./LINE
NUMBEROFLINES	EQU	24	;NUMBER OF LINES TO DISPLAY
DMEMSIZE	EQU	0800H	;DISPLAY MEMORY SIZE
SCREENSIZE	EQU	LINELENGTH * NUMBEROFLINES
;
INITCONFIG	EQU	87H	;POWER UP OPTIONS FOR 'DIP2SAVE'
				; xxxx xxx1  CURSOR
				;		0=UNDERLINE
				;		1=BLOCK
				; xxxx xx1x  CLICK
				;		0=NO CLICK
				;		1=CLICK
				; xxxx x1xx  END OF LINE
				;		0=DISCARD
				;		1=WRAP
				; xxxx 1xxx  AUTO LINEFEED
				;		0=NO
				;		1=YES
				; xxx1 xxxx  AUTO CRETURN
				;		0=NO
				;		1=YES
				; xx1x xxxx  MODE
				;		0=HEATH
				;		1=ANSI
				; x1xx xxxx  KEYPAD
				;		0=NORMAL
				;		1=SHIFTED
				; 1xxx xxxx  AUTO REPEAT
				;		0=NO
				;		1=YES
;
INITCONFIG2	EQU	08H	;START VALUE FOR 'MODEFLAG2'
				;(ALL OTHER BITS MUST BE 0)
				; xxxx 1xxx  ON-LINE
				;		0=NO
				;		1=YES
				; xx1x xxxx  CAPS LOCK (ALPHA FLIP)
				;		0=NORMAL
				;		1=SHIFTED
				; 1xxx xxxx  25th LINE
				;		0=DISABLED
				;		1=ENABLED
;
KEYDELAY	EQU	24	;INITIAL REPEAT KEY DELAY
REPRATE		EQU	4	;REPEAT KEY RATE
;
PAGE
JAMES		EQU	1	;1 EQUAL TRUE, 0 EQUALS FALSE 
				; JAMES KEYBAORD
MATRIXBREAK	EQU	0	;BREAK KEY	0=SEPERATE
				;		1=MATRIX
MATRIXLOCAL	EQU	1	;ON/OFF LINE	0=SEPERATE
				;		1=MATRIX
MATRIXSHIFTLOCK	EQU	1	;SHIFT LOCK	0=SEPERATE
				;		1=MATRIX
ENABDISKBOARD	EQU	0	;ENABLE/DISABLE
				;KEYBOARD	0= NO.
				;		1= YES.
				;IF THIS IS ENABLED BE SURE TO SET
				;'ENABDISKEY' TO PROPER KEY COORDINATES
				;IN THE TABLES.
				;
SCREENHOLD	EQU	0	;HOLD SCREEN MODE
				;		0= NO.
				;		1= YES.
				;
ANSI		EQU	0	;ANSI MODE	0=NO.
				;  INCLUDED	1=YES.
;
CPUCLOCK	EQU	0	;CPU CLOCK RATE
				;		0=1.536 MHz
				;		1=3.072 MHz
;
TRANSMASK	EQU	10H	;WAIT FOR TRUE TO TRANSMIT
				;  (ANY COMBINATION OK)
				;	00H = NONE
				;	10H = CTS
				;	20H = DSR
				;	80H = RLSD	
;
PAGE
TITLE'	TERMINAL LOGIC FIRMWARE		12/24/81'
;
PAGE 60
	MACLIB	Z80
	ORG	0000H
;
;	I/O PORTS
;
PIO1D	EQU	00H	;00   =	PIO PORT 1. DATA
PIO1C	EQU	01H	;01   =	PIO PORT 1. CONTROL
PIO2D	EQU	02H	;02   =	PIO PORT 2. DATA
PIO2C	EQU	03H	;03   =	PIO PORT 2. CONTROL
DIP1	EQU	20H	;20-3F= DIPSWITCH 1.
;			 40-5F= UART
UARTDL	EQU	40H	;	DATA / DIVISOR (LOW)
UARTDH	EQU	41H	;	INT. ENABLE / DIVISOR (HIGH)
UARTID	EQU	42H	;	INTERUPT IDENT.
UARTLC	EQU	43H	;	LINE CONTROL
UARTMC	EQU	44H	;	MODEM CONTROL
UARTLS	EQU	45H	;	LINE STATUS
UARTMS	EQU	46H	;	MODEM STATUS
;			60-7F= CRTC
CRTCWA	EQU	60H	;	WRITE REG. SELECT
CRTCWD	EQU	61H	;	WRITE REG. DATA
CRTCRD	EQU	63H	;	READ REG. DATA
CRTCIE	EQU	64H	;	INTERUPT ENABLE
CRTCAUX	EQU	68H	;	AUX. CHAR. SET
KEYCON	EQU	80H	;80-9F= KEYBOARD ENCODER
KEYINT	EQU	0A0H	;A0-BF= SPECIAL KEYS & INT. ID.
CLICK	EQU	0C0H	;C0-DF= CLICK
BEEP	EQU	0E0H	;E0-FF= BEEP
;
;	MEMORY LAYOUT
;
;	0 - 3FFFH	;PROM
;   4000H - 7FFFH	;RAM	256 OR 1024 BYTES
;   8000H - BFFFH	;NOT USED
;   C000H - FFFFH	;DISPLAY
;
CR	EQU	0DH
LF	EQU	0AH
FF	EQU	0CH
TAB	EQU	09H
BEL	EQU	07H
BSPACE	EQU	08H
ESC	EQU	1BH
;
;
PAGE
;*************************************************************
;
;	INTERNAL FLAG BITS
;
;*************************************************************
;
;MODEFLAG1 =	(INITIALIZED 00H)
;
;		01-HOLD SCREEN MODE
;		02-GRAPHICS
;		04-ALT. KEYPAD
;		08-AUX. CHAR. SET ENABLE
;		10-DISABLE CURSOR
;		20-BREAK MODE
;		40-INSERT MODE
;		80-REVERSE VIDEO/AUX. CHAR. MODE ENABLE
;
;MODEFLAG2 =	(SET TO 'INITCONFIG2' DEFINED IN SWITCH FILE)
;
;		01-ESC SEQUENCE
;		02- (SPARE)
;		04-KEYBOARD DISABLED
;		08-ON-LINE
;		10-X-OFF HAS BEEN SENT
;		20-CAPS. LOCK (ALPHA FLIP)
;		40- (SPARE)
;		80-25th LINE ENABLED
;
;DIP1SAVE =	(SET BY THE DIPSWITCH)
;
;		01- BAUD RATE
;		02-  "    "
;		04-  "    "
;		08-  "    "
;		10-PARITY ENABLE
;		20-EVEN PARITY
;		40-STICK PARITY
;		80-FULL DUPLEX
;
;DIP2SAVE =	(SET TO 'INITCONFIG' DEFINED IN SWITCH FILE)
;
;		01-BLOCK CURSOR
;		02-NO CLICK
;		04-WRAP AT END OF LINE
;		08-AUTO LF ON CR
;		10-AUTO CR ON LF
;		20-ANSI MODE
;		40-KEYPAD SHIFTED
;		80-AUTO KEY REPEAT
;
PAGE
;*************************************************************
;
;	MAIN ROUTINE STARTS HERE.
;
;*************************************************************
;
	JMP	INIT
MAINLOOP:
	LXI	H,MAINLOOP	;SET RETURN ADDR.
	PUSH	H
	CALL	RBUFEMP		;RECIEVE BUFFER EMPTY?
	JNC	DISPLAY
LB0002:	CALL	KBUFEMP		;KEY INT. BUFFER EMPTY?
	JNC	KEYBOARD
	CALL	UPDATECRTC	;UPDATE CRTC CURSOR
				;	& ENABLE CRTC INT.
	LDA	MODEFLAG2
	ANI	08H		;OFF LINE?
	RZ			;YES.-->
ENTRANSINT:
	DI
	LDA	TBUFCNT
	ORA	A
	JRNZ	SETTINT
	LDA	XOFFSAVE
	ORA	A		;ANYTHING TO TRANSMIT?
	JRZ	LB0004		;NO.-->
SETTINT:
	IN	UARTDH
	BIT	1,A		;TRANSMIT INT. ENABLED?
	JRNZ	LB0004		;YES.-->
	ORI	02H		;ENABLE TRANSMIT INT.
	OUT	UARTDH
LB0004:	EI
	RET
;
;
;*************************************************************
;
;	MASKABLE INTERRUPT VECTOR
;
;*************************************************************
;
	ORG	0038H		;INTERRUPT VECTOR
;
MSKINT:	EXAF
	EXX
	IN	KEYINT
	XRI	0F6H		;TOGGLE BITS
	MOV	C,A
;
IF	NOT MATRIXLOCAL
	LXI	H,MODEFLAG2
	BIT	0,A		;OFF LINE KEY?
	JRZ	SETOFFLINE
	SETB	3,M		;ON-LINE BIT
	JR	CHKBREAK
SETOFFLINE:
	RES	3,M		;RESET FLAG
CHKBREAK:
ENDIF
;
;IF	NOT MATRIXBREAK
;
;	BIT	2,A		;BREAK KEY?
;	CNZ	SETBREAKSEP	;YES.-->
;ENDIF
;
	JR	RECEIVEINT
;
;*************************************************************
;
;	NMI VECTORS HERE
;*************************************************************
;
	ORG	0066H
;
NMIINT:	PUSH	PSW
	PUSH	B
	PUSH	D
	PUSH	H
;
	LDA	MODEFLAG1
	ANI	08H		;GET AUX CHAR. GEN. BIT
	ORI	60H
	MOV	C,A
	MVI	A,6		;# OF ROWS REG.
	OUTP	A
	LDA	CRTCROWS
	INR	C
	OUTP	A
;
	MVI	B,0AH		;CURSOR START SCAN
	LHLD	CURSORTYPE
	DCR	C
	OUTP	B
	INR	C
	OUTP	H
	DCR	C
	INR	B
	OUTP	B		;CURSOR END SCAN
	INR	C
	OUTP	L
;
	LHLD	TOPCRTC
	DCR	C
	INR	B
	OUTP	B		;TOP OF PAGE (HIGH)
	INR	C
	OUTP	H
	DCR	C
	INR	B
	OUTP	B		;TOP OF PAGE (LOW)
	INR	C
	OUTP	L
;
	LHLD	CURSORCRTC
	DCR	C
	INR	B
	OUTP	B		;CURSOR (HIGH)
	INR	C
	OUTP	H
	DCR	C
	INR	B
	OUTP	B		;CURSOR (LOW)
	INR	C
	OUTP	L
;
	LDA	TIMER
	ORA	A		;CHECK REPEAT KEY TIMER
	CNZ	TIMEOUT
;
	POP	H
	POP	D
	POP	B
	POP	PSW
	RETN
;
;
;				;END OF NMI ROUTINE.
;*************************************************************
;	MASKABLE INTERRUPT ROUTINE CONTINUES...
;
RECEIVEINT:
	IN	UARTID
	CPI	04H		;RECEIVE DATA AVAILABLE?
	JRNZ	TBUFEMPINT	;NO.-->
	IN	UARTDL
	ANI	7FH
	JRZ	INTRETURN	;= NUL?   YES.-->
	BIT	3,C		;OFF LINE?
	JRZ	INTRETURN	;YES.-->
	MOV	D,A
	LDA	RBUFCNT
	CPI	70H		;RECEIVE BUFFER ALMOST FULL?
	JRC	NOXOFF
	MVI	A,13H		;DC3 (X-OFF)
	STA	XOFFSAVE	;SETUP TO OUTPUT X-OFF NEXT
	LDA	MODEFLAG2
	ORI	10H		;SET X-OFF FLAG
	STA	MODEFLAG2
NOXOFF:	MOV	A,D
	CALL	RECBUFIN	;PUT BYTE IN RECEIVE BUFFER
	JR	INTRETURN
TBUFEMPINT:
	CPI	02H		;TRANS. BUFF. EMPTY INTERRUPT?
	JRNZ	KBOARDINT	;NO.-->
	BIT	3,C		;OFF-LINE?
	JRZ	NOTRANS		;YES.-->
	IN	UARTLS
	ANI	60H		;TRANSMIT S/R & BUFFER EMPTY?
	CPI	60H
	JRNZ	NOTRANS
	IN	UARTMS
	ANI	TRANSMASK	;OK TO TRANSMIT?
	CPI	TRANSMASK
	JRNZ	NOTRANS		;NO.-->
	LXI	H,XOFFSAVE
	MOV	A,M
	MVI	M,0		;CLEAR THE X-OFF SAVE
	ORA	A		;SPECIAL TRANSMIT SAVE EMPTY?
	JRNZ	TRANSONE	;NO.--> 
	CALL	TBUFGET		;ANYTHING TO TRANSMIT?
	JRC	NOTRANS		;NO.-->
TRANSONE:
	OUT	UARTDL
	JR	INTRETURN
NOTRANS:
	IN	UARTDH
	ANI	0FDH		;DISABLE TRANS. INTERRUPTS
	OUT	UARTDH
	JR	INTRETURN
KBOARDINT:
	MOV	A,C
	ANI	80H		;KEYBOARD INTERRUPT?
	JRZ	INTRETURN	;NO.-->
	IN	KEYCON
	MOV	B,A
;
IF	MATRIXBREAK
	CPI	BREAKKEY	;SHOULD WE BREAK?
	JRZ	SETBREAK	;YES.-->
ENDIF
;
IF	ENABDISKBOARD
	CPI	ENABDISKEY	;PROPER KEY?
	LDA	MODEFLAG2
	JRNZ	LB0018
	XRI	04H		;TOGGLE KEYBOARD DISABLE FLAG
	STA	MODEFLAG2
	CALL	KSAVERET	;CLICK
	JR	INTRETURN
;
LB0018:	ANI	04H		;KEYBOARD DISABLED?
	JRNZ	INTRETURN	;YES.-->
ENDIF
;
LB0019:	CALL	KEYSAVE
;
INTRETURN:
	EXAF
	EXX
	EI
	RET
;
;*************************************************************
;
;		PUT KEY IN KEYBOARD BUFFER
;
KEYSAVE:
	LHLD	KBUFPNT
	MOV	A,L
	CPI	LOW(KBUFPNT)	;KEYBOARD BUFFER FULL?
	JRZ	BEEPRET		;YES.-->
	MOV	M,B
	INX	H
	MOV	M,C
	INX	H
	SHLD	KBUFPNT
SETTIMER:
	SBCD	KSAVE		;SAVE FOR REPEAT KEY
	MVI	A,KEYDELAY	;SET REPEAT KEY TIMER
	STA	TIMER
	JR	KSAVERET
BEEPRET:
	OUT	BEEP
KSAVERET:
	LDA	DIP2SAVE
	ANI	02H		;CLICK ENABLED?
	JRZ	LB0020		;NO.-->
	OUT	CLICK		;CLICK
LB0020:	RET
;*************************************************************
;
TIMEOUT:
	DCR	A
	STA	TIMER
	RNZ			;NOT TIME YET.
	CALL	RESBREAK	;JUST IN CASE BREAK WAS SET
	IN	KEYINT
	BIT	6,A		;KEYDOWN?
	RZ			;NO.-->
	BIT	7,A		;NEW KEY INTERRUPT?
	RZ			;YES.-->
	LBCD	KSAVE		;GET LAST KEY
;
IF	MATRIXBREAK
	MOV	A,B		;GET KEY COORDS.
	CPI	BREAKKEY	;BREAK KEY?
	RZ			;YES.-->
ENDIF
;
	LDA	DIP2SAVE
	BIT	7,A		;REPEAT KEY DISABLED?
	RZ			;YES.-->
	CALL	KEYSAVE
	MVI	A,REPRATE
	STA	TIMER		;RESET TIMER
	RET
;
RESBREAK:
	IN	UARTLC
	ANI	0BFH		;RESET BREAK
	OUT	UARTLC
	RET
;
IF	NOT MATRIXBREAK
SETBREAKSEP:
	IN	KEYCON		;CLEAR BREAK INTERRUPT
	MVI	B,BREAKKEY	;NEEDED FOR TIMEOUT
ENDIF
;
SETBREAK:
	IN	UARTLC
	ORI	40H		;SET BREAK
	OUT	UARTLC
	CALL	SETTIMER
	JR	INTRETURN
;
;	END OF INTERRUPT ROUTINES.
;
PAGE
;*************************************************************
;
;	TERMINAL KEYBOARD ROUTINES	3/19/81
;
;
;	LOOKUP TABLE FORMAT
;
; MAIN TABLE
;	0xxx xxxx	PRINTABLE
;	1xxx xxxx	FUNCTION KEY
;	11xx xxxx	CHECK KEYPAD SHIFTED FLAG
;	1x1x xxxx	CHECK ALT. MODE FLAG
;
; FUNCTION TABLE
;	0xxx xxxx	TYPE 0.(PRINTABLE)
;	100x xxxx	TYPE 1.(ESC ? or ESC O)
;	101x xxxx	TYPE 2.(OUTPUT STRING ONLY)
;	110x xxxx	TYPE 3.(ESC or ESC [)
;	111x xxxx	TYPE 4.(SPECIAL ROUTINE LOOKUP)
;
; FUNCTION DATA TABLE
;	1xxx xxxx	LAST BYTE OF THIS SEQUENCE
;				(MSB is RESET before output)
;
;**** IMPORTANT ****
;	ANY CHANGES TO THE TABLE AREA MUST NOT ALTER 
;		THE LENGTH.  'HOLES' IN THE TABLE MAY
;		BEEN FILLED WITH ROUTINES AND PADDED TO
;		MAINTAIN ALIGNMENT.
;
;*************************************************************
;
KEYBOARD:
	MOV	A,D
	ANI	7FH
;
IF	NOT JAMES
;
	BIT	5,E		;SHIFT?
	JRZ	NOSHIFT		;NO.-->
	ORI	80H
NOSHIFT:
	MOV	C,A
	MVI	B,0
	LXI	H,KTABLE
	DAD	B
	MOV	A,M
	ORA	A
	JZ	BELL		;INVALID KEY.
	BIT	7,A		;PRINTABLE?
	JRZ	OUTPRINT	;YES.-->
	BIT	6,A		;CHECK KEYPAD SHIFT?
	JRZ	PADOK		;NO.-->
	LDA	DIP2SAVE
	ANI	40H		;KEYPAD SHIFTED?
	JRZ	PADOK		;NO.-->
	MOV	A,C
	XRI	80H		;TOGGLE SHIFT BIT
	MOV	C,A
	LXI	H,KTABLE
	DAD	B
PADOK:	MOV	A,M		;GET FUNCTION CODE
	LXI	H,FUNTAB
	BIT	7,C		;SHIFT?
	JRZ	NOTSHIFT	;NO.-->
	LXI	H,FUNSHIFTTAB
NOTSHIFT:
	BIT	5,A		;CHECK ALT. MODE?
	JRZ	GETFUNCT	;NO.-->
	PUSH	B
	MOV	C,A		;SAVE IN C
	LDA	MODEFLAG1
	ANI	04H		;ALT. MODE?
	MOV	A,C		;NEED CODE IN A
	POP	B
	JRZ	GETFUNCT	;NO.-->
	LXI	H,ALTTAB
	BIT	7,C		;SHIFT?
	JRZ	GETFUNCT	;NO.-->
	LXI	H,ALTSHIFTTAB
GETFUNCT:
	ANI	1FH		;MAX. TABLE SIZE
	MOV	C,A
	DAD	B
	MOV	A,M		;GET BYTE FROM TABLE
	ORA	A		;VALID CODE?
	JZ	BELL		;NO.-->
	BIT	7,A		;PRINTABLE?
	JRZ	OUTA		;YES.-->
	ANI	1FH		;MAX. FUNCTION #
	MOV	C,A		;SAVE FUNCTION #
	MOV	A,M
	ANI	60H		;MASK ALL BUT TYPE BITS
	JRZ	TYPE1
	CPI	20H
	JRZ	TYPE2
	CPI	40H
	JRZ	TYPE3
;
TYPE4:	LXI	H,TYPE4TAB	;TABLE OF ROUTINE ADDR.
	DAD	B
	DAD	B		;WORD VALUE
	MOV	A,M
	INX	H
	MOV	H,M
	MOV	L,A
	PCHL			;GO TO IT.
;
;
OUTPRINT:
	MOV	B,A		;SAVE KEY
;
IF	MATRIXSHIFTLOCK
	LDA	MODEFLAG2
	ANI	20H		;CAPS. LOCK FLAG BIT?
ELSE
	BIT	1,E		;SEPERATE SHIFT LOCK KEY?
ENDIF
;
	MOV	A,B
	JRZ	OUTP1		;NO.-->
	CPI	41H
	JRC	OUTP1
	CPI	5BH
	JM	FLIPIT
	CPI	61H
	JRC	OUTP1
	CPI	7BH
	JRNC	OUTP1
FLIPIT:	XRI	20H		;SWAP UPPER & LOWER CASE
OUTP1:	BIT	4,E		;CONTROL KEY?
	JRZ	ASIS		;NO.-->
	ANI	1FH
	RZ			;DO NOT SEND 'NUL' (CTL-SPACE)
ASIS:	JR	OUTIT		;		OR (CTL-@)
;
OUTA:
	ANI	7FH
	BIT	4,E		;CONTROL KEY?
	JNZ	RECBUFIN	;YES.-->  LOCAL ONLY.
OUTIT:	JMP	OUTPUTA		;ON/OFF LINE.
				;FULL/HALF DUPLEX.
;
;
;
TYPE1:	LXI	H,TYPE1TAB
	DAD	B
OUTPUT1:PUSH	H		;THIS OUTPUTS ESC ? OR ESC O
	CALL	ESCOUT
	MVI	A,'?'
	JRZ	TYPE3LOOP	;HEATH
OUT1A:	MVI	A,'O'
	JR	TYPE3LOOP	;ANSI
;
;
TYPE2:	LXI	H,TYPE2TAB	;OUTPUT STRING FROM TABLE
	DAD	B
	DAD	B		;TABLE HAS TWO BYTE ENTRIES
OUTPUT2:PUSH	H
	JR	HEATH3
;
;
TYPE3:	LXI	H,TYPE3TAB	;ESC OR ESC [
	DAD	B
OUTPUT3:PUSH	H
	CALL	ESCOUT
	MVI	A,'['
	JRZ	HEATH3		;HEATH
TYPE3LOOP:			;LOOP UNTIL MSB = 1.
	CALL	OUTA
HEATH3:	POP	H
	MOV	A,M
	BIT	7,A		;LAST ONE?
	JRNZ	OUTA		;YES.-->
	INX	H		;NEXT.
	PUSH	H
	JR	TYPE3LOOP	;CONTINUE
;
ESCOUT:			;OUTPUT ESC & CHECK MODE	
	MVI	A,ESC
	CALL	OUTA
MODETEST:
	LDA	DIP2SAVE
	ANI	20H		;HEATH MODE?
	RET
;
;**************************************
;	TYPE 4 ROUTINES START HERE.
;
INSERTKEY:
	LXI	H,INSERTAB
	MVI	C,0
	LDA	MODEFLAG1
	ANI	40H		;INSERT ON?
	JRZ	TURNON		;NO.-->
	INR	C
ADDTWO:	INR	C
	INR	C
TURNON:	LDA	DIP2SAVE
	ANI	20H		;HEATH?
	JRZ	NOTANSI		;YES.-->
	INR	C
NOTANSI:
	DAD	B
	JR	OUTPUT3		;HL => STRING.
;
SCROLKEY:
	LXI	H,HOLDCNT	;LF COUNTER
	BIT	5,E		;SHIFT?
	JRZ	ONELF		;NO.-->
	MVI	M,17H		;23 LF'S
ONELF:	INR	M
BREAKRET:			;BREAK HANDLED IN INTERRUPT
				;ROUTINE, BUT VALID FUNCTION
				;NEEDED TO AVOID BEEP.
	RET
;
ERASEKEY:
	LXI	H,ERATAB
	MOV	C,B		;ZERO INTO C
	BIT	5,E		;SHIFT?
	JRZ	TURNON		;NO.-->
	JR	ADDTWO
;
DELCHARKEY:
	LXI	H,DELTAB
	MOV	C,B		;ZERO INTO C
	JR	TURNON
;
;
OFFLINE:			;ON/OFF LINE,ON/OFF CAPS. LOCK
	LXI	H,MODEFLAG2	;	DUAL FUNCTION
	BIT	5,E		;SHIFT?
	JRZ	LOC1		;NO.-->
	BIT	4,E		;CONTROL?
	JRZ	LOC2		;NO.-->
	SETB	5,M		;SET CAPS. LOCK FLAG
	RET
;
LOC2:	SETB	3,M		;ON-LINE
	RET
;
LOC1:	BIT	4,E		;CONTROL?
	JRZ	LOC3		;NO.-->
	RES	5,M		;RESET CAPS. LOCK FLAG
	RET
;
LOC3:	IN	KEYINT		;CHECK 'OFF-LINE' INPUT
	ANI	01H		;(THIS ALLOWS THE USE OF AN EXTERNAL
	RZ			;SWITCH TO LOCK IN ON-LINE MODE.)
	RES	3,M		;OFF-LINE
	RET
;
;				END OF KEYBOARD ROUTINES
ENDIF				; END OF JAMES NOT IF
IF JAMES
	JMP	OUTPUTA		; SEND CHARACTER TO OUTPUT BUFFER
ENDIF	
;
;*************************************************************
;
DISPLAY:
			;PROCESS KEYS FROM RECEIVE BUFFER
;
	MOV	D,A
	LDA	MODEFLAG2
	BIT	4,A		;DID WE SEND X-OFF?
	JRZ	LB0062		;NO.-->
	MOV	E,A
	LDA	RBUFCNT
	CPI	60H		;TIME TO SEND X-ON?
	MOV	A,E
	JRNC	LB0062		;NO.-->
	ANI	0EFH
	STA	MODEFLAG2	;RESET X-OFF FLAG
	MOV	E,A
	CALL	OUTPUTXON	;DC1 (X-ON)
	MOV	A,E
LB0062:	BIT	0,A		;WAS PREVIOUS BYTE ESC?
	MOV	A,D
	JRZ	LB0065		;NO.-->
	LDA	MODEFLAG2
	ANI	0FEH
	STA	MODEFLAG2	;RESET ESC FLAG
;
IF	ANSI
	CALL	MODETEST	;HEATH MODE?
	JRZ	LB0063		;YES.-->
;				ANSI LOOKUP
	MOV	A,D
	LXI	H,ANSITAB
	MVI	D,04H		;# OF ENTRYS
	MVI	E,03H		;TABLE INCREMENT
	JR	LB0064
ENDIF
;
;				HEATH LOOKUP
LB0063:	MOV	A,D
	LXI	H,HEATHTAB
	MVI	D,(HEATHEND-HEATHTAB)/3	;# OF ENTRYS
	MVI	E,03H		;TABLE INCREMENT
LB0064:	CALL	LOOKUP
	JRNC	LB0073
	RET			;NOT FOUND.-->
;
OUTPUTXON:
	MVI	A,11H		;DC1 (X-ON)
	JMP	OUTPUTA
;				COMMON DISPLAY
LB0065:
;
IF	SCREENHOLD
	CPI	LF		;LINEFEED?
	JRNZ	LB0071		;NO.-->
	LDA	MODEFLAG1
	ANI	01H		;HOLD SCREEN MODE?
	JRZ	HOLD1		;NO.-->
	LDA	ROW
	CPI	17H		;LAST LINE?
	JRNZ	HOLD1		;NO.-->
	LXI	H,HOLDCNT
	DCR	M		;DISPLAY NEXT LINE?
	JRNZ	HOLD1		;YES.-->
	MVI	A,13H		;DC3 (X-OFF)
	CALL	OUTPUTA
	CALL	ENTRANSINT
LB0066:	CALL	KBUFEMP		;ANY KEYS?
	JRNC	LB0068		;YES.-->
	CALL	UPDATECRTC
LB0067:	LDA	RBUFCNT
	CPI	80H		;RECEIVE BUFFER FULL?
	JRNZ	LB0066		;NO.-->
	MVI	A,01H
	STA	HOLDCNT		;DISPLAY A LINE
HOLD1:	JMP	LINEFEED
;
LB0068:	MOV	A,D
	ANI	7FH
	CPI	SCROLLKEY
	JRZ	LB0069		;YES.-->
	CALL	KEYBOARD
	CALL	ENTRANSINT
	JR	LB0067
;
;			HOLD SCREEN MODE COMES HERE
;
LB0069:	BIT	5,E		;SHIFT?
	MVI	A,01H
	JRZ	LB0070		;NO.-->
	MVI	A,18H
LB0070:	STA	HOLDCNT
	CALL	LINEFEED
;
ENDIF
;
;
LB0071:	CPI	7FH
	JRZ	LB0072
	CPI	20H
	JRNC	LB0074
LB0072:	LXI	H,COMMONTAB	;LOOKUP STANDARD CONTROL CODES
	MVI	D,07H		;# OF ENTRYS
	MVI	E,03H		;TABLE INCREMENT
	CALL	LOOKUP
	RC			;NOT FOUND.-->
LB0073:	MOV	E,A
	INX	H
	MOV	D,M
	XCHG
	LXI	B,DIP2SAVE
	LXI	D,MODEFLAG1
	PCHL
;
LB0074:	LDA	MODEFLAG1
	MOV	B,A		;SAVE IN B
	ANI	02H		;GRAPHICS?
	MOV	A,D
	JRZ	LB0076		;NO.-->
	CPI	5EH		;SPECIAL GRAPHIC CHAR.
	JC	LB0076
	JRNZ	LB0075
	MVI	A,7FH
	JR	LB0076
;
LB0075:	ANI	1FH
LB0076:	MOV	D,A
	MOV	A,B		;GET MODEFLAG1
	ANI	80H		;REVERSE VIDEO?
	ORA	D
	PUSH	PSW
	MOV	A,B		;GET MODEFLAG1
	ANI	40H		;INSERT MODE?
	CNZ	CINSERTSCROLL	;LINE SCROLL
	POP	PSW
	LHLD	CURSORABS
	MOV	M,A
	INX	H
	MOV	A,H
	ORI	0F8H
	MOV	H,A
	LDA	COLUMN
	CPI	4FH
	JRNZ	LB0077
	LDA	DIP2SAVE
	ANI	04H		;WRAP?
	RZ			;NO.-->
	CALL	DOLFEED
	JMP	DOCRETURN
;
LB0077:	INR	A
	STA	COLUMN
	SHLD	CURSORABS
	RET
;
;**************************************
;
;	COMMON ROUTINES START HERE.
;
BELL:	OUT	BEEP
	RET
;
BACKSPACE:
	LDA	COLUMN
	ORA	A		;BEGINNING OF LINE?
	RZ			;YES.-->
	DCR	A
	STA	COLUMN
	LHLD	CURSORABS
	DCX	H
;
LAB2:	MOV	A,H
	ORI	0F8H
	MOV	H,A
	SHLD	CURSORABS	;UPDATE CURSOR ABS.
	RET
;
HORIZTAB:
	LDA	COLUMN		;HORIZ TAB (09H)
	ADI	08H
	ANI	0F8H
	CPI	50H
	JRNZ	LB0216
	LDA	COLUMN
	CPI	4FH
	JRZ	LB0215
	INR	A
	JR	LB0216
LB0215:	MVI	A,4FH
	JR	LB0216
LB0216:	STA	COLUMN
	MOV	E,A
	MVI	D,00H
	LHLD	BEGLINABS
LAB5:	DAD	D
	JR	LAB2
;
LINEFEED:
	CALL	DOLFEED
	LDA	DIP2SAVE
	ANI	10H		;AUTO CR?
	JNZ	DOCRETURN	;YES.-->
	RET
;
CRETURN:
	LDA	DIP2SAVE
	ANI	08H		;AUTO LF?
	CNZ	DOLFEED		;YES.-->
DOCRETURN:
	MVI	A,00H
	STA	COLUMN
	LHLD	BEGLINABS
	SHLD	CURSORABS
	RET
;
SETESCFLAG:
	LDA	MODEFLAG2	;SET ESC FLAG
	ORI	01H
	STA	MODEFLAG2
	RET
;
;*******************************
;
;	HEATH ROUTINES START HERE.
;
IF	ANSI
ENTERANSI:			;ENTER ANSI MODE.
	LDAX	B
	ORI	20H
	STAX	B
	RET
;
ANSITRANS25:
	MOV	A,B		;TRANSMIT 25TH LINE
	ORA	A		;ANSI ESC [ Q
	RNZ
ENDIF
;
TRANS25TH:			;TRANSMIT 25TH LINE
	LDA	MODEFLAG2
	ANI	80H
	JRZ	LB0227
	LHLD	TOPABS
	LXI	D,SCREENSIZE
	DAD	D
	MOV	A,H
	ORI	0F8H
	MOV	H,A
	XCHG
	MVI	B,00H
	CALL	LINETOCOMM
LB0227:	MVI	A,CR		;OUTPUT A CR
	CALL	BYTETOCOMM
	OUT	BEEP
	RET
;
IF	ANSI
ANSITRANSPAGE:
	MOV	A,B		;TRANSMIT PAGE
	ORA	A		;ANSI ESC [ P
	RNZ
ENDIF
;
TRANSPAGE:			;HEATH TRANSMIT PAGE
	LHLD	TOPABS
	XCHG
	MVI	B,00H
	MVI	H,18H
LB0243:	PUSH	H
	CALL	LINETOCOMM
	POP	H
	DCR	H
	JRNZ	LB0243
	JR	LB0227		;OUTPUT CR THEN BEEP
;
INSERTCHARON:			;ENTER INSERT CHARACTER MODE
	XCHG
	SETB	6,M
	RET
;
CURSUP:	LDA	ROW
	ORA	A		;TOP LINE?
	RZ			;YES.-->
REVLFEED:
	LDA	ROW
	CPI	18H		;25TH LINE?
	RZ			;YES.-->
	LXI	D,-LINELENGTH
	CALL	LAB3
	LDA	ROW
	ORA	A
	JRNZ	LB0125
	LDA	MODEFLAG2
	ANI	80H		;25TH LINE ENABLED?
	JRZ	LB0124		;NO.-->
;
	LHLD	TOPABS			;MOVE 25TH LINE DATA TO NEW
	LXI	D,SCREENSIZE-LINELENGTH		;LOCATION.
	DAD	D
	XCHG
	LXI	H,LINELENGTH
	DAD	D
	CALL	SCROLL80
;
LB0124:	LHLD	TOPABS		;BACKUP ONE LINE
	LXI	D,-LINELENGTH
	JR	LAB4
;
LB0125:	DCR	A
	STA	ROW
	JR	LB0117
;
LAB3:	LHLD	BEGLINABS
	DAD	D
	MOV	A,H
	ORI	0F8H		;DISPLAY MEM. ABS.
	MOV	H,A
	SHLD	BEGLINABS
	RET
;
CURSDOWN:			;CURSOR DOWN
	LDCURSADDR:
	CALL	GETRECBYTE	;DIRECT CURSOR ADDR
	CPI	18H
	RZ
	CPI	20H
	JRC	LB0169
	CPI	38H
	JRC	LB0168
	JRNZ	LB0169
	MOVLL	LAB3
	LDA	ROW
	CPI	17H		;LAST LINE?
	JRNZ	LB0116		;NO.-->
	LDA	MODEFLAG2
	ANI	80H		;25TH LINE ENABLED?
	JRZ	LB0113		;NO.-->
;
	PUSH	H		;MOVE 25TH LINE DATA TO NEW
	LXI	D,LINELENGTH	;	LOCATION.
	DAD	D
	XCHG
	POP	H
	PUSH	H
	CALL	SCROLL80
	POP	H
;
LB0113:	CALL	SPACE80		;CLEAR ONE LINE AHEAD
	LHLD	TOPABS
	LXI	D,LINELENGTH
;
LAB4:	DAD	D
	MOV	A,H
	ORI	0F8H
	MOV	H,A
	SHLD	TOPABS		;UPDATE TOP OF PAGE ABS.
	MOV	A,H
	ANI	07H
	MOV	H,A
	SHLD	TOPCRTC		;UPDATE TOP OF PAGE CRTC.
	JR	LB0117
;
LB0116:	INR	A
	STA	ROW
LB0117:	LHLD	CURSORABS
	CALL	LAB5		;UPDATE CURSOR ABS.
	JMP	ENABSYNCINT
;
CURSFORWARD:			;CURSOR FORWARD
	LDA	COLUMN
	CPI	LINELENGTH-1	;END OF LINE?
	RZ			;YES.-->
	INR	A
	STA	COLUMN
	LHLD	CURSORABS
	INX	H
	JMP	LAB2
;
IF	ANSI
ANSIERASEDIS:
	MOV	A,B		;ERASE IN DISPLAY
	ORA	A		;ANSI ESC [ J
	JRZ	ERASEEOP
	XCHG
	MOV	A,M
	ORA	A
	JRZ	ERASEEOP
	DCR	A
	JZ	ERASEBEGDIS
	DCR	A
	RNZ
ENDIF
;
CLEARHOME:			;ERASE SCREEN
	LDA	ROW
	CPI	18H		;25TH LINE?
	JZ	ERASELINE	;YES.-->
	CALL	CURSHOME
	MVI	B,LOW(SCREENSIZE/16)
	JMP	SPACE16
;
SETGRAPHICS:
	XCHG			;ENTER GRAPHICS MODE
	SETB	1,M
	RET
;
RESETGRAPHICS:
	XCHG			;EXIT GRAPHICS MODE
	RES	1,M
	RET
;
CURSHOME:			;HOME CURSOR
	LXI	H,0
	SHLD	COLUMN
	LHLD	TOPABS
	SHLD	CURSORABS
	SHLD	BEGLINABS
	RET
;
ERASEEOP:			;ERASE TO END OF PAGE
	LDA	ROW
	CPI	18H		;25TH LINE?
	JRZ	ERASEEOL	;YES.-->
	LHLD	TOPABS
	LXI	D,SCREENSIZE-1
	DAD	D
	STC
	CMC
	LDED	CURSORABS
	DSBC	D
	INX	H
	XCHG
	MOV	A,D
	ANI	07H
	MOV	D,A
	LHLD	CURSORABS
	JMP	SPACEDE
;
ERASEEOL:			;ERASE TO END OF LINE
	LHLD	CURSORABS
	LDA	COLUMN
	NEG
	ADI	50H
	MOV	E,A
	MVI	D,00H
	JMP	SPACEDE
;
INSERTLINE:			;INSERT A LINE
	LDA	ROW
	CPI	18H		;25th LINE?
	JZ	ERASELINE	;YES.-->
	NEG
	ADI	17H
	JRZ	LB0189
	MOV	C,A
	ADD	C
	ADD	C
	ADD	C
	ADD	C
	MOV	C,A
	LDED	TOPABS
	LXI	H,SCREENSIZE-1
	DAD	D
	PUSH	H
	LXI	H,SCREENSIZE-LINELENGTH-1
	DAD	D
	POP	D
LB0188:	PUSH	B
	CALL	LAB6
	LDDR
	POP	B
	DCR	C
	JRNZ	LB0188
	MOV	A,D
	ORI	0F8H
	MOV	D,A
	MOV	A,H
	ORI	0F8H
	MOV	H,A
	LDD
LB0189:	CALL	DOCRETURN
	JMP	SPACE80
;
LAB6:	MOV	A,D
	ORI	0F8H
	MOV	D,A
	MOV	A,H
	ORI	0F8H
	MOV	H,A
	MVI	C,10H
	MVI	B,00H
	RET
;
DELLINE:			;DELETE A LINE
	CALL	DOCRETURN
	XCHG
	LXI	H,LINELENGTH
	DAD	D
	LDA	ROW
	CPI	18H		;25th LINE?
	JZ	ERASELINE	;YES.-->
	NEG
	ADI	17H
	JRZ	LB0184
	MOV	C,A
	ADD	C
	ADD	C
	ADD	C
	ADD	C
	MOV	C,A
LB0183:	PUSH	B
	CALL	LAB6
	LDIR
	POP	B
	DCR	C
	JRNZ	LB0183
LB0184:	XCHG
	JMP	SPACE80
;
DELCHAR:			;DELETE A CHARACTER
	LDA	COLUMN
	NEG
	ADI	4FH
	MOV	C,A
	MVI	B,00H
	LDED	CURSORABS
	JRZ	LB0181
	LHLD	CURSORABS
	INX	H
LB0180:	MOV	A,D
	ORI	0F8H
	MOV	D,A
	MOV	A,H
	ORI	0F8H
	MOV	H,A
	LDI
	MOV	A,B
	ORA	C
	JRNZ	LB0180
LB0181:	MVI	A,20H
	STAX	D
	RET
;
INSERTCHAROFF:			;EXIT INSERT CHARACTER MODE
	XCHG
	RES	6,M
	RET
;
IDENTVT52:			;HEATH IDENTIFY AS VT52
	CALL	OUTESC
	MVI	A,'/'
	CALL	OUTPUTA
	MVI	A,'K'
	JMP	OUTPUTA
;
CURSADDR:
	CALL	GETRECBYTE	;DIRECT CURSOR ADDR
	LHLD	COLUMN		;GET PRESENT POSITION
	CPI	18H
	RZ
	CPI	20H
	JRC	LB0169
	CPI	38H
	JRC	LB0168
	JRNZ	LB0169
	MOV	B,A
	LDA	MODEFLAG2
	ANI	80H		;25TH LINE ENABLED?
	JRZ	LB0169		;NO.-->
	MOV	A,B
LB0168:	SUI	20H
	MOV	H,A		;NEW ROW TO H
LB0169:	CALL	GETRECBYTE
	CPI	18H
	JRZ	LB0172
	CPI	20H
	JRC	LB0170
	CPI	70H
	JRC	LB0171
LB0170:	MVI	A,6FH
LB0171:	SUI	20H
	MOV	L,A		;NEW COLUMN TO L
LB0172:	SHLD	COLUMN		;SET NEW ROW & COLUMN
	JMP	CURSTOSAVED
;
ERASEBEGDIS:			;ERASE FROM BEGINNING OF PAGE
	LDA	ROW
	CPI	18H		;25TH LINE?
	JRZ	ERASEBOL	;YES.-->
	ORA	A		;TOP LINE?
	JRZ	ERASEBOL	;YES.-->
	MOV	L,A
	MOV	E,A
	MVI	H,00H
	MOV	D,A
	DAD	D
	DAD	D
	DAD	D
	DAD	D
	MOV	B,L
	LHLD	TOPABS
	CALL	SPACE16
ERASEBOL:			;ERASE FROM BEGINNING OF LINE
	LHLD	BEGLINABS
	LDA	COLUMN
	INR	A
	MOV	B,A
	MVI	A,20H
LB0135:	MOV	M,A
	INX	H
	DJNZ	LB0135
	RET
;
SETAUXCHAR:			;ENABLE AUX CHAR. GENERATOR
	XCHG
	SETB	3,M
	RET
;
RESETAUXCHAR:			;DISABLE AUX CHAR. GENERATOR
	XCHG
	MOV	A,M
	RES	3,M
	RET
;
IF	ANSI
ANSISAVECURS:
	MOV	A,B		;SAVE CURSOR POSITION
	ORA	A		;ANSI ESC [ S
	RNZ
ENDIF
;
SAVECURS:
	LHLD	COLUMN		;HEATH SAVE CURSOR POSITION
	SHLD	CURSORSAVE
	RET
;
IF	ANSI
ANSICURSTOSAVE:
	MOV	A,B		;SET CURSOR TO SAVED POSITION
	ORA	A		;ANSI ESC [ U
	RNZ
ENDIF
;
SETCURS:
	LHLD	CURSORSAVE	;HEATH SET CURSOR TO SAVED
	SHLD	COLUMN
CURSTOSAVED:
	LDA	ROW
	CALL	ATIMES80
	XCHG
	LHLD	TOPABS
	DAD	D
	MOV	A,H
	ORI	0F8H
	MOV	H,A
	SHLD	BEGLINABS
	LDA	COLUMN
	MOV	E,A
	MVI	D,00H
	JMP	LAB5
;
IF	ANSI
ANSIERASELINE:			;ERASE IN LINE
	MOV	A,B		;ANSI ESC [ K
	ORA	A
	JZ	ERASEEOL
	XCHG
	MOV	A,M
	ORA	A
	JZ	ERASEEOL
	DCR	A
	JRZ	ERASEBOL
	DCR	A
	RNZ
ENDIF
;
ERASELINE:			;ERASE CURSOR LINE
	LHLD	BEGLINABS
	JMP	SPACE80
;
CURSPOSREPORT:			;CURSOR POSITION REPORT
	CALL	OUTESC
	MVI	A,'I'
	CALL	OUTPUTA
	LDA	ROW
	CALL	ADIOUT
	LDA	COLUMN
ADIOUT:	ADI	20H		;MAKE PRINTABLE
	JMP	OUTPUTA
;
SETREVVIDEO:
	XCHG			;ENTER REVERSE VIDEO MODE
	SETB	7,M
	RET
;
RESETREVVIDEO:
	XCHG			;EXIT REVERSE VIDEO MODE
	RES	7,M
	RET
;
MODBAUDRATE:
	CALL	GETRECBYTE	;MODIFY BAUD RATE
	CPI	41H
	RC
	CPI	50H
	RNC
	ANI	0FH
BAUDMOD1:
	MOV	B,A
	LDA	DIP1SAVE
	ANI	0F0H
	ORA	B
	STA	DIP1SAVE
SETBAUD:
	MVI	A,80H
	OUT	UARTLC
	LDA	DIP1SAVE
	MOV	C,A
	ANI	0FH
	JRZ	LB0157
	DCR	A
LB0157:	MOV	B,A
	RLC
	LXI	H,BAUDTAB	;BAUD RATE DIVISOR TABLE
	ADD	L
	MOV	L,A
	MOV	A,M
	OUT	UARTDL
	INX	H
	MOV	A,M
	OUT	UARTDH
	XRA	A
	ORA	B
	MVI	B,00H
	JRNZ	LB0158
	MVI	B,04H
LB0158:	MOV	A,C
	ANI	70H
	RRC
	ORA	B
	MOV	B,A
	ANI	08H
	MVI	A,02H
	JRNZ	LB0159
	MVI	A,03H
LB0159:	ORA	B
	OUT	UARTLC
	MVI	A,01H
	OUT	UARTDH
	MVI	A,03H
	OUT	UARTMC
	RET
;
SETLINEWRAP:			;WRAP AT END OF LINE
	LDAX	B
	ORI	04H
	STAX	B
	RET
;
SETNOWRAP:			;DISCARD AT END OF LINE
	LDAX	B
	ANI	0FBH
	STAX	B
	RET
;
RESETMODE:
	LXI	H,RESETTAB	;HEATH RESET MODE
	JR	LB0203
;
SETMODE:
	LXI	H,SETTAB	;HEATH SET MODE
LB0203:	CALL	GETRECBYTE
	CPI	18H
	RZ
	CPI	30H
	JC	DISPLAY
	CPI	3BH
	JNC	DISPLAY
	ANI	0FH
DOMODE:	RLC
	ADD	L
	MOV	L,A
	MOV	A,M
	INX	H
	MOV	H,M
	MOV	L,A
	LXI	B,DIP2SAVE
	LXI	D,MODEFLAG1
	PCHL
;
IF	ANSI
ANSIREINIT:			;RESET TO POWER UP CONFIG.
	MOV	A,B		;ANSI ESC [ Z
	ORA	A
	RNZ
ENDIF
;
REINIT:	DI			;RESET TO POWER UP CONFIG.
INIT:	LXI	H,RECBUF
	LXI	D,RECBUF+1
	LXI	B,RAMSIZE	;CPU RAM SIZE TO CLEAR
	MVI	M,00H
	LDIR
	LXI	H,KEYBUF
	SHLD	KBUFPNT
	MVI	A,18H
	STA	CRTCROWS
	IN	DIP1
	ORI	01H		;SET LSB
	STA	DIP1SAVE	;UART CONFIGURATION
	MVI	A,INITCONFIG	;INITIAL OPTIONS
	STA	DIP2SAVE
	MVI	A,INITCONFIG2	;SET INITIAL MODE
	STA	MODEFLAG2
	LXI	SP,STACK
	LXI	H,-DMEMSIZE	;DISPLAY MEM. START
	SHLD	TOPABS
	SHLD	BEGLINABS
	SHLD	CURSORABS
	LXI	D,DMEMSIZE
	CALL	SPACEDE
	CALL	INITCRTC
	CALL	SETBAUD
	CALL	CURSON
	IM1			;SET INTERRUPT MODE 1.
	MVI	A,00H
	OUT	CRTCIE		;ENABLE SYNC INTERRUPT
	EI
	JMP	MAINLOOP
;
;
IF	ENABDISKBOARD
ENABKBOARD:			;ENABLE KEYBOARD
	LDA	MODEFLAG2
	ANI	0FBH
	STA	MODEFLAG2
	RET
DISABKBOARD:			;DISABLE KEYBOARD
	LDA	MODEFLAG2
	ORI	04H
	STA	MODEFLAG2
	RET
;
ENDIF
;
;**************************************
;	SET/RESET MODE ROUTINES
;
SETFULLDUP:			;SET FULL DUPLEX
	LDA	DIP1SAVE
	ORI	80H
SETDUP:	STA	DIP1SAVE
	RET
;
RESETFULLDUP:			;SET HALF DUPLEX
	LDA	DIP1SAVE
	ANI	7FH
	JR	SETDUP
;
ENAB25TH:
	LDA	MODEFLAG2	;ENABLE 25TH LINE
	MOV	B,A
	ANI	80H		;ALREADY ENABLED?
	RNZ			;YES.-->
	MVI	A,80H
	ORA	B		;SET FLAG
	STA	MODEFLAG2
	LHLD	TOPABS
	LXI	D,SCREENSIZE
	DAD	D
	CALL	SPACE80		;CLEAR 25th LINE
	MVI	A,19H		;UPDATE # OF ROWS
SETROWS:
	STA	CRTCROWS
	RET
;
DISAB25TH:
	LDA	MODEFLAG2	;DISABLE 25TH LINE
	ANI	7FH
	STA	MODEFLAG2
	MVI	A,18H
	JR	SETROWS
;
DISABCLICK:
	LDAX	B		;DISABLE CLICK
	ANI	0FDH
	STAX	B
	RET
;
ENABCLICK:
	LDAX	B		;ENABLE CLICK
	ORI	02H
	STAX	B
	RET
;
SETHOLDSCREEN:
	XCHG			;ENTER HOLD SCREEN MODE
	SETB	0,M
	MVI	A,01H
	STA	HOLDCNT
	RET
;
RESETHOLDSCREEN:
	XCHG			;EXIT HOLD SCREEN MODE
	RES	0,M
	RET
;
SETBLOCKCURS:
	LDAX	B		;SET BLOCK CURSOR
	ORI	01H
	STAX	B
	LDAX	D
	ANI	10H
	RNZ
BLOCKCURS:
	MVI	H,60H
	MVI	L,09H
BLOC1:	SHLD	CURSORTYPE
	RET
;
SETULINECURS:
	LDAX	B		;SETUP UNDERLINE CURSOR
	ANI	0FEH
	STAX	B
	LDAX	D
	ANI	10H
	RNZ
ULINECURS:
	MVI	H,49H
	MVI	L,09H
	JR	BLOC1
;
DISABCURS:
	LDA	MODEFLAG1	;DISABLE CURSOR
	ORI	10H
	STA	MODEFLAG1
	MVI	H,20H
	MVI	L,00H
	SHLD	CURSORTYPE
	RET
;
CURSON:	LDA	MODEFLAG1
	ANI	0EFH
	STA	MODEFLAG1
	LDA	DIP2SAVE
	ANI	01H		;BLOCK CURSOR?
	JNZ	BLOCKCURS	;YES.-->
	JMP	ULINECURS
;
SETPADSHIFT:
	LDAX	B		;ENTER KEYPAD SHIFTED MODE
	ORI	40H
	STAX	B
	RET
;
RESETPADSHIFT:
	LDAX	B		;EXIT KEYPAD SHIFTED MODE
	ANI	0BFH
	STAX	B
	RET
;
SETALTPAD:
	LDAX	D		;ENTER ALT. KEYPAD MODE
	ORI	04H
	STAX	D
	RET
;
RESETALTPAD:
	LDAX	D		;EXIT ALT. KEYPAD MODE
	ANI	0FBH
	STAX	D
	RET
;
AUTOLFON:
	LDAX	B		;AUTO LF ON CR
	ORI	08H
	STAX	B
	RET
;
AUTOLFOFF:
	LDAX	B		;RESET AUTO LF ON CR
	ANI	0F7H
	STAX	B
	RET
;
AUTOCRON:			;AUTO CR ON LF
	LDAX	B
	ORI	10H
	STAX	B
	RET
;
AUTOCROFF:			;NO AUTO CR
	LDAX	B
	ANI	0EFH
	STAX	B
	RET
;
AUTOREPEAT:
	LDAX	B		;SET REPEAT KEY FLAG
	ORI	80H
	STAX	B
	RET
;
NOAUTOREPEAT:
	LDAX	B		;RESET REPEAT KEY FLAG
	ANI	7FH
	STAX	B
	RET
;
;**************************************
IF	ANSI
;
;		ANSI ROUTINES START HERE.
;
ANSIBRACKET:
	CALL	ANSIPARAMEVAL	;ANSI "ESC [" LOOKUP
	PUSH	B
	MVI	D,17H
	MVI	E,03H
	LXI	H,ANSITAB2
	CALL	LOOKUP
	POP	B
	RC
	INX	H
	MOV	H,M
	MOV	L,A
	LXI	D,ANSISAVE
	PCHL
;
SETRESETMODE:
	MOV	A,B		;ANSI ESC [ >
	ORA	A
	RNZ
	CALL	ANSIPARAMEVAL
	CPI	'H'
	JRZ	ANSISET
	CPI	'L'
	JRZ	ANSIRESET
	RET
;
ANSISET:
	MOV	A,B
	ORA	A
	RZ
LB0087:	LXI	H,SETTAB	;SET MODE TABLE
	LDAX	D
	CPI	0BH		;CHECK RANGE
	RNC
	PUSH	D
	CALL	DOMODE
	POP	D
	INX	D
	JR	LB0087
;
ANSIRESET:
	MOV	A,B
	ORA	A
	RZ
LB0089:	LXI	H,RESETTAB	;RESET MODE TABLE
	LDAX	D
	CPI	0BH		;CHECK RANGE
	RNC
	PUSH	D
	CALL	DOMODE
	POP	D
	INX	D
	JR	LB0089
;
CHANGEMODE:
	MOV	A,B		;ANSI ESC [ ?
	ORA	A
	RNZ
	CALL	ANSIPARAMEVAL
	CPI	'H'
	JRZ	LB0082
	CPI	'L'
	JRZ	LB0084
	RET
;
LB0082:	MOV	A,B
	ORA	A
	RZ
	XCHG
	LXI	B,DIP2SAVE
LB0083:	MOV	A,M
	CPI	02H
	CZ	SETHEATHMODE
	MOV	A,M
	CPI	07H
	CZ	SETLINEWRAP
	MOV	A,M
	CPI	'H'
	RZ
	INX	H
	JR	LB0083
;
LB0084:	MOV	A,B
	ORA	A
	RZ
	XCHG
	LXI	B,DIP2SAVE
LB0085:	MOV	A,M
	CPI	07H
	CZ	SETNOWRAP
	MOV	A,M
	CPI	'L'
	RZ
	INX	H
	JR	LB0085
;
ANSICURSUP:
	MOV	A,B		;CURSOR UP
	ORA	A		;ANSI ESC [ A
	JZ	CURSUP
	LDAX	D
	ORA	A
	JZ	CURSUP
LB0095:	PUSH	PSW
	CALL	CURSUP
	POP	PSW
	DCR	A
	JRNZ	LB0095
	RET
;
ANSICURSDOWN:
	MOV	A,B		;CURSOR DOWN
	ORA	A		;ANSI ESC [ B
	JZ	CURSDOWN
	LDAX	D
	ORA	A
	JZ	CURSDOWN
LB0090:	PUSH	PSW
	CALL	CURSDOWN
	POP	PSW
	DCR	A
	JRNZ	LB0090
	RET
;
ANSICURSFOR:
	MOV	A,B		;CURSOR FORWARD
	ORA	A		;ANSI ESC [ C
	JZ	CURSFORWARD
	MOV	B,A
LB0094:	CALL	CURSFORWARD
	DCR	B
	JRNZ	LB0094
	RET
;
ANSICURSBACK:
	MOV	A,B		;CURSOR BACKWARDS
	ORA	A		;ANSI ESC [ D
	JZ	BACKSPACE
	LDAX	D
	ORA	A
	JZ	BACKSPACE
	MOV	B,A
LB0091:	CALL	BACKSPACE
	DCR	B
	JRNZ	LB0091
	RET
;
ANSICURSADDR:
	MOV	A,B		;SET CURSOR ADDRESS
	ORA	A		;ANSI ESC [ H OR F
	JZ	CURSHOME
	XCHG
	MOV	A,M
	ORA	A
	JRZ	LB0096
	DCR	A
LB0096:	CPI	18H
	JRC	LB0097
	JRNZ	LB0098
	MOV	B,A
	LDA	MODEFLAG2
	ANI	80H
	JRZ	LB0098
	MOV	A,B
LB0097:	STA	ROW
LB0098:	INX	H
	MOV	A,M
	ORA	A
	JRZ	LB0099
	DCR	A
LB0099:	CPI	50H
	JRC	LB0100
	MVI	A,4FH
LB0100:	STA	COLUMN
	JMP	CURSTOSAVED
;
ANSIINSERTLINE:
	MOV	A,B		;INSERT LINE(S)
	ORA	A		;ANSI ESC [ L
	JZ	INSERTLINE
	LDAX	D
	ORA	A
	JZ	INSERTLINE
LB0103:	PUSH	PSW
	CALL	INSERTLINE
	POP	PSW
	DCR	A
	JRNZ	LB0103
	RET
;
ANSIDELLINE:
	MOV	A,B		;DELETE LINE(S)
	ORA	A		;ANSI ESC [ M
	JZ	DELLINE
	LDAX	D
	ORA	A
	JZ	DELLINE
LB0102:	PUSH	PSW
	CALL	DELLINE
	POP	PSW
	DCR	A
	JRNZ	LB0102
	RET
;
ANSIDELCHAR:
	MOV	A,B		;DELETE CHARACTER(S)
	ORA	A		;ANSI ESC [ P
	JZ	DELCHAR
	LDAX	D
	ORA	A
	JZ	DELCHAR
LB0101:	PUSH	PSW
	CALL	DELCHAR
	POP	PSW
	DCR	A
	JRNZ	LB0101
	RET
;
ANSIMODMODE:			;CHANGE MODES
	MOV	A,B		;ANSI ESC [ H
	ORA	A
	RZ
	XCHG
	LXI	B,DIP2SAVE
LB0110:	LXI	D,MODEFLAG1
;
IF	ENABDISKBOARD
	MOV	A,M
	CPI	02H
	CZ	DISABKBOARD
ENDIF
;
	MOV	A,M
	PUSH	H
	CPI	04H
	CZ	INSERTCHARON
	POP	H
	MOV	A,M
	CPI	14H
	CZ	AUTOCRON
	MOV	A,M
	CPI	'H'
	RZ
	INX	H
	JR	LB0110
;
ANSIMODMODE2:			;CHANGE MODES
	MOV	A,B		;ANSI ESC [ L
	ORA	A
	RZ
	XCHG
	LXI	B,DIP2SAVE
LB0104:	LXI	D,MODEFLAG1
;
IF	ENABDISKBOARD
	MOV	A,M
	CPI	02H
	CZ	ENABKBOARD
ENDIF
;
	MOV	A,M
	CPI	04H
	PUSH	H
	CZ	INSERTCHAROFF
	POP	H
	MOV	A,M
	CPI	14H
	CZ	AUTOCROFF
	MOV	A,M
	CPI	'L'
	RZ
	INX	H
	JR	LB0104
;
ANSIDISPLAYMODE:		;CHANGE DISPLAY MODE
	MOV	A,B		;ANSI ESC [ M
	ORA	A
	JRZ	LB0107
	PUSH	D
LB0106:	POP	D
	LDAX	D
	INX	D
	PUSH	D
	LXI	H,LB0106
	PUSH	H
LB0107:	LXI	H,ANSITAB3
	MVI	D,04H
	MVI	E,03H
	CALL	LOOKUP
	JRC	LB0108
	INX	H
	MOV	H,M
	MOV	L,A
	LXI	D,MODEFLAG1
	PCHL
;
LB0108:	POP	H
	POP	D
	RET
;
ANSICURSREPORT:
	MOV	A,B		;REPORT CURSOR POSITION
	ORA	A		;ANSI ESC [ H
	RZ
	LDAX	D
	CPI	06H
	RNZ
	CALL	OUTESC
	MVI	A,'['
	CALL	OUTPUTA
	LDA	ROW
	INR	A
	CALL	BINDEC
	MOV	A,D
	ORA	A
	JRZ	LB0092
	ORI	30H
	CALL	OUTPUTA
LB0092:	MOV	A,E
	ORI	30H
	CALL	OUTPUTA
	MVI	A,';'
	CALL	OUTPUTA
	LDA	COLUMN
	INR	A
	CALL	BINDEC
	MOV	A,D
	ORA	A
	JRZ	LB0093
	ORI	30H
	CALL	OUTPUTA
LB0093:	MOV	A,E
	ORI	30H
	CALL	OUTPUTA
	MVI	A,'R'
	JMP	OUTPUTA
;
BINDEC:	MVI	D,00H
LB0138:	CPI	0AH
	JRC	LB0139
	SUI	0AH
	INR	D
	JR	LB0138
LB0139:	MOV	E,A
	RET
;
;
ANSIMODBAUD:
	MOV	A,B		;MODIFY BAUD RATE
	ORA	A		;ANSI ESC [ R
	JRZ	LB0105
	XCHG
	MOV	A,M
	CPI	10H
	RNC
LB0105:	JMP	BAUDMOD1
;
;**************************************
;
ANSIPARAMEVAL:			;EVALUATE ANSI PARAMETERS
	LXI	H,ANSISAVE
	LXI	B,0
	MOV	M,B
	CALL	GETRECBYTE
	CPI	';'
	RZ
LB0191:	CPI	'0'
	JRC	LB0195
	CPI	':'
	JRNC	LB0192
	ANI	0FH
	MOV	D,A
	MOV	A,M
	RLC
	MOV	M,A
	RLC
	RLC
	ADD	M
	ADD	D
	MOV	M,A
	INR	B
	INR	C
	JR	LB0194
LB0192:	CPI	';'
	JRNZ	LB0195
	MOV	A,L
	CPI	0DAH
	JRZ	LB0193
	INX	H
LB0193:	MVI	C,00H
	MVI	M,00H
LB0194:	CALL	GETRECBYTE
	JR	LB0191
LB0195:	MOV	D,A
	XRA	A
	ORA	C
	JRZ	LB0196
	INX	H
LB0196:	MOV	M,D
	MOV	A,D
	LXI	D,ANSISAVE
	RET
;
SETHEATHMODE:			;ENTER HEATH MODE
	LDA	DIP2SAVE
	ANI	0DFH
	STA	DIP2SAVE
	RET
;
ENDIF			;END OF ANSI ROUTINES
;
;*******************************
;	INITIALIZE CRTC
;
INITCRTC:
	IN	DIP1
	ANI	01H
	LXI	H,CRTCTAB2	;CRTC TABLE 2.
	JRNZ	LB0163
	LXI	H,CRTCTAB1	;CRTC TABLE 1.
LB0163:	MVI	B,10H
	MVI	C,00H
LB0164:	MOV	A,C
	OUT	CRTCWA
	MOV	A,M
	OUT	CRTCWD
	INR	C
	INX	H
	DJNZ	LB0164
	RET
;
ATIMES80:
	MOV	L,A
	MVI	H,00H
	DAD	H
	DAD	H
	DAD	H
	DAD	H
	MOV	D,H
	MOV	E,L
	DAD	D
	DAD	D
	DAD	D
	DAD	D
	RET
;
SPACE80:
	MVI	B,05H		;FILL 80 SPACES AT HL.
	JMP	SPACE16
;
SCROLL80:
	MVI	B,05H		;SCROLL 80 BYTES FROM HL TO DE
;
SCROLL16:
	MOV	A,D
	ORI	0F8H
	MOV	D,A
	MOV	A,H
	ORI	0F8H
	MOV	H,A
	MOV	A,M
	STAX	D
	INX	H
	INX	D
	MOV	A,M
	STAX	D
	INX	H
	INX	D
	MOV	A,M
	STAX	D
	INX	H
	INX	D
	MOV	A,M
	STAX	D
	INX	H
	INX	D
	MOV	A,M
	STAX	D
	INX	H
	INX	D
	MOV	A,M
	STAX	D
	INX	H
	INX	D
	MOV	A,M
	STAX	D
	INX	H
	INX	D
	MOV	A,M
	STAX	D
	INX	H
	INX	D
	MOV	A,M
	STAX	D
	INX	H
	INX	D
	MOV	A,M
	STAX	D
	INX	H
	INX	D
	MOV	A,M
	STAX	D
	INX	H
	INX	D
	MOV	A,M
	STAX	D
	INX	H
	INX	D
	MOV	A,M
	STAX	D
	INX	H
	INX	D
	MOV	A,M
	STAX	D
	INX	H
	INX	D
	MOV	A,M
	STAX	D
	INX	H
	INX	D
	MOV	A,M
	STAX	D
	INX	H
	INX	D
	DJNZ	SCROLL16
	RET
;
SPACEDE:
	MOV	A,L
	ANI	0FH
	JRZ	LB0221
	MVI	C,20H
	MOV	M,C
	INX	H
	DCX	D
	JR	SPACEDE
LB0221:	MVI	B,04H
LB0222:	MOV	A,D
	RAR
	MOV	D,A
	MOV	A,E
	RAR
	MOV	E,A
	DJNZ	LB0222
	MOV	A,E
	ORA	A
	RZ
	MOV	B,E
SPACE16:
	MVI	C,20H
LB0224:	MOV	A,H
	ORI	0F8H
	MOV	H,A
	MOV	M,C
	INX	H
	MOV	M,C
	INX	H
	MOV	M,C
	INX	H
	MOV	M,C
	INX	H
	MOV	M,C
	INX	H
	MOV	M,C
	INX	H
	MOV	M,C
	INX	H
	MOV	M,C
	INX	H
	MOV	M,C
	INX	H
	MOV	M,C
	INX	H
	MOV	M,C
	INX	H
	MOV	M,C
	INX	H
	MOV	M,C
	INX	H
	MOV	M,C
	INX	H
	MOV	M,C
	INX	H
	MOV	M,C
	INX	H
	DJNZ	LB0224
	RET
;
;**********************************************************
;
;	BUFFER I/O STARTS HERE.
;
;**********************************************************
;
RBUFEMP:			;GET BYTE FROM RECEIVE BUFFER
	LDA	RBUFCNT		
	ORA	A
	STC
	RZ
	DI
	LDA	RBUFCNT
	DCR	A
	STA	RBUFCNT
	LDA	RECBUFPNTR
	MOV	B,A
	LXI	H,RECBUF
	ADD	L
	MOV	L,A
	MOV	A,B
	INR	A
	ANI	7FH
	STA	RECBUFPNTR
	MOV	A,M
	STC
	CMC
	EI
	RET
;
TBUFGET:			;GET BYTE FROM TRANSMIT BUFFER
	LDA	TBUFCNT
	ORA	A
	STC
	RZ
	DCR	A
	STA	TBUFCNT
	LDA	TBUFPNTR
	MOV	B,A
	LXI	H,TRANSBUF
	ADD	L
	MOV	L,A
	MOV	A,B
	INR	A
	ANI	1FH
	STA	TBUFPNTR
	MOV	A,M
	STC
	CMC
	RET
;
GETRECBYTE:		;WAIT FOR BYTE FROM RECEIVE BUFFER
	PUSH	B	
	PUSH	D
	PUSH	H
LB0151:	CALL	LB0002
	CALL	RBUFEMP
	JRC	LB0151
	POP	H
	POP	D
	POP	B
	RET
;
KBUFEMP:		;GET KEY FROM KEY INTERRUPT BUFFER
	PUSH	B
	PUSH	H
	DI
	LHLD	KBUFPNT
	MOV	A,L
	CPI	0A4H		;KEY INTERRUPT BUFFER EMPTY?
	STC			;YES.--> SET CARRY & RETURN
	JRZ	LB0154
	DCX	H
	DCX	H
	SHLD	KBUFPNT
	LXI	H,KEYBUF
	MOV	D,M
	INX	H
	MOV	E,M
	INX	H
	PUSH	D
	LXI	D,KEYBUF
	LXI	B,KBUFPNT-KEYBUF-2	;KEY BUFFER SIZE-2
	LDIR
	STC
	CMC
	POP	D
LB0154:	POP	H
	POP	B
	EI
	RET
;
RECBUFIN:			;PUT BYTE IN RECEIVE BUFFER
	MOV	C,A
	LDA	RBUFCNT
	CPI	80H		;RECEIVE BUFFER FULL?
	STC			;YES.--> BEEP & RETURN
	JZ	BELL
	MOV	B,A
	LDA	RECBUFPNTR
	ADD	B
	ANI	7FH
	LXI	H,RECBUF
	ADD	L
	MOV	L,A
	MOV	M,C
	MOV	A,B
	INR	A
	STA	RBUFCNT
	STC
	CMC
	MOV	A,C
	RET
;
LB0174:	MOV	C,A
	LDA	MODEFLAG2
	ANI	08H		;ON-LINE?
	JRNZ	LB0175		;YES.-->
	MOV	A,C
	CALL	RECBUFIN
	RET
;
LB0175:	LDA	DIP1SAVE
	ANI	80H		;FULL DUPLEX?
	MOV	A,C
	CZ	RECBUFIN	;NO.-->
	LDA	TBUFCNT
	CPI	20H		;TRANSMIT BUFFER FULL?
	STC
	JZ	BELL		;YES.--> BEEP & RETURN
	MOV	B,A
	LDA	TBUFPNTR
	ADD	B
	ANI	1FH
	LXI	H,TRANSBUF
	ADD	L
	MOV	L,A
	MOV	M,C
	MOV	A,B
	INR	A
	STA	TBUFCNT
	STC
	CMC
	MOV	A,C
LB0176:	RET
;
OUTESC:	MVI	A,ESC
OUTPUTA:			;OUTPUT A TO COMM.
	DI			;ON/OFF LINE, HALF/FULL DUPLEX
	CALL	LB0174
	EI
	RET
;
CINSERTSCROLL:			;CHAR. INSERT SCROLL
	LDA	COLUMN
	NEG
	ADI	4FH
	RZ
	MOV	C,A
	MVI	B,00H
	LHLD	BEGLINABS
	LXI	D,LINELENGTH-1
	DAD	D
	MOV	D,H
	MOV	E,L
	DCX	H
LB0186:	MOV	A,D
	ORI	0F8H
	MOV	D,A
	MOV	A,H
	ORI	0F8H
	MOV	H,A
	LDD
	MOV	A,B
	ORA	C
	JRNZ	LB0186
	LHLD	CURSORABS
	RET
;
LOOKUP:	MOV	C,A		;LOOKUP A IN TABLE
LB0211:	XRA	M		;HL = TABLE
	JRZ	LB0213		;D  = # OF ENTRYS
	MOV	B,E		;E  = TABLE INCREMENT
	MOV	A,C		;CARRY = 1 IF NOT FOUND.
LB0212:	INX	H
	DJNZ	LB0212
	DCR	D
	JRNZ	LB0211
	MOV	A,C
	STC
	RET
LB0213:	INX	H
	MOV	A,M
	RET
;
UPDATECRTC:			;UPDATE CRTC CURSOR SAVE
	LDA	ROW		;AND ENABLE SYNC. INTERRUPT
	CALL	ATIMES80
	LDA	COLUMN
	MOV	E,A
	MVI	D,00H
	DAD	D
	XCHG
	LHLD	TOPABS
	MOV	A,H
	ANI	07H
	MOV	H,A
	DAD	D
	MOV	A,H
	ANI	3FH
	MOV	H,A
	SHLD	CURSORCRTC
ENABSYNCINT:
	LDA	MODEFLAG1
	ANI	08H		;GET AUX CHAR. GEN. FLAG BIT
	ORI	CRTCIE
	MOV	C,A
	XRA	A
	OUTP	A		;ENABLE SYNC INTERRUPT
	RET
;
;
BYTETOCOMM:
	PUSH	PSW
LB0229:	IN	UARTLS
	ANI	60H		;TRANSMIT S/R & BUFFER EMPTY?
	CPI	60H
	JRNZ	LB0229		;NO.-->
LB0230:	IN	UARTMS
	ANI	TRANSMASK	;OK TO TRANSMIT?
	CPI	TRANSMASK
	JRNZ	LB0230		;NO.-->
	POP	PSW
	OUT	UARTDL
	RET
;
LINETOCOMM:
	MVI	H,50H
LB0231:	PUSH	H
	LDAX	D
	MOV	C,A
	INX	D
	MOV	A,D
	ORI	0F8H
	MOV	D,A
	PUSH	D
	MOV	A,C
	ANI	7FH
	CPI	20H		;GRAPHIC CHAR?
	JRC	LB0232		;YES.-->
	CPI	7FH		;SPECIAL GRAPHIC?
	JRNZ	LB0236		;NO.-->
	MVI	A,5EH
	JR	LB0234
;
LB0232:	CPI	1FH		;SPECIAL CASE?
	JRNZ	LB0233		;NO.-->
	MVI	A,5FH		;MAKE PRINTABLE
	JR	LB0234
;
LB0233:	ORI	60H		;MAKE PRINTABLE
LB0234:	PUSH	PSW
	BIT	1,B		;GRAPHIC FLAG?
	JRNZ	LB0238		;YES.-->
	SETB	1,B		;SET FLAG
;
IF	ANSI
	CALL	MODETEST	;HEATH MODE?
	JRZ	LB0235		;YES.-->
	CALL	STRINGCOMMONLY
	DB	'[10','m' OR 80H	;ANSI ENTER GRAPHICS
ENDIF
;
LB0235:	CALL	STRINGCOMMONLY
	DB	'F' OR 80H	;ENTER GRAPHICS MODE
	JR	LB0238
;
LB0236:	PUSH	PSW
	BIT	1,B		;GRAPHIC FLAG?
	JRZ	LB0238		;NO.-->
	RES	1,B		;RESET FLAG
;
IF	ANSI
	CALL	MODETEST	;HEATH MODE?
	JRZ	LB0237		;YES.-->
	CALL	STRINGCOMMONLY
	DB	'[11','m' OR 80H	;ANSI EXIT GRAPHICS
ENDIF
;
LB0237:	CALL	STRINGCOMMONLY
	DB	'G' OR 80H	;EXIT GRAPHICS MODE
LB0238:	BIT	7,C		;REVERSE VIDEO?
	JRZ	LB0240		;NO.-->
	BIT	2,B		;REVERSE VIDEO FLAG?
	JRNZ	LB0242		;YES.-->
	SETB	2,B		;SET FLAG
;
IF	ANSI
	CALL	MODETEST	;HEATH MODE?
	JRZ	LB0239		;YES.-->
	CALL	STRINGCOMMONLY
	DB	'[7','m' OR 80H	;ANSI ENTER REVERSE VIDEO
ENDIF
;
LB0239	CALL	STRINGCOMMONLY
	DB	'p' OR 80H	;ENTER REVERSE VIDEO MODE
	JR	LB0242
;
LB0240:	BIT	2,B		;REVERSE VIDEO FLAG?
	JRZ	LB0242		;NO.-->
	RES	2,B		;RESET FLAG
;
IF	ANSI
	CALL	MODETEST	;HEATH MODE?
	JRZ	LB0241		;YES.-->
	CALL	STRINGCOMMONLY
	DB	'[','m' OR 80H	;ANSI EXIT REVERSE VIDEO
ENDIF
;
LB0241:	CALL	STRINGCOMMONLY
	DB	'q' OR 80H	;EXIT REVERSE VIDEO MODE
LB0242:	POP	PSW
	CALL	BYTETOCOMM
	POP	D
	POP	H
	DCR	H
	JNZ	LB0231
	RET
;
STRINGCOMMONLY:			;ESC STRING TO COMM. ONLY
	MVI	A,ESC		;STRING FOLLOWS CALL AND
	CALL	BYTETOCOMM	;ENDS WITH MSB=1.
	POP	D
LB0245:	LDAX	D
	INX	D
	ORA	A
	JM	LB0246
	CALL	BYTETOCOMM
	JR	LB0245
;
LB0246:	ANI	7FH
	CALL	BYTETOCOMM
	PUSH	D
	RET
;
;**********************************
;
	DB	'JHCTERM'
;
TABLESTART	EQU	$	;NEED THIS FOR SPLICING IN
				;	THE TABLES.
;
;**********************************************************
	ORG	4000H		;START OF RAM
RECBUF:		DS	80H
TRANSBUF:	DS	20H
RECBUFPNTR:	DB	0
RBUFCNT:	DB	0
TBUFPNTR:	DB	0
TBUFCNT:	DB	0
KEYBUF:		DS	10H	;THIS MAKES IT:
				;16 BYTES IF 256 BYTE RAM OR
				;272 BYTES IF 1K RAM
KBUFPNT:	DW	0
TOPABS:		DW	0
BEGLINABS:	DW	0
COLUMN:		DB	0
ROW:		DB	0
CURSORABS:	DW	0
CRTCROWS:	DB	0
CURSORTYPE:	DW	0
TOPCRTC:	DW	0
CURSORCRTC:	DW	0
CURSORSAVE:	DW	0
MODEFLAG1:	DB	0
DIP2SAVE:	DB	0
MODEFLAG2:	DB	0
DIP1SAVE:	DB	0
;
TIMER:		DB	0
KSAVE:		DW	0
XOFFSAVE:	DB	0
;
HOLDCNT:	DB	0
ANSISAVE:	DS	4
;
STACK		EQU	43FFH	;=40FFH IF, 256 BYTE RAM.
;
;**********************************
;
;	TABLES START HERE.		3/19/81
;
;**********************************
;
;	SPECIAL KEY DEFINES
;
SCROLLKEY	EQU	4FH	;SCROLL KEY COORDINATES
BREAKKEY	EQU	62H	;BREAK KEY COORDINATES
ENABDISKEY	EQU	06H	;ENABLE/DISABLE KEYBOARD TOGGLE
				;KEY COORDINATES
;
;*************************************************************
	ORG	TABLESTART	;THIS SPLICES THE TABLES AFTER
				;	THE PROGRAM
IF NOT JAMES
;
;	KEYBOARD TABLES START HERE.
KTABLE:
;	THIS TABLE IS FOR THE CHERRY (ADDS) KEYBOARD
;
;	START OF UN-SHIFTED TABLE.
	DB	'4'	;00
	DB	5FH	;01 - UNDERLINE
	DB	'u'	;02
	DB	0E9H	;03 - 9 KP
	DB	'z'	;04
	DB	'.'	;05
	DB	0B0H	;06 - F1
	DB	0E1H	;07 - 1 KP
	DB	'['	;08
	DB	0	;09
	DB	0E6H	;0A - 6 KP
	DB	0B6H	;0B - F7
	DB	'-'	;0C - '-' KP
	DB	'l'	;0D
	DB	0	;0E
	DB	0	;0F
	DB	'6'	;10
	DB	'g'	;11
	DB	0	;12
	DB	0	;13
	DB	0	;14
	DB	5CH	;15 - BACK SLASH
	DB	8CH	;16 - ERASE
	DB	0BAH	;17 - RT. ARROW FUNCT.
	DB	'j'	;18
	DB	0EAH	;19 - '.' KP
	DB	0B3H	;1A - F4
	DB	0B5H	;1B - F6
	DB	5EH	;1C - UP ARROW CHAR.
	DB	'i'	;1D
	DB	0E5H	;1E - 5 KP
	DB	'x'	;1F
	DB	'h'	;20
;
	DB	0	;21
	DB	0	;22
	DB	0	;23
	DB	0	;24
	DB	0	;25
	DB	0	;26
	DB	0	;27
	DB	0	;28
	DB	0	;29
	DB	0	;2A
	DB	0	;2B
;
	DB	0BBH	;2C - UP ARROW FUNCT.
	DB	'5'	;2D
	DB	'p'	;2E
	DB	'w'	;2F
	DB	'8'	;30
	DB	's'	;31
	DB	0E4H	;32 - 4 KP
	DB	']'	;33
	DB	'n'	;34
	DB	';'	;35
	DB	20H	;36 - SPACE BAR
	DB	'r'	;37
	DB	'9'	;38
	DB	'1'	;39
	DB	'e'	;3A
	DB	'7'	;3B
	DB	'/'	;3C
	DB	0	;3D
	DB	8EH	;3E - LOCAL (OFF-LINE)
	DB	0	;3F
	DB	'b'	;40
;
	DB	0	;41
	DB	0	;42
	DB	0	;43
	DB	0	;44
	DB	0	;45
	DB	0	;46
	DB	0	;47
	DB	0	;48
	DB	0	;49
	DB	0	;4A
;
	DB	'q'	;4B
	DB	'o'	;4C
	DB	'c'	;4D
	DB	0E8H	;4E - 8 KP
	DB	0AFH	;4F - AUX (SCROLL)
	DB	','	;50 - ',' KP
;
	DB	0	;51
	DB	0	;52
	DB	0	;53
	DB	0	;54
	DB	0	;55
	DB	0	;56
	DB	0	;57
	DB	0	;58
	DB	0	;59
	DB	0	;5A
;
	DB	TAB	;5B - TAB
	DB	0B7H	;5C - F8
	DB	0B8H	;5D - DOWN ARROW FUNCT.
	DB	0EBH	;5E - ENTER KP
	DB	0B4H	;5F - F5
	DB	0	;60
	DB	'f'	;61
	DB	8DH	;62 - BREAK
	DB	'k'	;63
	DB	LF	;64 - LINE FEED
	DB	ESC	;65 - ESCAPE
	DB	','	;66
	DB	BSPACE	;67 - BACKSPACE
	DB	0E2H	;68 - 2 KP
	DB	'0'	;69
	DB	'y'	;6A
	DB	':'	;6B
	DB	'3'	;6C
	DB	'v'	;6D
	DB	CR	;6E - NEW LINE
	DB	'2'	;6F
	DB	'a'	;70
	DB	'd'	;71
	DB	'm'	;72
	DB	0BCH	;73 - HOME
	DB	40H	;74
	DB	't'	;75
	DB	0B9H	;76 - LEFT ARROW FUNCT.
	DB	0E7H	;77 - 7 KP
	DB	0E3H	;78 - 3 KP
	DB	0E0H	;79 - 0 KP
	DB	0B2H	;7A - F3
;
	DB	0	;7B
	DB	0	;7C
	DB	0	;7D
	DB	0	;7E
;
	DB	0B1H	;7F - F2

;	START OF SHIFT TABLE.
	DB	'$'	;00
	DB	'='	;01
	DB	'U'	;02
	DB	0E9H	;03 - 9 KP
	DB	'Z'	;04
	DB	'>'	;05
	DB	0B0H	;06 - F1
	DB	0E1H	;07 - 1 KP
	DB	7BH	;08 - OPEN BRACKET
	DB	0	;09
	DB	0E6H	;0A - 6 KP
	DB	0B6H	;0B - F7
	DB	0	;0C - '-' KP
	DB	'L'	;0D
	DB	0	;0E
	DB	0	;0F
	DB	'&'	;10
	DB	'G'	;11
	DB	0	;12
	DB	0	;13
	DB	0	;14
	DB	7CH	;15 - BROKEN VERT. BAR
	DB	8CH	;16 - ERASE
	DB	0BAH	;17 - RT. ARROW FUNCT.
	DB	'J'	;18
	DB	'.'	;19 - '.' KP
	DB	0B3H	;1A - F4
	DB	0B5H	;1B - F6
	DB	7EH	;1C - TILDA
	DB	'I'	;1D
	DB	0E5H	;1E - 5 KP
	DB	'X'	;1F
	DB	'H'	;20
;
	DB	0	;21
	DB	0	;22
	DB	0	;23
	DB	0	;24
	DB	0	;25
	DB	0	;26
	DB	0	;27
	DB	0	;28
	DB	0	;29
	DB	0	;2A
	DB	0	;2B
;
	DB	0BBH	;2C - UP ARROW FUNCT.
	DB	'%'	;2D
	DB	'P'	;2E
	DB	'W'	;2F
	DB	'('	;30
	DB	'S'	;31
	DB	0E4H	;32 - 4 KP
	DB	7DH	;33 - CLOSE BRACKET
	DB	'N'	;34
	DB	'+'	;35
	DB	20H	;36 - SPACE BAR
	DB	'R'	;37
	DB	')'	;38
	DB	21H	;39 - EXCLAIMATION
	DB	'E'	;3A
	DB	27H	;3B - "'"
	DB	'?'	;3C
	DB	0	;3D
	DB	8EH	;3E - LOCAL (ON-LINE)
	DB	0	;3F
	DB	'B'	;40
;
	DB	0	;41
	DB	0	;42
	DB	0	;43
	DB	0	;44
	DB	0	;45
	DB	0	;46
	DB	0	;47
	DB	0	;48
	DB	0	;49
	DB	0	;4A
;
	DB	'Q'	;4B
	DB	'O'	;4C
	DB	'C'	;4D
	DB	0E8H	;4E - 8 KP
	DB	0AFH	;4F - AUX
	DB	0	;50 - ',' KP
;
	DB	0	;51
	DB	0	;52
	DB	0	;53
	DB	0	;54
	DB	0	;55
	DB	0	;56
	DB	0	;57
	DB	0	;58
	DB	0	;59
	DB	0	;5A
;
	DB	TAB	;5B - TAB
	DB	0B7H	;5C - F8
	DB	0B8H	;5D - DOWN ARROW FUNCT.
	DB	CR	;5E - ENTER KP
	DB	0B4H	;5F - F5
	DB	0	;60
	DB	'F'	;61
	DB	8DH	;62 - BREAK
	DB	'K'	;63
	DB	LF	;64 - LINE FEED
	DB	ESC	;65 - ESCAPE
	DB	'<'	;66
	DB	7FH	;67 - DELETE
	DB	0E2H	;68 - 2 KP
	DB	0	;69 - 0 KEY
	DB	'Y'	;6A
	DB	'*'	;6B
	DB	'#'	;6C
	DB	'V'	;6D
	DB	CR	;6E - NEW LINE
	DB	'"'	;6F
	DB	'A'	;70
	DB	'D'	;71
	DB	'M'	;72
	DB	0BCH	;73 - HOME
	DB	60H	;74
	DB	'T'	;75
	DB	0B9H	;76 - LEFT ARROW FUNCT.
	DB	0E7H	;77 - 7 KP
	DB	0E3H	;78 - 3 KP
	DB	'0'	;79 - 0 KP
	DB	0B2H	;7A - F3
;
	DB	0	;7B
	DB	0	;7C
	DB	0	;7D
	DB	0	;7E
;
	DB	0B1H	;7F - F2
;
;	END OF MAIN TABLE.
;**************************************
;	FUNCTION KEY TABLES
;	UN-SHIFTED
FUNTAB:			;MSB = 0 IF PRINTABLE.
			;		MAIN TABLE VALUE B4-B0
	DB	'0'	;KEYPAD 0		00
	DB	'1'	;	1		01
	DB	'2'	;	2		02
	DB	'3'	;	3		03
	DB	'4'	;	4		04
	DB	'5'	;	5		05
	DB	'6'	;	6		06
	DB	'7'	;	7		07
	DB	'8'	;	8		08
	DB	'9'	;	9		09
	DB	'.'	;	.		0A
	DB	CR	;	ENTER		0B
;
	DB	0E2H	;ERASE			0C
	DB	0E4H	;BREAK			0D
	DB	0E5H	;LOCAL			0E
	DB	0E1H	;AUX			0F
	DB	13H	;F1 (CTL-S)		10
	DB	03H	;F2 (CTL-C)		11
	DB	10H	;F3 (CTL-P)		12
	DB	0	;F4			13
	DB	0	;F5			14
	DB	0	;F6			15
	DB	0	;F7			16
	DB	0	;F8			17
	DB	0C1H	;DOWN ARROW		18
	DB	0C3H	;LEFT ARROW		19
	DB	0C5H	;RIGHT ARROW		1A
	DB	0C6H	;UP ARROW		1B
	DB	0C4H	;HOME			1C
;
;		SHIFTED FUNCTIONS
FUNSHIFTTAB:
	DB	0	;KEYPAD 0
	DB	0C0H	;	1	INSERT LINE
	DB	0C1H	;	2	CURSOR DOWN
	DB	0C2H	;	3	DELETE LINE
	DB	0C3H	;	4	CURSOR LEFT
	DB	0C4H	;	5	CURSOR HOME
	DB	0C5H	;	6	CURSOR RIGHT
	DB	0E0H	;	7	INSERT CHAR. TOGGLE
	DB	0C6H	;	8	CURSOR UP
	DB	0E3H	;	9	DELETE CHAR.
	DB	0	;	.
	DB	0	;	ENTER
;
	DB	0E2H	;ERASE
	DB	0E4H	;BREAK
	DB	0E5H	;LOCAL
	DB	0E1H	;AUX
;	DB	0	;F1	PRESENTLY UNASSIGNED
;	DB	0	;F2
;	DB	0	;F3
;	DB	0	;F4
;	DB	0	;F5
;	DB	0	;F6
;	DB	0	;F7
;	DB	0	;F8
;	DB	0	;DOWN ARROW
;	DB	0	;LEFT ARROW
;	DB	0	;RIGHT ARROW
;	DB	0	;UP ARROW
;	DB	0	;HOME
;
;		ALTERNATE FUNCTIONS  (SETUP FOR 'WORDSTAR')
ALTTAB:
	DB	'0'	;KEYPAD 0
	DB	'1'	;	1
	DB	'2'	;	2
	DB	'3'	;	3
	DB	'4'	;	4
	DB	'5'	;	5
	DB	'6'	;	6
	DB	'7'	;	7
	DB	'8'	;	8
	DB	'9'	;	9
	DB	'.'	;	.
	DB	CR	;	ENTER
;
 	DB	0	;ERASE
 	DB	0	;BREAK
 	DB	0E5H	;LOCAL
 	DB	12H	;AUX		SCREEN UP
 	DB	16H	;F1		INSERT MODE
 	DB	02H	;F2		REFORM PARA.
 	DB	0A0H	;F3		BOLDFACE
 	DB	0A1H	;F4		UNDERSCORE
 	DB	0A2H	;F5		CENTER LINE
 	DB	0A3H	;F6		JUSTIFY ON/OFF
 	DB	0A4H	;F7		WORD WRAP ON/OFF
 	DB	0	;F8
 	DB	18H	;DOWN ARROW	DOWN LINE
 	DB	13H	;LEFT ARROW	LEFT CHAR.
 	DB	04H	;RIGHT ARROW	RIGHT CHAR.
 	DB	05H	;UP ARROW	UP LINE
 	DB	0A5H	;HOME		BEGIN. OF FILE
;
;	ALTERNATE SHIFTED FUNCTIONS
ALTSHIFTTAB:
	DB	16H	;KEYPAD 0	CTL-V
	DB	13H	;	1	 " -S
	DB	04H	;	2	 " -D
	DB	14H	;	3	 " -T
	DB	01H	;	4	 " -A
	DB	06H	;	5	 " -F
	DB	17H	;	6	 " -W
	DB	18H	;	7	 " -X
	DB	05H	;	8	 " -E
	DB	1AH	;	9	 " -Z
	DB	02H	;	.	 " -B
	DB	19H	;	ENTER	 " -Y
;
	DB	0	;ERASE
 	DB	0	;BREAK
 	DB	0E5H	;LOCAL
 	DB	03H	;AUX		SCREEN DOWN
 	DB	7FH	;F1		DELETE CHAR. LEFT
 	DB	07H	;F2		  "     "    RIGHT
 	DB	14H	;F3		  "    WORD RIGHT
 	DB	0A6H	;F4		  "    BEGIN. OF LINE
 	DB	0A7H	;F5		  "    END OF LINE
 	DB	19H	;F6		  "    ENTIRE LINE
 	DB	0A8H	;F7		ALTERNATE PITCH
 	DB	0	;F8
 	DB	17H	;DOWN ARROW	SCROLL DOWN
 	DB	01H	;LEFT ARROW	LEFT WORD
 	DB	06H	;RIGHT ARROW	RIGHT WORD
 	DB	1AH	;UP ARROW	SCROLL UP
 	DB	0A9H	;HOME		END OF FILE
;
;	TYPE 1 FUNCTIONS DATA TABLE
;
TYPE1TAB:	;NONE USED
;
;	TYPE 2 FUNCTIONS DATA TABLE
;
TYPE2TAB:			;ALT.FUNCTIONS  (WORDSTAR)
	DB	'P' AND 1FH,'B' OR 80H	;BOLDFACE
	DB	'P' AND 1FH,'S' OR 80H	;UNDERSCORE
	DB	'O' AND 1FH,'C' OR 80H	;CENTER LINE
	DB	'O' AND 1FH,'J' OR 80H	;JUSTIFY ON/OFF
	DB	'O' AND 1FH,'W' OR 80H	;WORD WRAP ON/OFF
	DB	'Q' AND 1FH,'R' OR 80H	;BEGIN. OF FILE
	DB	'Q' AND 1FH,7FH OR 80H	;DELETE BEGIN. OF LINE
	DB	'Q' AND 1FH,'Y' OR 80H	;DELETE END OF LINE
	DB	'P' AND 1FH,'A' OR 80H	;DELETE ENTIRE LINE
	DB	'Q' AND 1FH,'C' OR 80H	;END OF FILE

;
;	TYPE 3 FUNCTIONS DATA TABLE
;
TYPE3TAB:			;SHIFTED PAD (EXCEPT TYPE 4)
	DB	'L' OR 80H	;1
	DB	'B' OR 80H	;2
	DB	'M' OR 80H	;3
	DB	'D' OR 80H	;4
	DB	'H' OR 80H	;5
	DB	'C' OR 80H	;6
	DB	'A' OR 80H	;8
;
;	TYPE 4 FUNCTIONS ROUTINE ADDR. TABLE
;
TYPE4TAB:
	DW	INSERTKEY		;PAD 7 SHIFT
	DW	SCROLKEY		;AUX
	DW	ERASEKEY		;ERASE
	DW	DELCHARKEY		;PAD 9 SHIFT
	DW	BREAKRET		;BREAK (RETURN ONLY)
;
	DW	OFFLINE			;LOCAL (CAPS. LOCK)
					;DUAL FUNCTION
;
;	TYPE 4 ROUTINES DATA TABLES
;
INSERTAB:
	DB	40H OR 80H		;HEATH INSERT ON
	DB	'4','h' OR 80H		;ANSI
	DB	'O' OR 80H		;HEATH INSERT OFF
	DB	'4','l' OR 80H		;ANSI
;
ERATAB:	DB	'J' OR 80H		;HEATH ERASE EOP
	DB	'J' OR 80H		;ANSI
	DB	'E' OR 80H		;HEATH CLEAR & HOME
	DB	'2','J' OR 80H		;ANSI
;
DELTAB:	DB	'N' OR 80H		;HEATH DELETE CHAR.
	DB	'P' OR 80H		;ANSI
;
;	END OF KEYBOARD TABLES
ENDIF
;
;*******************************
;
;	HEATH MODE TABLE
HEATHTAB:
	DB	ESC
	DW	SETESCFLAG
;
IF	ANSI
	DB	'<'		;ENTER ANSI MODE
	DW	ENTERANSI
ENDIF
;
	DB	'#'		;TRANSMIT PAGE
	DW	TRANSPAGE
	DB	'@'		;ENTER INSERT CHAR. MODE
	DW	INSERTCHARON
	DB	'A'		;CURSOR UP
	DW	CURSUP
	DB	'B'		;CURSOR DOWN
	DW	CURSDOWN
	DB	'C'		;CURSOR FORWARD
	DW	CURSFORWARD
	DB	'D'		;CURSOR BACK (BACKSPACE)
	DW	BACKSPACE
	DB	'E'		;CLEAR SCREEN & HOME CURSOR
	DW	CLEARHOME
	DB	'F'		;ENTER GRAPHICS MODE
	DW	SETGRAPHICS
	DB	'G'		;EXIT GRAPHICS MODE
	DW	RESETGRAPHICS
	DB	'H'		;HOME CURSOR
	DW	CURSHOME
	DB	'I'		;REVERSE LINEFEED
	DW	REVLFEED
	DB	'J'		;ERASE TO END OF PAGE
	DW	ERASEEOP
	DB	'K'		;ERASE TO END OF LINE
	DW	ERASEEOL
	DB	'L'		;INSERT LINE
	DW	INSERTLINE
	DB	'M'		;DELETE LINE
	DW	DELLINE
	DB	'N'		;DELETE CHARACTER
	DW	DELCHAR
	DB	'O'		;EXIT INSERT CHAR. MODE
	DW	INSERTCHAROFF
	DB	'Y'		;DIRECT CURSOR ADDRESS
	DW	CURSADDR
	DB	'Z'		;IDENTIFY AS VT52
	DW	IDENTVT52
;
IF	SCREENHOLD
	DB	'['		;ENTER HOLD SCREEN MODE
	DW	SETHOLDSCREEN
	DB	'\'		;EXIT HOLD SCREEN MODE
	DW	RESETHOLDSCREEN	
ENDIF
	DB	']'		;TRANSMIT 25th LINE
	DW	TRANS25TH
	DB	'b'		;ERASE BEGINNING OF SCREEN
	DW	ERASEBEGDIS
	DB	'c'		;ENABLE AUX CHAR. GENERATOR
	DW	SETAUXCHAR
	DB	'd'		;DISABLE AUX CHAR. GENERATOR
	DW	RESETAUXCHAR
	DB	'j'		;SAVE CURSOR POSITION
	DW	SAVECURS
	DB	'k'		;SET CURSOR TO SAVED POSITION
	DW	SETCURS
	DB	'l'		;ERASE LINE
	DW	ERASELINE
	DB	'n'		;CURSOR POSITION REPORT
	DW	CURSPOSREPORT
	DB	'o'		;ERASE FROM BEGINNING OF LINE
	DW	ERASEBOL
	DB	'p'		;ENTER REVERSE VIDEO MODE
	DW	SETREVVIDEO
	DB	'q'		;EXIT REVERSE VIDEO MODE
	DW	RESETREVVIDEO
	DB	'r'		;MODIFY BAUD RATE
	DW	MODBAUDRATE
	DB	'v'		;WRAP AT END OF LINE
	DW	SETLINEWRAP
	DB	'w'		;DISCARD AT END OF LINE
	DW	SETNOWRAP
	DB	'x'		;SET MODE
	DW	SETMODE
	DB	'y'		;RESET MODE
	DW	RESETMODE
	DB	'z'		;RESET TO POWER UP CONFIG.
	DW	REINIT
;
IF	ENABDISKBOARD
	DB	7BH	;OPEN BRACKET
	DW	ENABKBOARD		;ENABLE KEYBOARD
	DB	7DH	;CLOSE BRACKET
	DW	DISABKBOARD		;DISABLE KEYBOARD
ENDIF
HEATHEND: EQU	 $
;
;	END OF HEATH MODE TABLE
;
;***************************************
IF	ANSI
;
;	ANSI MODE TABLE
ANSITAB:
	DB	'='		;ENTER ALT. KEYPAD MODE
	DW	SETALTPAD
	DB	'>'		;EXIT ALT. KEYPAD MODE
	DW	RESETALTPAD
	DB	'M'		;REVERSE LINEFEED
	DW	REVLFEED
	DB	'['		;STRING LEAD-IN CHAR.
	DW	ANSIBRACKET
;
;	ANSI "ESC [" LOOKUP TABLE
ANSITAB2:
	DB	'>'
	DW	SETRESETMODE
	DB	'?'
	DW	CHANGEMODE
	DB	'A'
	DW	ANSICURSUP
	DB	'B'
	DW	ANSICURSDOWN
	DB	'C'
	DW	ANSICURSFOR
	DB	'D'
	DW	ANSICURSBACK
	DB	'H'
	DW	ANSICURSADDR
	DB	'J'
	DW	ANSIERASEDIS
	DB	'K'
	DW	ANSIERASELINE
	DB	'L'
	DW	ANSIINSERTLINE
	DB	'M'
	DW	ANSIDELLINE
	DB	'P'
	DW	ANSIDELCHAR
	DB	'f'
	DW	ANSICURSADDR
	DB	'h'
	DW	ANSIMODMODE
	DB	'l'
	DW	ANSIMODMODE2
	DB	'm'
	DW	ANSIDISPLAYMODE
	DB	'n'
	DW	ANSICURSREPORT
	DB	'p'
	DW	ANSITRANSPAGE
	DB	'q'
	DW	ANSITRANS25
	DB	'r'
	DW	ANSIMODBAUD
	DB	's'
	DW	ANSISAVECURS
	DB	'u'
	DW	ANSICURSTOSAVE
	DB	'z'
	DW	ANSIREINIT
;
;
;			ANSI ESC [ M TABLE
;
ANSITAB3:
	DB	0
	DW	RESETREVVIDEO
	DB	07H
	DW	SETREVVIDEO
	DB	0AH
	DW	SETGRAPHICS
	DB	0BH
	DW	RESETGRAPHICS
;
;	END OF ANSI TABLES
;
ENDIF
;
;**************************************
;
;	COMMON TABLE
;
COMMONTAB:
	DB	BEL
	DW	BELL
	DB	BSPACE
	DW	BACKSPACE
	DB	TAB
	DW	HORIZTAB
	DB	LF
	DW	LINEFEED
	DB	FF
	DW	CLEARHOME
	DB	CR
	DW	CRETURN
	DB	ESC
	DW	SETESCFLAG
;
;	END OF COMMON TABLE
;
;*******************************
;
;	BAUD RATE TABLE		1.536 or 3.072 MHz.
;
BAUDTAB:		;     DIPSWITCH	      MODIFY
IF	CPUCLOCK	;		3.072 MHz TABLE
	DW	1745	;110	7		A
	DW	1280	;150	-		B
	DW	640	;300	6		C
	DW	640	;300	-		D
	DW	320	;600	5		E
	DW	107	;1800	-		F
	DW	160	;1200	4		G
	DW	96	;2000	-		H
	DW	80	;2400	3		I
	DW	53	;3600	-		J
	DW	40	;4800	2		K
	DW	27	;7200	-		L
	DW	20	;9600	1		M
	DW	20	;9600	-		N
	DW	10	;19200	0		O
ELSE
;					1.536 MHz TABLE
	DW	873	;110	7		A
	DW	640	;150	-		B
	DW	320	;300	6		C
	DW	320	;300	-		D
	DW	160	;600	5		E
	DW	53	;1800	-		F
	DW	80	;1200	4		G
	DW	48	;2000	-		H
	DW	40	;2400	3		I
	DW	27	;3600	-		J
	DW	20	;4800	2		K
	DW	13	;7200	-		L
	DW	10	;9600	1		M
	DW	10	;9600	-		N
	DW	5	;19200	0		O
ENDIF
;
;	SET MODE TABLE
;
SETTAB:	DW	SETFULLDUP	;0
	DW	ENAB25TH	;1
	DW	ENABCLICK	;2
	DW	SETHOLDSCREEN	;3
	DW	SETBLOCKCURS	;4
	DW	CURSON		;5
	DW	SETPADSHIFT	;6
	DW	SETALTPAD	;7
	DW	AUTOLFON	;8
	DW	AUTOCRON	;9
	DW	AUTOREPEAT	;:
;
;
;	RESET MODE TABLE
;
RESETTAB:
	DW	RESETFULLDUP	;0
	DW	DISAB25TH	;1
	DW	DISABCLICK	;2
	DW	RESETHOLDSCREEN	;3
	DW	SETULINECURS	;4
	DW	DISABCURS	;5
	DW	RESETPADSHIFT	;6
	DW	RESETALTPAD	;7
	DW	AUTOLFOFF	;8
	DW	AUTOCROFF	;9
	DW	NOAUTOREPEAT	;:
;
;
;	CRTC INITIALIZATION TABLE 1.
CRTCTAB1:
	DB	96	;R0 = HORIZ. TOTAL-1
	DB	80	;R1 = HORIZ. DISPLAYED
	DB	84	;R2 = HORIZ. SYNC. POSITION
	DB	8	;R3 = HORIZ. SYNC. WIDTH
	DB	25	;R4 = VERT. TOTAL-1
	DB	15	;R5 = VERT. SYNC. ADJUST
	DB	24	;R6 = VERT. DISPLAYED
	DB	25	;R7 = VERT. SYNC. POSITION
	DB	0	;R8 = INTERLACE MODE
	DB	9	;R9 = MAX. SCAN LINE-1
	DB	48H	;R10= CURSOR START SCAN/BLINK RATE
	DB	8	;R11= CURSOR END SCAN
	DB	0	;R12= TOP OF PAGE (HIGH)
	DB	0	;R13= TOP OF PAGE (LOW)
	DB	0	;R14= CURSOR ADDR. (HIGH)
	DB	0	;R15= CURSOR ADDR> (LOW)
p;
;	CRTC TABLE 2.
CRTCTAB2:
	DB	5CH	;R0 = HORIZ. TOTAL-1
	DB	50H	;R1 = HORIZ. DISPLAYED
	DB	51H	;R2 = HORIZ. SYNC. POSITION
	DB	08H	;R3 = HORIZ. SYNC. WIDTH
	DB	19H	;R4 = VERT. TOTAL-1
	DB	0FH	;R5 = VERT. SYNC. ADJUST
	DB	18H	;R6 = VERT. DISPLAYED
	DB	18H	;R7 = VERT. SYNC. POSITION
	DB	3	;R8 = INTERLACE MODE
	DB	9	;R9 = MAX. SCAN LINE-1
	DB	48H	;R10= CURSOR START SCAN/BLINK RATE
	DB	8	;R11= CURSOR END SCAN
	DB	0	;R12= TOP OF PAGE (HIGH)
	DB	0	;R13= TOP OF PAGE (LOW)
	DB	0	;R14= CURSOR ADDR. (HIGH)
	DB	0	;R15= CURSOR ADDR> (LOW)
;
;
		END
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                