/* KeyedCollection is the formal, parent class for any class
 where the elements are accessed symbolically rather than by
 physical integer offsets.

 For example, you could access the cities in Illinois by saying
 Cities["Illinois"].  In this case, the key is "Illinois", and
 the element would be something like Set("Chicago" "Rockford"
 "Evanston" "Belleville" ).

 Elements of a KeyedCollection are inherently unordered. */!!

/* inherit(Collection, #KeyedCollection, #(tally ),
2, 1) */!!

now(class(KeyedCollection))!!

setName(VImage, "work2.ima");!!

now(KeyedCollection)!!

/* Initialize the KeyedCollection by
  setting the tally instance variable
  to 0. */
 Def init(self)
{       tally := 0;
}!!


/* Grow a keyed collection so that it
  can hold more elements.  Works by copying
  elements into larger collection and then
  swapping object pointers with the new
  collection.  */
 Def  grow(self | newColl)
{ newColl := new(class(self), limit(self) * 2);
  keysDo(self,
  {using(elem)  add(newColl, elem,
    self[elem])
  });
  swap(self, newColl);
}!!

/* Return the current size of the
  KeyedCollection. */
 Def size(self)
{ ^tally;
}!!


/* Evaluate the one-argument block over
  the keys of the receiver.  */
 Def keysDo(self,aBlock)
{       ^do(limit(self),
  {using(idx | aKey)
    if  (aKey := at(self:Object, idx))
    then  eval(aBlock, aKey);
    else  nil
    endif;
  });
}!!

/* sysPrint the keyed collection onto
  the specified stream.  sysPrint for keyed
  collections is the same as printOn
  because KeyedCollections have no
  unformatted output.  */
 Def sysPrintOn(self, aStrm)
{ printOn(self, aStrm);
}!!


/* Print the KeyedCollection onto the
  specified Stream.  The keys of the
  collection are what gets placed onto
  the stream.  */
 Def printOn(self, aStrm)
{       nextPutAll(aStrm,
  class(self).name +
  "(" );
  keysDo(self,
  { using(el) sysPrintOn(el, aStrm);
    if aStrm.position > HugeSize
    then  nextPutAll(aStrm,
      "...)");
      ^self;
    endif;
    nextPutAll(aStrm,
    " ");
  });
  nextPutAll(aStrm,
  ")" );
}!!


/* Equals method for KeyedCollections.
  KeyedCollections are equal if they are
  the same size and have equal keys and
  elements. */
 Def =(self, aColl)
{
  if size(self) ~= size(aColl)
  then ^nil
  endif;
  keysDo(self,
  {using(key)
    if not(self[key] = aColl[key])
    then ^nil
    endif;
  });
}!!

/* Return the hash value of the
  receiver.  A keyed collection hashes by
  adding the hash values of its keys and
  producing an Int. */
 Def hash(self | sum)
{ sum := 0;
  keysDo(self,
  { using(key) sum := sum + hash(key);
  });
  ^asInt(sum bitAnd 0x3fff);
}!!

/* Return a Set containing all of the
  keys in the collection.  */
 Def keys(self | set)
{ set := new(Set, size(self) + 2);
  keysDo(self,
  {using(key) add(set, key);
  });
  ^set;
}!!



/* Clear the receiver KeyedCollection.
  All keys and elements are set to nil.  */
Prim clear(self):KeyedCollection!!
