/* This class is a formal class which acts as the parent
 class for all numeric objects, such as Int, Long, and
 Real.  Its major purpose is to handle mixed-mode
 arithmetic, such as 3.4+4 (adding a Real to an Int).

 Handling these situations is called coercion, and many
 of the methods in this class handle arithmetic coercion.  */!!

inherit(Magnitude, #Number, nil, nil, nil);!!

now(class(Number));!!

/* Reports an error if new is sent to a number. */
Def new(self)
{ error(self, stackTop(0), #newError);
}!!

#define pi 3.141592654
#define Pi 3.141592654

now(Number);!!

/* Determines which number has the highest generality, self or aNum,
  converts the one with the lowest generality to the class of the other,
  and then performs the specified operation.  */
Def coerce(self, aNum, oper)
{ if generality(self) < generality(aNum):Int
  then ^perform(perform(self, converterFor(aNum)), aNum, oper);
  else ^perform(self, perform(aNum, converterFor(self)), oper);
  endif;
}!!

/* Mixed-mode addition operator.  Example: 3+4.3.  */
Def +(self, aNum)
{       ^coerce(self:Number, aNum, #+);
}!!

/* Mixed-mode subtraction operator.  Example: 3L-4.   */
Def -(self, aNum)
{       ^coerce(self:Number, aNum, #-);
}!!

/* Mixed-mode multiplication operator.  Example: 3*4.7.  */
Def *(self, aNum)
{       ^coerce(self:Number, aNum, #*);
}!!

/* Mixed-mode division operator.  Example: 3/4.0.  */
Def /(self, aNum)
{       ^coerce(self:Number, aNum, #/);
}!!

/* Mixed-mode modulus operator.  Example 3 mod 4.  For a definition of
  modulus, see the mod methods for the Int and Long classes.  */
Def mod(self, aNum)
{       ^coerce(self:Number, aNum, #mod);
}!!

/* Mixed-mode bitwise AND operator.  (See bitAnd for the Int and Long
  classes).  */
Def bitAnd(self, aNum)
{       ^coerce(self:Number, aNum, #bitAnd);
}!!

/* Mixed-mode bitwise OR operator.  (See bitAnd for the Int and Long
  classes).  */
Def bitOr(self, aNum)
{       ^coerce(self:Number, aNum, #bitOr);
}!!

/* Mixed-mode bitwise XOR operator.  (See bitXor for the Int and Long
  classes).  */
Def bitXor(self, aNum)
{       ^coerce(self:Number, aNum, #bitXor);
}!!

/* Mixed-mode equals method.  If self = aNum, then return true.  */
Def =(self, aNum)
  { ^
  if generality(aNum) < 0
  then nil
  else coerce(self:Number, aNum, #=);
  endif;
  }!!


/* Mixed-mode less than.  If the argument is less
  than the receiver, then this method
  returns true.  */
Def <(self, aNum)
{       ^coerce(self:Number, aNum, #<);
}!!

/* Mixed-mode greater than.  If the argument is greater
  than the receiver, then this method
  returns true.  */
Def >(self, aNum)
{       ^coerce(self:Number, aNum, #>);
}!!

/* Mixed-mode not equal.  If the argument is not equal to
  the receiver, then this method
  returns true.  */
Def <>(self, aNum)
  { ^
  if generality(aNum) < 0
  then true
  else coerce(self:Number, aNum, #<>);
  endif;
  }!!

/* Mixed-mode less than or equal to.  If the argument is less than or equal
  to the receiver, then this method
  returns true.  */
Def <=(self, aNum)
{       ^coerce(self:Number, aNum, #<= );
}!!

/* Mixed-mode greater than or equal to.  If the argument is greater than or
  equal to the receiver, then this method
  returns true.  */
Def >=(self, aNum)
{       ^coerce(self:Number, aNum, #>= );
}!!

/* Return the absolute value of the receiver.  For example,
  abs(-4.0) returns 4.0, and abs(37L) returns 37L.  */
Def abs(self)
{       if self < 0
        then ^negate(self);
        endif;
}!!

/* Return true if receiver is greater than zero.  */
Def positive(self)
{       ^self > 0l;
}!!

/* Return true if receiver is greater than zero.  */
Def negative(self)
{       ^self < 0l;
}!!

/* Return true if receiver is equal to zero.  */
Def zero(self)
{       ^self = 0l;
}!!

/* Return true if receiver is not equal to zero.  */
Def nonZero(self)
{ ^self <> 0l;
}!!

setAncestor(Int, Number);!!
setAncestor(Long, Number);!!
setAncestor(Real, Number);!!

/* Return the Real cosine of the receiver.  The receiver is treated as
  an angle in radians. */
Def cos(self)
{ ^cos(asReal(self));
}!!

/* Return the Real square root of the receiver. */
Def sqrt(self)
{ ^sqrt(asReal(self))
}!!

/* Return the Real tangent of the receiver. The receiver is treated as
  an angle in radians. */
Def tan(self)
{ ^tan(asReal(self));
}!!

/* Return the Real sine of the receiver.  The receiver is treated as
  an angle in radians. */
Def sin(self)
{ ^sin(asReal(self));
}!!

/* Return the Real exponential of the receiver (e raised to the
  self power). */
Def exp(self)
{ ^exp(asReal(self));
}!!

/* Return the Real natural logarithm (base e) of the receiver. */
Def log(self)
{ ^log(asReal(self));
}!!

/* Return aNum raised to the receiver
  power as a Real.  For instance, 2**3 
  returns 8.0.  */
 Def **(self, aNum)
{ ^pwr(asReal(self), asReal(aNum));
}!!


/* Return the receiver, which is assumed to be in radians,
  converted to degrees.  A Real is returned.  */
Def radToDeg(self)
{ ^radToDeg(asReal(self));
}!!

/* Return the receiver, which is assumed to be in degrees,
  converted to radians.  A Real is returned.  */
Def degToRad(self)
{ ^degToRad(asReal(self));
}!!

/* Return the arcCosine of the receiver.  The receiver is treated as
  an angle in radians. */
Def arcCos(self)
{       ^pi/2. - arcSin(self);
}!!

/* Return the arcSine of the receiver.  The receiver is treated as
  an angle in radians. */
Def arcSin(self)
{       select
        case  self < -1 or self >1
        is  error(self, stackTop(0), #arcSinError);
        endCase

        case self = 1
        is ^pi / 2.
        endCase

        case self = -1
        is ^negate(pi/2.);
        endCase

        default
                ^arcTan(self / sqrt(1 - self * self));
        endSelect;
}!!

/* Initializes the array of coerced operators. */
Actor[#CoercedOps] := #( #+ #- #* #/ #< #>
 #= #<= #>= #<> bitAnd bitOr bitXor mod
positive negative zero nonZero );!!

/* This routine should be part of an applications init function. */!!
do(CoercedOps, {using(op | func)
        if class(func := Int.methods[op]) == Primitive
        then setFail(func, Number.methods[op]);
        endif;

        if class(func := Long.methods[op]) == Primitive
        then setFail(func, Number.methods[op]);
        endif;

        if class(func := Real.methods[op]) == Primitive
        then setFail(func, Number.methods[op]);
        endif;

        setFail(Int.methods[#min], Magnitude.methods[#min]);
        setFail(Int.methods[#max], Magnitude.methods[#max]);
});!!

/* Return a Point object with an x value
  of the receiver and a y value of the
  argument.  For example, point(3L,7)
  returns 3L@7.  */
Def  point(self, yVal | aPoint)
{       aPoint := new(Point);
        aPoint.x := self;
        aPoint.y := yVal;
        ^aPoint
} !!

/* Convert a packed point as received
  from Windows to an Actor point object.
  A "packed point" has the x coordinate
  stored in the lower two bytes of a number,
  and the y coordinate is stored
  in the upper two bytes.  */
Def asPoint(self)
{ ^asPoint(asLong(self));
}!!

/*  Failure function for integer increment primitive.
  Return self, converted to a Long, incremented by one.  */
Def inc(self)
  { ^asLong(self) + 1l;
  }!!

/*  Failure function for integer increment by two primitive.
  Return self, converted to a Long, incremented by two.  */
Def inc2(self)
  { ^asLong(self) + 2l;
  }!!

/*  Failure function for integer decrement primitive.
  Return self, converted to a Long, decremented by one.  */
Def dec(self)
  { ^asLong(self) - 1l;
  }!!

/*  Failure function for integer decrement by two primitive.
  Return self, converted to a Long, decremented by two.  */
Def dec2(self)
  { ^asLong(self) - 2l;
  }!!

/* Return a Long that has the receiver as
  its low word (lower two bytes) and the argument as its high
  word (highest two bytes). */
Def pack(self, hi)
{ ^(hi * HEX10K) bitOr (self bitAnd HEXffff);
}!!

/* Define an interval from receiver to 
  arg by 1.  Both must be within Int range. 
  */
Def over(self, arg)
{  ^over(asInt(self):Int, asInt(arg))
} !!

